/** @format */
type Optional<T> = { [P in keyof T]?: T[P] }

function request(option) {
  if (String(option) !== '[object Object]') return undefined
  option.method = option.method ? option.method.toUpperCase() : 'GET'
  option.data = option.data || {}
  var formData = []
  for (var key in option.data) {
    formData.push(''.concat(key, '=', option.data[key]))
  }
  option.data = formData.join('&')

  if (option.method === 'GET' && option.data) {
    option.url += option.url.indexOf('?') < 0 ? ''.concat('?', option.data) : ''.concat('&', option.data)
  }

  var xhr = new XMLHttpRequest()
  xhr.responseType = option.responseType || 'plain'
  xhr.onreadystatechange = function () {
    if (xhr.readyState === 4) {
      if (xhr.status === 200) {
        if (option.success && typeof option.success === 'function') {
          option.success(xhr.response)
        }
      } else {
        if (option.error && typeof option.error === 'function') {
          option.error()
        }
      }
    }
  }
  xhr.open(option.method, option.url, true)
  if (option.method === 'POST') {
    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded')
  }
  xhr.send(option.method === 'POST' ? option.data : null)
}

function getRequestUrl() {
  // 如果有测试链接 就用测试的链接
  return window.localStorage.getItem('__ipc_test_url__') || 'https://activity.tuiatest.cn/land/getIcpInfo'
}

function getIPCByDomain(domain: string) {
  return new Promise((resolve, reject) => {
    request({
      url: `${getRequestUrl()}?domain=${encodeURIComponent(domain)}`,
      method: 'GET',
      success(res) {
        console.log('当前域名的IPC备案为', res)
        resolve(res)
      },
      error(err) {
        reject(err)
      }
    })
  })
}

type IPCFooterOptions = {
  /** 自定义域名，默认自动从url取 */
  domain?: string
  /** 要挂载的dom节点，默认为body */
  dom?: HTMLElement | string
  /** 延迟几秒渲染，默认300毫秒 */
  delay?: number
  /** 页脚样式 */
  footerStyles?: Optional<CSSStyleDeclaration>
  /** 页脚内链接的样式 */
  boardStyles?: Optional<CSSStyleDeclaration>
}

function addStyles(ele: HTMLElement, styles: Optional<CSSStyleDeclaration> = {}) {
  Object.entries(styles).forEach(([key, val]) => {
    ele.style[key] = val
  })
}

const defaultFooterStyles: Optional<CSSStyleDeclaration> = {
  width: '100vw',
  position: 'fixed',
  left: '0',
  right: '0',
  bottom: '0',
  display: 'flex',
  justifyContent: 'center',
  alignItems: 'center'
}
const defaultBoardStyles: Optional<CSSStyleDeclaration> = {
  display: 'inline-block',
  textDecoration: 'none',
  color: '#FFF',
  backgroundColor: 'rgba(0,0,0,.4)',
  margin: '4px auto',
  height: '22px',
  lineHeight: '22px',
  padding: '0 20px',
  borderRadius: '4px',
  fontSize: '12px'
}

let currentTaskId = null

/**
 * 挂载
 * @param {IPCFooterOptions} opts
 */
async function mountIPCFooter(opts: IPCFooterOptions = {}) {
  // 如果已经挂载了一个IPCFooter 就先卸载掉，避免重复挂载
  unmount()
  // 当前任务的id
  const taskId = new Date().getTime()
  currentTaskId = taskId

  let IPCNumberString = ''
  try {
    IPCNumberString = (await getIPCByDomain(opts.domain || location.host)) as string
  } catch (err) {
    console.log('备案号获取失败')
  }

  const IPCFooter = document.createElement('div')
  const IPCBoard = document.createElement('a')
  IPCFooter.id = '__IPC_footer__'
  IPCBoard.id = '__IPC_board__'
  addStyles(IPCFooter, { ...defaultFooterStyles, ...(opts.footerStyles || {}) })
  addStyles(IPCBoard, { ...defaultBoardStyles, ...(opts.boardStyles || {}) })
  IPCBoard.innerText = (IPCNumberString as string) || 'IPC备-'
  IPCBoard.href = 'https://beian.miit.gov.cn/#/Integrated/index'
  IPCFooter.appendChild(IPCBoard)

  let tryCount = 10

  function __mount__() {
    let targetDom
    if (opts.dom === undefined || opts.dom === null) {
      targetDom = document.querySelector('body')
    } else if (typeof opts.dom === 'string') {
      targetDom = document.querySelector(opts.dom)
    } else {
      targetDom = opts.dom
    }
    if (!targetDom) {
      // 可能目标dom节点还没挂载
      if (tryCount > 0) {
        tryCount--
        return setTimeout(__mount__, 500)
      }
      return
    }

    setTimeout(() => {
      // 当taskId和currentTaskId不相等时，说明有新的挂载任务进行中，当前任务就abort掉
      if (taskId !== currentTaskId) {
        return
      }
      targetDom.appendChild(IPCFooter)
      currentTaskId = null
    }, opts.delay || 300)
  }
  __mount__()
}

/**
 * 卸载
 */
function unmount() {
  let target = document.getElementById('__IPC_footer__')
  if (target) {
    let parent = target.parentElement
    parent.removeChild(target)
  }
}

/**
 * 重新挂载
 * @param {IPCFooterOptions} opts
 */
function remount(opts: IPCFooterOptions = {}) {
  mountIPCFooter(opts)
}

export default { getIPCByDomain, mountIPCFooter, unmount, remount }
