import { Solver } from './Solver';
import { FrictionEquation } from '../equations/FrictionEquation';

export class GSSolver extends Solver {

    /**
     * 计算次数
     * The max number of iterations to do when solving. More gives better results, but is more expensive.
     * @property iterations
     * @type {Number}
     */
    iterations: number;
    /**
     * 误差
     * The error tolerance, per constraint. If the total error is below this limit, the solver will stop iterating. Set to zero for as good solution as possible, but to something larger than zero to make computations faster.
     * @property tolerance
     * @type {Number}
     * @default 1e-7
     */
    tolerance: number;
    /**
     * Number of solver iterations that are used to approximate normal forces used for friction (F_friction = mu * F_normal). These friction forces will override any other friction forces that are set. If you set frictionIterations = 0, then this feature will be disabled.
     *
     * Use only frictionIterations > 0 if the approximated normal force (F_normal = mass * gravity) is not good enough. Examples of where it can happen is in space games where gravity is zero, or in tall stacks where the normal force is large at bottom but small at top.
     *
     * @property frictionIterations
     * @type {Number}
     * @default 0
     */
    frictionIterations: number;
    /**
     * The number of iterations that were made during the last solve. If .tolerance is zero, this value will always be equal to .iterations, but if .tolerance is larger than zero, and the solver can quit early, then this number will be somewhere between 1 and .iterations.
     * @property {Number} usedIterations
     */
    usedIterations: number;


    /**
     * Iterative Gauss-Seidel constraint equation solver.
     *
     * @class GSSolver
     * @constructor
     * @extends Solver
     * @param {Object} [options]
     * @param {Number} [options.iterations=10]
     * @param {Number} [options.tolerance=0]
     */
    constructor(options?: any) {
        super(options, Solver.GS);
        options = options || {};


        this.iterations = options.iterations || 10;


        this.tolerance = options.tolerance !== undefined ? options.tolerance : 1e-7;


        this.frictionIterations = options.frictionIterations !== undefined ? 0 : options.frictionIterations;


        this.usedIterations = 0;
    }


    /**
     * Solve the system of equations
     * @method solve
     * @param  {Number}  h       Time step
     * @param  {World}   world    World to solve
     */
    solve(h, world): void {

        this.sortEquations();

        var iter = 0,
            maxIter = this.iterations,
            maxFrictionIter = this.frictionIterations,
            equations = this.equations,
            Neq = equations.length,
            tolSquared = Math.pow(this.tolerance * Neq, 2),
            bodies = world.bodies,
            Nbodies = bodies.length;

        this.usedIterations = 0;

        if (Neq) {
            for (var i = 0; i !== Nbodies; i++) {
                var b = bodies[i];

                // Update solve mass
                b.updateSolveMassProperties();
            }
        }

        for (var i = 0; i !== Neq; i++) {
            var c: any = equations[i];
            c.lambda = 0;
            if (c.timeStep !== h || c.needsUpdate) {
                c.timeStep = h;
                c.update();
            }
            c.B = c.computeB(c.a, c.b, h);
            c.invC = c.computeInvC(c.epsilon);

            c.maxForceDt = c.maxForce * h;
            c.minForceDt = c.minForce * h;
        }

        var c, deltalambdaTot, i: number, j;

        if (Neq !== 0) {

            for (i = 0; i !== Nbodies; i++) {
                var b = bodies[i];

                // Reset vlambda
                b.resetConstraintVelocity();
            }

            if (maxFrictionIter) {
                // Iterate over contact equations to get normal forces
                for (iter = 0; iter !== maxFrictionIter; iter++) {

                    // Accumulate the total error for each iteration.
                    deltalambdaTot = 0.0;

                    for (j = 0; j !== Neq; j++) {
                        c = equations[j];

                        var deltalambda = iterateEquation(c/*, h*/);
                        deltalambdaTot += Math.abs(deltalambda);
                    }

                    this.usedIterations++;

                    // If the total error is small enough - stop iterate
                    if (deltalambdaTot * deltalambdaTot <= tolSquared) {
                        break;
                    }
                }

                updateMultipliers(equations, 1 / h);

                // Set computed friction force
                for (j = 0; j !== Neq; j++) {
                    var eq = equations[j];
                    if (eq instanceof FrictionEquation) {
                        var f = 0.0;
                        for (var k = 0; k !== eq.contactEquations.length; k++) {
                            f += eq.contactEquations[k].multiplier;
                        }
                        f *= eq.frictionCoefficient / eq.contactEquations.length;
                        eq.maxForce = f;
                        eq.minForce = -f;

                        eq.maxForceDt = f * h;
                        eq.minForceDt = -f * h;
                    }
                }
            }

            // Iterate over all equations
            for (iter = 0; iter !== maxIter; iter++) {

                // Accumulate the total error for each iteration.
                deltalambdaTot = 0.0;
                for (j = 0; j !== Neq; j++) {
                    c = equations[j];

                    var deltalambda = iterateEquation(c/*, h*/);
                    deltalambdaTot += Math.abs(deltalambda);
                }

                this.usedIterations++;

                // If the total error is small enough - stop iterate
                if (deltalambdaTot * deltalambdaTot < tolSquared) {
                    break;
                }
            }

            // Add result to velocity
            for (i = 0; i !== Nbodies; i++) {
                bodies[i].addConstraintVelocity();
            }

            updateMultipliers(equations, 1 / h);
        }
    };
}
// Sets the .multiplier property of each equation
function updateMultipliers(equations, invDt) {
    var l = equations.length;
    while (l--) {
        var eq = equations[l];
        eq.multiplier = eq.lambda * invDt;
    }
}

function iterateEquation(eq) {
    // Compute iteration
    var B = eq.B,
        eps = eq.epsilon,
        invC = eq.invC,
        lambdaj = eq.lambda,
        GWlambda = eq.computeGWlambda(),
        maxForce_dt = eq.maxForceDt,
        minForce_dt = eq.minForceDt;

    var deltalambda = invC * (B - GWlambda - eps * lambdaj);

    // Clamp if we are not within the min/max interval
    var lambdaj_plus_deltalambda = lambdaj + deltalambda;
    if (lambdaj_plus_deltalambda < minForce_dt) {
        deltalambda = minForce_dt - lambdaj;
    } else if (lambdaj_plus_deltalambda > maxForce_dt) {
        deltalambda = maxForce_dt - lambdaj;
    }
    eq.lambda += deltalambda;
    eq.addToWlambda(deltalambda);

    return deltalambda;
}
