import Container from "./Container";
import { RENDERER_TYPE } from "../const"
// import SystemRenderer from "../renderers/SystemRenderer";
import { Rectangle, Point } from "../math";
import { EventDispatcher } from "../events/EventDispatcher";
import { Event } from "../events/Event";
// import { FloatDisplay } from "./FloatDisplay";
import { DisplayObject } from "./DisplayObject";
import { MouseEvent } from "../events/MouseEvent";

import { WebglRenderer } from "../renderers/WebglRenderer";
import { getRequestAnimationFrame, getCancelAnimationFrame } from "../utils";
import { Stats } from "../../Stats";
import { CanvasRenderer } from "../renderers/CanvasRenderer";

export class Stage extends Container {

    /**
     * 当前stage所使用的渲染器
     * 渲染器有两种,一种是canvas 一种是webGl
     * @property renderObj
     * @public
     * @since 1.0.0
     * @type {IRender}
     * @default null
     */
    public renderObj: WebglRenderer = null;


    /**
     * 直接获取stage的引用，避免总是从Event.ADD_TO_STAGE 事件中去获取stage引用
     * @property getStage
     * @param {string} stageName
     * @return {any}
     * @since 2.0.0
     */
    public static getStage(stageName: string = "cusEngine"): Stage {
        return Stage._stageList[stageName];
    }

    /**
     * @property _stageList
     * @static
     * @type {Object}
     * @private
     */
    private static _stageList: any = {};
    /**
     * 是否暂停
     * @property pause
     * @static
     * @type {boolean}
     * @public
     * @since 1.0.0
     * @default false
     */
    static get pause(): boolean {
        return this._pause;
    }
    static set pause(value: boolean) {
        this._pause = value;
        if (value != this._pause) {
            //触发事件
            // GDispatcher.dispatchEvent("onStagePause", { pause: value });
        }
    }

    /**
     * @property _pause
     * @type {boolean}
     * @private
     * @static
     */
    private static _pause: boolean = false;


    private static _flushId = -1;
    /**
     * 舞台在设备里截取后的可见区域,有些时候知道可见区域是非常重要的,因为这样你就可以根据舞台的可见区域做自适应了。
     * @property viewRect
     * @public
     * @readonly
     * @since 1.0.0
     * @type {Rectangle}
     * @default {x:0,y:0,width:0,height:0}
     * @readonly
     */
    public viewRect: Rectangle = new Rectangle();

    /**
     * 舞台的尺寸宽,也就是我们常说的设计尺寸
     * @property desWidth
     * @public
     * @since 1.0.0
     * @default 320
     * @type {number}
     * @readonly
     */
    public desWidth: number = 0;
    /**
     * 舞台的尺寸高,也就是我们常说的设计尺寸
     * @property desHeight
     * @public
     * @since 1.0.0
     * @default 240
     * @type {number}
     * @readonly
     */
    public desHeight: number = 0;
    /**
     * 舞台在当前设备中的真实高
     * @property divHeight
     * @public
     * @since 1.0.0
     * @default 320
     * @type {number}
     * @readonly
     */
    public divHeight: number = 0;
    /**
     * 舞台在当前设备中的真实宽
     * @property divWidth
     * @public
     * @since 1.0.0
     * @default 240
     * @readonly
     * @type {number}
     */
    public divWidth: number = 0;
    /**
     * 舞台的背景色
     * 默认就是透明背景
     * 可能设置一个颜色值改变舞台背景
     * @property bgColor
     * @public
     * @since 1.0.0
     * @type {number}
     * @default "";
     */
    private _bgColor: number = 0x000000;

    public get bgColor(): number {
        return this._bgColor;
    }
    /**
     * 设置颜色，即改变渲染器颜色
     */
    public set bgColor(value: number) {
        if (this._bgColor === value) return
        this._bgColor = value;
        this.renderObj.backgroundColor = value;
    }

    /**
     * 上一次鼠标或触碰经过的显示对象列表
     * @property _lastDpList
     * @type {Object}
     * @private
     */
    private _lastDpList: any = {};

    dpi: number
    /**
     * canvas模式下有问题，画布没按style缩小
     * @param {gl} context 
     * @param {*} data desWidth,desHeight,divWidth,divHeight,canWidth,canHeight
     */
    public constructor(
        context: WebGLRenderingContext | CanvasRenderingContext2D,
        data: {
            type: RENDERER_TYPE
            desWidth: number,
            desHeight: number,
            divWidth: number,
            divHeight: number,
            canWidth: number,
            canHeight: number
        }
    ) {
        super();
        let s: Stage = this;
        this._instanceType = "Stage";
        Stage._stageList["canvas"] = s;
        s.stage = this;
        s.name = "stageInstance_" + s.instanceId;
        s.desWidth = data.desWidth || 750;
        s.desHeight = data.desHeight || 1206;
        s.divWidth = data.divWidth;
        s.divHeight = data.divHeight;

        s.dpi = data.canWidth / data.divWidth
        // my.alert({
        //   title: '55729:'+data.canWidth+","+data.canHeight
        // });
        // s.renderObj = new CanvasRenderer(context, data.canWidth, data.canHeight);
        if (data.type == RENDERER_TYPE.CANVAS) {
            //@ts-ignore
            s.renderObj = new CanvasRenderer(context, data.canWidth, data.canHeight);
        } else {
            //@ts-ignore
            s.renderObj = new WebglRenderer(context, data.canWidth, data.canHeight);
        }

        //不加刚好置顶
        s.anchorX = data.desWidth >> 1;
        s.anchorY = data.desHeight >> 1;
        s.x = (data.canWidth - data.desWidth) >> 1;
        s.y = (data.canHeight - data.desHeight) >> 1;
        //定宽
        var scale = data.canWidth / data.desWidth;
        s.scaleX = scale;
        s.scaleY = scale;

        s.viewRect = new Rectangle();
        s.viewRect.x = (data.desWidth - data.canWidth / scale) >> 1;
        s.viewRect.y = (data.desHeight - data.canHeight / scale) >> 1;
        s.viewRect.width = data.desWidth - s.viewRect.x * 2;
        s.viewRect.height = data.desHeight - s.viewRect.y * 2;
        // my.alert({
        //   title: StageManager.ins.width
        // });

        setTimeout(function () {
            //同时添加到主更新循环中
            Stage.addUpdateObj(s);
            s.dispatchEvent(Event.INIT_STAGE);
            // }
        }, 100);
    }


    /**
     * 这个是鼠标事件的MouseEvent对象池,因为如果用户有监听鼠标事件,如果不建立对象池,那每一秒将会new Fps个数的事件对象,影响性能
     * @property _ml
     * @type {Array}
     * @private
     */
    private _ml: any = [];
    /**
     * 这个是事件中用到的Point对象池,以提高性能
     * @property _mp
     * @type {Array}
     * @private
     */
    private _mp: any = [];

    /**
     * 刷新mouse或者touch事件
     * @method _initMouseEvent
     * @private
     */
    private _initMouseEvent(event: MouseEvent, cp: Point, sp: Point, identifier: number): void {
        event["_pd"] = false;
        event.clientX = cp.x;
        event.clientY = cp.y;
        event.stageX = sp.x;
        event.stageY = sp.y;
        event.identifier = identifier;
    }

    //每一个手指事件的对象池
    /**
     * @property _mouseDownPoint
     * @type {Object}
     * @private
     */
    private _mouseDownPoint: any = {};

    /**
     * 循环刷新页面的函数
     * @method flush
     * @private
     * @return {void}
     */
    public flush(): void {
        this.renderObj.render(this);
    }

    /**
     * 当一个stage不再需要使用,或者要从浏览器移除之前,请先停止它,避免内存泄漏
     * @method kill
     * @since 1.0.0
     * @public
     */
    public kill(): void {
        Stage.removeUpdateObj(this);
    }

    /**
     * html的鼠标或单点触摸对应的引擎事件类型名
     * @property _mouseEventTypes
     * @type {{mousedown: string, mouseup: string, mousemove: string, touchstart: string, touchmove: string, touchend: string}}
     * @private
     */
    private _mouseEventTypes: any = {
        touchstart: "onMouseDown",
        touchmove: "onMouseMove",
        touchend: "onMouseUp",
        touchStart: "onMouseDown",//小程序返回的时间名是驼峰的
        touchMove: "onMouseMove",
        touchEnd: "onMouseUp"
    };

    /**
     * 无多指，无拖动
     * @method onMouseEvent
     * @param e
     * @private
     */
    onMouseEvent(e: any): void {
        let s: Stage = this;
        //检查mouse或touch事件是否有，如果有的话，就触发事件函数
        if (EventDispatcher._totalMEC > 0) {
            let points: any;
            //事件类型

            let item = s._mouseEventTypes[e.type];
            let events: any;
            let event: any;
            //stageMousePoint
            let sp: Point;
            //localPoint;
            let lp: Point;
            //clientPoint
            let cp: Point;
            //事件个数
            let eLen: number;
            let identifier: any;

            points = [e.changedTouches[0]];
            //  points = [e.touches[0]];
            // my.alert({
            //   title: '55729:' + points.length
            // });
            for (let o = 0; o < points.length; o++) {
                if (!points[o]) continue;
                eLen = 0;
                events = [];
                identifier = "m" + points[o].identifier;
                if (s._mp.length > 0) {
                    cp = s._mp.shift();
                } else {
                    cp = new Point();
                }
                cp.x = (points[o].x || points[o].b) * s.dpi// devicePixelRatio;
                cp.y = (points[o].y || points[o].c) * s.dpi// devicePixelRatio;
                // my.alert({
                //   title: JSON.stringify(points[o])
                // });
                //计算舞台中的点
                sp = s.globalToLocal(cp, DisplayObject._bp);
                //检查是否有鼠标事件
                if (EventDispatcher.getMouseEventCount() > 0) {
                    if (!s._ml[eLen]) {
                        event = new MouseEvent(item);
                        s._ml[eLen] = event;
                    } else {
                        event = s._ml[eLen];
                        event.type = item;
                    }
                    events[events.length] = event;
                    s._initMouseEvent(event, cp, sp, identifier);
                    eLen++;
                }
                if (item == "onMouseDown") {
                    s._mouseDownPoint[identifier] = cp;
                    //清空上次存在的显示列表
                } else if (item == "onMouseUp") {
                    if (s._mouseDownPoint[identifier]) {
                        if (Point.distance(s._mouseDownPoint[identifier], cp) < 20) {
                            //检查是否有添加对应的click事件
                            if (EventDispatcher.getMouseEventCount("onMouseClick") > 0) {
                                if (!s._ml[eLen]) {
                                    event = new MouseEvent("onMouseClick");
                                    s._ml[eLen] = event;
                                } else {
                                    event = s._ml[eLen];
                                    event.type = "onMouseClick";
                                }
                                events[events.length] = event;
                                s._initMouseEvent(event, cp, sp, identifier);
                                eLen++;
                            }
                        }
                    }
                }
                if (eLen > 0) {
                    //有事件开始遍历显示列表
                    //找出最底层的显示对象
                    let d: DisplayObject = s.hitTestPoint(cp, true);
                    // console.log(d)
                    let displayList: Array<DisplayObject> = [];
                    // my.alert({
                    //   title: '55729:' + d.instanceId
                    // });
                    if (d) {
                        //证明有点击到事件,然后从最底层追上来,看看一路是否有人添加过mouse或touch事件,还要考虑mousechildren和阻止事件方法
                        //找出真正的target,因为有些父级可能会mouseChildren=false;
                        while (d) {
                            if (d["mouseChildren"] === false) {
                                //丢掉之前的层级,因为根本没用了
                                displayList.length = 0;
                            }
                            displayList[displayList.length] = d;
                            d = d.parent;
                        }
                    } else {
                        displayList[displayList.length] = s;
                    }
                    let len: number = displayList.length;
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[0];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j]);
                                }
                            }
                        }
                    }
                    //这里一定要反转一下，因为会影响mouseOut mouseOver
                    displayList.reverse();
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type, false) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[eLen - 1];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j], null, false);
                                }
                            }
                        }
                    }
                    //最后要和上一次的遍历者对比下，如果不相同则要触发onMouseOver和onMouseOut
                    if (item != "onMouseDown") {
                        if (EventDispatcher.getMouseEventCount("onMouseOver") > 0 || EventDispatcher.getMouseEventCount("onMouseOut") > 0) {
                            if (s._lastDpList[identifier]) {
                                //从第二个开始，因为第一个对象始终是stage顶级对象
                                let len1 = s._lastDpList[identifier].length;
                                let len2 = displayList.length;
                                len = len1 > len2 ? len1 : len2;
                                let isDiff = false;
                                let overEvent: MouseEvent;
                                let outEvent: MouseEvent;
                                for (let i = 1; i < len; i++) {
                                    if (!isDiff) {
                                        if (s._lastDpList[identifier][i] != displayList[i]) {
                                            //确定哪些有onMouseOver,哪些有onMouseOut
                                            isDiff = true;
                                            if (!s._ml[eLen]) {
                                                overEvent = new MouseEvent("onMouseOver");
                                                s._ml[eLen] = overEvent;
                                            } else {
                                                overEvent = s._ml[eLen];
                                                overEvent.type = "onMouseOver";
                                            }
                                            s._initMouseEvent(overEvent, cp, sp, identifier);
                                            eLen++;
                                            if (!s._ml[eLen]) {
                                                outEvent = new MouseEvent("onMouseOut");
                                                s._ml[eLen] = outEvent;
                                            } else {
                                                outEvent = s._ml[eLen];
                                                outEvent.type = "onMouseOut";
                                            }
                                            s._initMouseEvent(outEvent, cp, sp, identifier);
                                        }
                                    }
                                    if (isDiff) {
                                        if (s._lastDpList[identifier][i]) {
                                            //触发onMouseOut事件
                                            if (!outEvent["_pd"]) {
                                                d = s._lastDpList[identifier][i];
                                                if (d.hasEventListener("onMouseOut")) {
                                                    outEvent.currentTarget = d;
                                                    outEvent.target = s._lastDpList[identifier][len1 - 1];
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    outEvent.localX = lp.x;
                                                    outEvent.localY = lp.y;
                                                    d.dispatchEvent(outEvent);
                                                }
                                            }
                                        }
                                        if (displayList[i]) {
                                            //触发onMouseOver事件
                                            if (!overEvent["_pd"]) {
                                                d = displayList[i];
                                                if (d.hasEventListener("onMouseOver")) {
                                                    overEvent.currentTarget = d;
                                                    overEvent.target = displayList[len2 - 1];
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    overEvent.localX = lp.x;
                                                    overEvent.localY = lp.y;
                                                    d.dispatchEvent(overEvent);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        s._mp[s._mp.length] = cp;
                    }
                    if (item == "onMouseUp") {
                        delete s._mouseDownPoint[identifier];
                        delete s._lastDpList[identifier];
                    } else {
                        s._lastDpList[identifier] = displayList;
                    }
                }
            }
        }
    };

    public getBounds(): Rectangle {
        return this.viewRect;
    }

    /**
     * 要循环调用 flush 函数对象列表
     * @method allUpdateObjList
     * @static
     * @since 1.0.0
     * @type {Array}
     */
    private static allUpdateObjList: Array<any> = [];


    static requestAnimationFrame;
    static cancelAnimationFrame;
    private static _stats: any
    /**
     * 刷新所有定时器
     * @static
     * @private
     * @since 1.0.0
     * @method flushAll
     */
    static flushAll(): void {
        if (!Stage.requestAnimationFrame) {
            Stage.requestAnimationFrame = getRequestAnimationFrame();
            Stage.cancelAnimationFrame = getCancelAnimationFrame();
        }
        if (!Stage.requestAnimationFrame) return;
        if (!Stage._pause) {
            if (Stage._stats) Stage._stats.begin();
            let len = Stage.allUpdateObjList.length;
            for (let i = 0; i < len; i++) {
                Stage.allUpdateObjList[i] && Stage.allUpdateObjList[i].flush();
            }
            if (Stage._stats) Stage._stats.end();
        }
        Stage._flushId = Stage.requestAnimationFrame(Stage.flushAll);
    }
    //页面关闭时要执行，隐藏或显示只修改pause，
    public static stop() {
        //取消帧循环
        Stage.cancelAnimationFrame(Stage._flushId);
        //去掉帧率检测
        Stage._stats = null;
        //去掉循环函数，因为时跟着canvas的
        Stage.requestAnimationFrame = null;
        Stage.cancelAnimationFrame = null

    }

    /**
     * 添加stats帧率显示
     * @param canvasId 
     */
    public static addFPS(canvasId: string) {
        //@ts-ignore
        if (my) Stage._stats = new Stats(canvasId);
    }

    /**
     * 添加一个刷新对象，这个对象里一定要有一个 flush 函数。
     * 因为一但添加，这个对象的 flush 函数会以stage的fps间隔调用
     * 如，你的stage是30fps 那么你这个对象的 flush 函数1秒会调用30次。
     * @method addUpdateObj
     * @param target 要循化调用 flush 函数的对象
     * @public
     * @static
     * @since
     */
    public static addUpdateObj(target: any): void {
        let isHave: boolean = false;
        let len = Stage.allUpdateObjList.length;
        for (let i = 0; i < len; i++) {
            if (Stage.allUpdateObjList[i] === target) {
                isHave = true;
                break;
            }
        }
        if (!isHave) {
            Stage.allUpdateObjList.unshift(target);
        }
    }

    /**
     * 移除掉已经添加的循环刷新对象
     * @method removeUpdateObj
     * @param target
     * @public
     * @static
     * @since 1.0.0
     */
    public static removeUpdateObj(target: any): void {
        let len = Stage.allUpdateObjList.length;
        for (let i = 0; i < len; i++) {
            if (Stage.allUpdateObjList[i] === target) {
                Stage.allUpdateObjList.splice(i, 1);
                break;
            }
        }
    }

    public destroy(): void {
        let s = this;
        Stage.removeUpdateObj(s);
        // s.rootDiv = null;
        // s._floatDisplayList = null;
        s.renderObj.destroy();
        s.renderObj = null;
        s.viewRect = null;
        s._lastDpList = null;
        s._ml = null;
        super.destroy();
    }
}