import { DATA_TYPE, FORMAT_TYPE, SCALE_MODES, WRAP_MODES } from "../const";
import { EventDispatcher } from "../events";
import { loadImage } from "../loader";
import { BaseTextureCache, isPow2, TextureCache, uid } from "../utils";

/**
 * 每个texture都有一个BaseTexture，多数用于图集，texture可自身设置属性
 * 实际绘制用的都是BaseTexture
 * @class
 * @extends EventDispatcher
 */
export class BaseTexture extends EventDispatcher {
    /**
     * 贴图回收时用到，
     * 标记是否被使用过
     */
    _touchedId: number = 0;
    /**
     * 批处理时用到的标志位
     * 被使用着的id
     */
    _enabledId: number = 0;
    /**
     * 数据有修改时++，初始0
     * source改变，宽高改变，
     * premultipliedAlpha、dataFormat、dataType、flipY等改变都需要
     */
    _dataId: number = 0;
    /**
     * 样式有修改时更新，修改时++，初始0，预留吧，TODO
     */
    _styleId: number = 0;
    /**
     * The ids under which this BaseTexture has been added to the base texture cache. This is
     * automatically set as long as BaseTexture.addToCache is used, but may not be set if a
     * BaseTexture is added directly to the BaseTextureCache array.
     * @member {string[]}
     */
    textureCacheIds: string[] = [];
    /**
     * image对象，canvas对象，或者imageData（canvas渲染模式用不了，是否考虑去掉），以后类型增多后再说
     */
    source = null;
    /**
     * 加载完成会设置为true
     * 加载失败或没有贴图数据则为false，或者宽高为0等都为false
     * 通常用于标记基础纹理是否可用，TODO
     */
    hasLoaded: boolean = false;

    protected _width: number = 100;
    /**
     * 宽度
     */
    get width(): number {
        return this._width;
    };
    protected _height: number = 100;
    /**
     * 高度
     */
    get height(): number {
        return this._height;
    };
    //////webgl需要用到的属性///////
    private _premultipliedAlpha: boolean = true;
    /**
     * rgb预乘alpha，webgl用到，png图片设置必为true，否则色值会出问题，一般png图片导出时不会预乘
     * 默认true
     */
    get premultipliedAlpha() {
        return this._premultipliedAlpha
    }
    set premultipliedAlpha(v: boolean) {
        if (this._premultipliedAlpha == v) return;
        this._premultipliedAlpha = v;
        this._dataId++;
    }
    private _dataFormat: FORMAT_TYPE = FORMAT_TYPE.RGBA;
    /**
     * 纹理格式，默认RGBA  还有RGB等，还没有地方维护，TODO
     */
    get dataFormat() {
        return this._dataFormat
    }
    set dataFormat(v: FORMAT_TYPE) {
        if (this._dataFormat == v) return;
        this._dataFormat = v;
        this._dataId++;
    }
    private _dataType: DATA_TYPE = DATA_TYPE.UNSIGNED_BYTE;
    /**
     * 纹理数据类型，默认UNSIGNED_BYTE   //https://developer.mozilla.org/zh-CN/docs/Web/API/WebGLRenderingContext/texImage2D
     */
    get dataType() {
        return this._dataType;
    }
    set dataType(v: DATA_TYPE) {
        if (this._dataType == v) return;
        this._dataType = v;
        this._dataId++;
    }
    private _flipY: boolean = false;
    /**
     * 是否翻转纹理，默认false
     */
    get flipY() {
        return this._flipY;
    }
    set flipY(v: boolean) {
        if (this._flipY == v) return;
        this._flipY = v;
        this._dataId++;
    }

    ///////////下面的是纹理webgl属性
    protected _isPowerOfTwo: boolean = false;
    /**
     * 是否尺寸为2的次方，尽可能图集尺寸都为2的次方，gpu处理方便，并且能做mipmap缓存，性能更好
     */
    get isPowerOfTwo(): boolean {
        return this._isPowerOfTwo;
    };

    private _scaleMode: SCALE_MODES = SCALE_MODES.LINEAR;
    /**
     * SCALE_MODES，默认LINEAR
     */
    get scaleMode() {
        return this._scaleMode
    }
    set scaleMode(v: SCALE_MODES) {
        if (this._scaleMode == v) return;
        this._scaleMode = v;
        this._styleId++;
    }

    private _mipmap: boolean = true;
    /**
     * 尺寸是2的次方下才能设置true，用于生成mipmap缓存
     * @default true
     */
    get mipmap() {
        return this._mipmap;
    }
    set mipmap(v: boolean) {
        if (!this._isPowerOfTwo) return;
        if (this._mipmap == v) return;
        this._mipmap = v;
        this._styleId++;
    }

    private _wrapMode: WRAP_MODES = WRAP_MODES.CLAMP;
    /**
     * 非2的次方时要设置CLAMP
     * WebGL Texture wrap mode
     * @default WRAP_MODES.CLAMP
     */
    get wrapMode() {
        return this._wrapMode;
    }
    set wrapMode(v: WRAP_MODES) {
        if (this._wrapMode == v) return;
        if (v != WRAP_MODES.CLAMP && !this._isPowerOfTwo) return;
        this._wrapMode = v;
        this._styleId++;
    }
    ////////////
    /**
     * @param {} [source] - 源数据,image对象，canvas对象，或者imageData
     */
    constructor(source = null) {
        super();
        if (!source) return
        //有source要加载，构造函数不缓存baseTexture，要缓存用fromUrl
        //是imageData
        if (source._data) {
            this._sourceLoaded(source);
            //数据类型改一下
            if (source._data instanceof Float32Array) {
                this.dataType = DATA_TYPE.FLOAT;
            }
            else {
                this.dataType = DATA_TYPE.UNSIGNED_BYTE;
            }
        }
        //是canvas
        else if (source.getContext) {
            this._sourceLoaded(source);
        }
        //普通图片
        else {
            if (source.complete) {
                this._sourceLoaded(source);
                // this.dispatchEvent("loaded");//上面的方法会执行loaded
            } else {
                var self = this;
                //会覆盖onload方法
                source.onload = function () {
                    self._sourceLoaded(source);
                    // self.dispatchEvent("loaded");//上面的方法会执行loaded
                };
                //失败的
                source.onerror = function () {
                    self.dispatchEvent("loaded");
                };
            }
        }
    }
    /**
     * 会触发更新事件，一般用于纹理source变更或修改后调用
     * @fires BaseTexture#update
     */
    update() {
        if (!this.source) return;
        //尺寸重置
        this._setSize(this.source.width, this.source.height);
        //再来判断一次
        this.hasLoaded = this.width && this.height ? true : false;
        //数据改变
        this._dataId++;
        //触发绑定过的更新事件，比如webgl纹理更新
        this.dispatchEvent("update");
    }
    /**
     * 不希望外部使用
     * @param width 
     * @param height 
     * @returns 
     */
    protected _setSize(width: number, height: number) {
        if (this._width == width && this._height == height) return;
        //尺寸重置
        this._width = width;
        this._height = height;
        let oriP2 = this._isPowerOfTwo;
        this._isPowerOfTwo = isPow2(this.width) && isPow2(this.height);
        this._dataId++;
        if (oriP2 !== this._isPowerOfTwo) this._styleId++;
    }
    /**
     * 原先的_sourceChange改成了_sourceLoaded，且变成私有
     */
    private _sourceLoaded(source) {
        //赋值
        this.source = source;
        //路径，暂时没用，需要时再处理，先找src 再path（基本没有）
        // this.imageUrl = source.src || source.path || null;
        //更新类型,暂时没用，也没做，需要是处理
        // this.imageType = this.source.type || null;
        //加载完成
        this.hasLoaded = true;
        //更新
        this.update();
        //触发事件
        this.dispatchEvent("loaded");
    }

    /**
     * 销毁 base texture
     * 基本不会销毁纹理
     */
    destroy() {
        let imageUrl = this.source ? (this.source.src || this.source.path) : null;
        if (imageUrl) {
            //普通纹理删除缓存，万一有用到
            delete TextureCache[imageUrl];
        }
        this.source = null;
        this.dispose();
        BaseTexture.removeFromCache(this);
        this.textureCacheIds = null;
    }

    /**
     * 用于释放gpu缓存，并不销毁纹理，需要时可再上传到GPU
     * @fires BaseTexture#dispose
     */
    dispose() {
        //用于触发TextureManager中监听的
        this.dispatchEvent("dispose")
    }

    //辅助静态方法
    /**
     * 根据路径，会缓存baseTexture
     * @param {string} url  路径
     */
    static fromUrl(url: string) {
        if (BaseTextureCache[url]) return BaseTextureCache[url];
        let baseTexture = new BaseTexture();
        loadImage(url,
            (res) => {
                baseTexture._sourceLoaded(res);
                //添加进缓存
                if (url.indexOf('data:') !== 0) BaseTexture.addToCache(baseTexture, url);
            },
            () => {
                //失败的时候也需要派发加载完成事件
                baseTexture.dispatchEvent("loaded");
            }
        )
        return baseTexture
    }

    /**
     * 随便啥形式的，比如IImageData形式，
     * @param data 
     */
    static fromData(data: IImageData) {
        return new BaseTexture(data);
    }

    /**
     * 从离屏canvas创建的，会给canvas加唯一标识_canvasId，并缓存
     */
    static fromCanvas(canvas: HTMLCanvasElement, origin: string = 'canvas') {
        //标记canvasId
        if (!canvas["_canvasId"]) {
            canvas["_canvasId"] = `${origin}_${uid()}`;
        }
        let baseTexture = BaseTextureCache[canvas["_canvasId"]];
        if (!baseTexture) {
            baseTexture = new BaseTexture(canvas);
            BaseTexture.addToCache(baseTexture, canvas["_canvasId"]);
        }
        return baseTexture;
    }
    /**
     * 根据图片
     * @param image 
     */
    static fromImage(image: HTMLImageElement) {
        //图片标签
        const imageUrl = image.src;
        let baseTexture = BaseTextureCache[imageUrl];
        if (!baseTexture) {
            baseTexture = new BaseTexture(image);
            //不缓存base64，如需要，设名字，手动缓存
            if (imageUrl && imageUrl.indexOf('data:') !== 0) BaseTexture.addToCache(baseTexture, imageUrl);
        }
        return baseTexture;
    }
    /**
     * 所有形式，图片路径，canvas标签，图片标签，或者数据
     * @param anything 
     * @returns 
     */
    static from(anything: string | HTMLCanvasElement | HTMLImageElement | IImageData) {
        //路径
        if (typeof anything === 'string') {
            return BaseTexture.fromUrl(anything);
        }
        //@ts-ignore
        else if (anything._data) {
            //@ts-ignore
            return BaseTexture.fromData(anything);
        }
        //@ts-ignore canvas
        else if (anything.getContext) {
            //@ts-ignore
            return BaseTexture.fromCanvas(anything);
        } else {
            //@ts-ignore
            return BaseTexture.fromImage(anything);
        }
    }

    /**
     * 加入全局基础纹理缓存
     * @static
     * @param {BaseTexture} baseTexture
     * @param {string} id
     */
    static addToCache(baseTexture: BaseTexture, id: string) {
        if (id) {
            if (baseTexture.textureCacheIds.indexOf(id) === -1) {
                baseTexture.textureCacheIds.push(id);
            }

            if (BaseTextureCache[id]) {
                //覆盖
                console.warn(`rewrite cached baseTexture: ${id}`);
            }
            BaseTextureCache[id] = baseTexture;
        }
    }

    /**
     * 移除缓存
     * @static
     * @param {string|BaseTexture} baseTexture id或者BaseTexture
     * @return {BaseTexture|null} 移除的BaseTexture或null
     */
    static removeFromCache(baseTexture: string | BaseTexture): BaseTexture {
        if (typeof baseTexture === 'string') {
            const baseTextureFromCache = BaseTextureCache[baseTexture];
            if (baseTextureFromCache) {
                const index = baseTextureFromCache.textureCacheIds.indexOf(baseTexture);
                if (index > -1) {
                    baseTextureFromCache.textureCacheIds.splice(index, 1);
                }
                delete BaseTextureCache[baseTexture];
                return baseTextureFromCache;
            }
        }
        else if (baseTexture && baseTexture.textureCacheIds) {
            for (let i = 0; i < baseTexture.textureCacheIds.length; ++i) {
                delete BaseTextureCache[baseTexture.textureCacheIds[i]];
            }
            baseTexture.textureCacheIds.length = 0;
            return baseTexture;
        }
        return null;
    }
}
/**
 * 图片原始数据形式的数据接口
 */
export interface IImageData {
    //这里需要修改下data改为_data，tt里的Image对象居然有data（ArrayBuffer虽然能用，但是有偏移），导致判断有问题，需要同步修改TextureManager
    /**
     * 原始图片数据
     * 比如从getImageData获取的数据
     * var imgData = ctx.getImageData(0, 0, w, h);
     * var _data = new Uint8Array(imgData.data)
     */
    _data: Uint8Array | ArrayBuffer | Float32Array | Uint8ClampedArray,
    width: number,
    height: number,
    /**
     * 暂时不必要
     */
    type?: string,
    /**
     * 暂时不必要
     */
    path?: string
}