import { DATA_TYPE, GC_MODES, SCALE_MODES, WRAP_MODES } from "../../const";
import { BaseTexture, Texture } from "../../texture";
import { BaseRenderTexture } from "../../texture/BaseRenderTexture";
import { Dict } from "../../utils";
import { WebglRenderer } from "../WebglRenderer";

export interface IWebglTexture {
    dataId: number,
    styleId: number,
    webglTexture: WebGLTexture
}
//
export class TextureManager {
    private map: WeakMap<BaseTexture, IWebglTexture>
    /**
     * 绑定的纹理，通道作为索引，filterManager，batchrender等有用到
     */
    boundTextures: BaseTexture[];
    /**
     * 只放两个，2d和cube
     */
    private emptyTextures: Dict<WebGLTexture>;
    private _nextTextureLocation: number = -1;
    /**
     * 当前激活的纹理通道，和bind无关
     */
    private activeLocation: number;
    /**
     * 用于gc或销毁用，注意gc的有问题，判断帧缓存时
     */
    _managedTextures: Array<BaseTexture>;

    private textureFloatExtension
    /**
     * @param {WebGLRenderer} renderer
     */
    constructor(public renderer: WebglRenderer) {
        this._managedTextures = [];
        this.boundTextures = [];
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);
        this.renderer.addEventListener('onPostRender', this.onPostRender, this);

    }
    onContextChange() {
        //新的映射
        this.map = new WeakMap();
        const gl = this.renderer.gl;
        //扩展重新开
        this.textureFloatExtension = gl.getExtension("OES_texture_float");
        //MAX_COMBINED_TEXTURE_IMAGE_UNITS
        const maxTextures = gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS);
        //重置
        this.boundTextures.length = maxTextures;
        for (var i = 0; i < maxTextures; i++)this.boundTextures[i] = null;
        //两种空纹理加上
        this.emptyTextures = {};
        this.emptyTextures[gl.TEXTURE_2D] = this.createTexture(gl.TEXTURE_2D, gl.TEXTURE_2D, 1);
        this.emptyTextures[gl.TEXTURE_CUBE_MAP] = this.createTexture(gl.TEXTURE_CUBE_MAP, gl.TEXTURE_CUBE_MAP_POSITIVE_X, 6);

        //全部通道都加上
        for (let i = 0; i < this.boundTextures.length; i++) {
            this.bind(null, i);
        }
        //
        this._managedTextures.length = 0;
        //GC的
        this._count = 0;
        this.checkCount = 0;
        this.maxIdle = 60 * 60;
        this.checkCountMax = 60 * 10;
        this.mode = GC_MODES.AUTO;
    }
    private _count: number;
    get count() {
        return this._count
    }
    private checkCount: number;
    private maxIdle: number;
    private checkCountMax: number;
    mode: GC_MODES;
    //处理GC
    onPostRender() {
        if (!this.renderer.renderingToScreen) return;
        this._count++;
        if (this.mode === GC_MODES.MANUAL) return;
        this.checkCount++;
        //计数没到
        if (this.checkCount <= this.checkCountMax) return;
        this.checkCount = 0;
        //执行移除
        const managedTextures = this._managedTextures;
        //会出现移除，数组长度会改变
        for (let i = managedTextures.length - 1; i >= 0; i--) {
            const texture = managedTextures[i];
            //去掉长期不用的纹理，不处理帧缓存的纹理
            if (!(texture as BaseRenderTexture).framebuffer && this._count - texture._touchedId > this.maxIdle) {
                //里面会移除managedTextures里对应的纹理
                this.destroyTexture(texture);
            }
        }
    }
    get(texture: Texture | BaseTexture): IWebglTexture {
        return this.map.get(this.getTrueBaseTexture(texture));
    }
    private getTrueBaseTexture(texture: Texture | BaseTexture): BaseTexture {
        if (!texture) return null;
        return (texture as Texture).baseTexture || texture as BaseTexture;
    }
    /**
     * 
     * @param {Texture} texture - TODO，可以是Texture或BaseTexture
     * @param {number} location - 纹理通道，传了就表示强制使用该通道
     * @return {number} 纹理通道
     */
    bind(texture: Texture | BaseTexture, location?: number): number {
        const gl = this.renderer.gl;
        //转成base；
        let baseTexture: BaseTexture = this.getTrueBaseTexture(texture);
        //没有就表示bind空
        if (!baseTexture || !baseTexture.hasLoaded) {
            location = location || 0;
            if (this.activeLocation !== location) {
                this.activeLocation = location;
                gl.activeTexture(gl.TEXTURE0 + location);
            }
            //绑定一个空的
            gl.bindTexture(gl.TEXTURE_2D, this.emptyTextures[gl.TEXTURE_2D]);
            this.boundTextures[location] = null;
            return location;
        }
        //用于gc标记
        baseTexture._touchedId = this.count;

        //从原先绑定过的纹理里找，如果location没传
        if (location == undefined) {
            for (let i = 0; i < this.boundTextures.length; i++) {
                if (this.boundTextures[i] === baseTexture) {
                    location = i;
                    break;
                }
            }
            if (location == undefined) {
                //没找到，尽量从后面的通道开始使用
                this._nextTextureLocation++;
                this._nextTextureLocation %= this.boundTextures.length;
                location = this.boundTextures.length - this._nextTextureLocation - 1;
            }
        }

        let glTexture = this.map.get(baseTexture);
        if (!glTexture) {
            glTexture = {
                dataId: -1,//保证下面的会更新
                styleId: -1,
                webglTexture: gl.createTexture()
            };
            //销毁的事件加上，destroyTexture做些啥，TODO
            baseTexture.addEventListener('dispose', this.destroyTexture, this);
            this.map.set(baseTexture, glTexture);
            //进管理
            this._managedTextures.push(baseTexture);
        }
        //是否当前激活的通道，不是就激活
        if (this.activeLocation !== location) {
            this.activeLocation = location;
            gl.activeTexture(gl.TEXTURE0 + location);
        }
        let hasBind: boolean = false;
        //维护的不一致说明该通道不对，修改
        if (this.boundTextures[location] !== baseTexture) {
            gl.bindTexture(gl.TEXTURE_2D, glTexture.webglTexture);
            this.boundTextures[location] = baseTexture;
            hasBind = true;
        }

        const { source, dataType, dataFormat, width, height, _dataId,
            _styleId, isPowerOfTwo, scaleMode, mipmap, wrapMode } = baseTexture

        //还有部分帧缓存的，TODO
        //如果数据变了，比如离屏canvas的，需要重新传数据
        if (glTexture.dataId !== _dataId) {
            glTexture.dataId = _dataId;
            if (!hasBind) {
                hasBind = true;
                gl.bindTexture(gl.TEXTURE_2D, glTexture.webglTexture);
            }
            //正常有source处理
            if (source) {
                //是否预乘纹理通道，好像需要在传数据前执行，pixelStorei设置纹理相关，应该都是要在数据上传前执行
                //所以这部分属性的维护，算进_dataId，后面可能还会有flip-y
                gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, baseTexture.flipY);
                gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, baseTexture.premultipliedAlpha);
                // texSubImage2D只有特殊时候才用到，直接使用texImage2D，
                //传gpu
                if (source._data) {//是data形式
                    //是否开了扩展
                    if (dataType == DATA_TYPE.FLOAT && !this.textureFloatExtension) {
                        throw new Error('floating point textures not available');
                    }
                    //TEXTURE_2D以后取base的target，TODO
                    gl.texImage2D(gl.TEXTURE_2D, 0, gl[dataFormat], width, height, 0, gl[dataFormat], gl[dataType], source._data);
                } else {
                    gl.texImage2D(gl.TEXTURE_2D, 0, gl[dataFormat], gl[dataFormat], gl[dataType], source);
                }
            }
            //如果是renderTexture，没有source的
            else {
                gl.texImage2D(
                    gl.TEXTURE_2D,
                    0,
                    gl[dataFormat],
                    width,
                    height,
                    0,
                    gl[dataFormat],
                    gl[dataType],
                    null
                );
            }
        }
        //属性的，现在是一次性的，_styleId在baseTexture里还没维护,TODO
        if (glTexture.styleId !== _styleId) {
            glTexture.styleId = _styleId;
            if (!hasBind) {
                hasBind = true;
                gl.bindTexture(gl.TEXTURE_2D, glTexture.webglTexture);
            }
            let glWrapMode = gl.CLAMP_TO_EDGE;
            if (isPowerOfTwo) {
                if (mipmap) {
                    gl.generateMipmap(gl.TEXTURE_2D);
                }
                if (wrapMode === WRAP_MODES.REPEAT) {
                    glWrapMode = gl.REPEAT;
                }
                else if (wrapMode === WRAP_MODES.MIRRORED_REPEAT) {
                    glWrapMode = gl.MIRRORED_REPEAT;
                }
            }
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, glWrapMode);
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, glWrapMode);

            let minMode, magMode;
            if (scaleMode === SCALE_MODES.NEAREST) {
                minMode = mipmap ? gl.NEAREST_MIPMAP_NEAREST : gl.NEAREST;
                magMode = gl.NEAREST;
            }
            else {
                minMode = mipmap ? gl.LINEAR_MIPMAP_LINEAR : gl.LINEAR;
                magMode = gl.LINEAR;
            }
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, minMode);
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, magMode);
        }

        return location;
    }

    /**
     * 解除纹理的绑定
     * @param texture 
     */
    unbind(texture: Texture | BaseTexture) {
        const gl = this.renderer.gl;

        let baseTexture: BaseTexture = this.getTrueBaseTexture(texture);

        for (let i = 0; i < this.boundTextures.length; i++) {
            if (this.boundTextures[i] === baseTexture) {
                this.boundTextures[i] = null;
                if (this.activeLocation !== i) {
                    gl.activeTexture(gl.TEXTURE0 + i);
                    this.activeLocation = i;
                }
                gl.bindTexture(gl.TEXTURE_2D, this.emptyTextures[gl.TEXTURE_2D]);
            }
        }
    }

    /**
     * TODO
     * @param {BaseTexture|Texture} texture 
     */
    destroyTexture(texture) {
        //如果是事件返回的
        if (texture.instanceType === "Event") {
            //选择它的target
            texture = texture.target;
        }
        texture = texture.baseTexture || texture;

        if (!texture.hasLoaded) {//TODO待处理，应该不这么判断了
            return;
        }

        const glTexture = this.map.get(texture);

        if (glTexture) {
            this.unbind(texture);
            //从gpu移除
            this.renderer.gl.deleteTexture(glTexture.webglTexture);
            //事件移除
            texture.removeEventListener('dispose', this.destroyTexture, this);
            //映射数据移除
            this.map.delete(texture);
            //
            const i = this._managedTextures.indexOf(texture);
            if (i !== -1) {
                this._managedTextures.splice(i, 1)
            }
        }
    }

    private createTexture(type: number, target: number, count: number) {
        const gl = this.renderer.gl;
        const data = new Uint8Array(4);
        const texture = gl.createTexture();
        gl.bindTexture(type, texture);
        //用NEAREST
        gl.texParameteri(type, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
        gl.texParameteri(type, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
        for (let i = 0; i < count; i++) {
            gl.texImage2D(target + i, 0, gl.RGBA, 1, 1, 0, gl.RGBA, gl.UNSIGNED_BYTE, data);
        }
        return texture;
    }

    destroy() {
        // destroy managed textures
        for (let i = this._managedTextures.length - 1; i >= 0; i--) {
            const texture = this._managedTextures[i];
            this.destroyTexture(texture);
        }
        //移除绑定事件
        this.renderer.removeEventListener('onContextChange', this.onContextChange, this);
        this.renderer.removeEventListener('onPostRender', this.onPostRender, this);
        this.renderer = null;
        this._managedTextures = null;
    }
}