import { BUFFER_TYPE } from "../../const";
import { DataAttribute, IDataAttribute } from "../webgl/DataAttribute";
import { InterleavedData, InterleavedDataAttribute } from "../webgl/InterleavedDataAttribute";
import { WebglRenderer } from "../WebglRenderer";

export interface IBufferData {
    buffer: WebGLBuffer,
    //gl.FLOAT
    type: number,
    /**
     * 比如4,是不是也可以是arrayBuffer(那type也得是各自的类型)
     */
    bytesPerElement: number,
    /**
     * 更新id，默认-1开始
     */
    updateId: number
}
//用于处理attribute对应的webglbuffer，传顶点数据用
export class BufferManager {
    private map: WeakMap<IDataAttribute, IBufferData>
    constructor(public renderer: WebglRenderer) {
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);
    }
    onContextChange() {
        this.map = new WeakMap();
    }
    get(attribute: DataAttribute | InterleavedDataAttribute | InterleavedData) {
        return this.map.get(this.getTrueAttribute(attribute));
    }
    /**
     * 主要方法，用于更新顶点数据
     * @param attribute 
     * @param bufferType 
     */
    update(attribute: DataAttribute | InterleavedDataAttribute | InterleavedData, bufferType: BUFFER_TYPE) {
        let attr = this.getTrueAttribute(attribute);
        const buffer = this.map.get(attr);
        if (!buffer) {
            this.map.set(attr, this.create(attr, bufferType));
        } else if (buffer.updateId < attr._updateId) {
            let gl = this.renderer.gl;
            gl.bindBuffer(gl[bufferType], buffer.buffer);
            //先都这样，后面有更新范围再说
            gl.bufferSubData(gl[bufferType], 0, attr.array);
            buffer.updateId = attr._updateId;
        }
    }
    remove(attribute: DataAttribute | InterleavedDataAttribute | InterleavedData) {
        let attr = this.getTrueAttribute(attribute)
        const buffer = this.map.get(attr);
        if (buffer) {
            this.renderer.gl.deleteBuffer(buffer.buffer);
            this.map.delete(attr);
        }
    }
    private create(attribute: IDataAttribute, bufferType: BUFFER_TYPE): IBufferData {
        const gl = this.renderer.gl;
        const { array, usage } = attribute;
        const buffer = gl.createBuffer();
        gl.bindBuffer(gl[bufferType], buffer);
        gl.bufferData(gl[bufferType], array, gl[usage]);
        let type = gl.FLOAT;
        if (array instanceof Float32Array) {
            type = gl.FLOAT;
        } else if (array instanceof Uint16Array) {
            type = gl.UNSIGNED_SHORT;
        } else if (array instanceof Int16Array) {
            type = gl.SHORT;
        } else if (array instanceof Uint32Array) {
            type = gl.UNSIGNED_INT;
        } else if (array instanceof Int32Array) {
            type = gl.INT;
        } else if (array instanceof Int8Array) {
            type = gl.BYTE;
        } else if (array instanceof Uint8Array) {
            type = gl.UNSIGNED_BYTE;
        } else if (array instanceof Uint8ClampedArray) {
            type = gl.UNSIGNED_BYTE;
        }

        return {
            buffer: buffer,
            type,
            bytesPerElement: array.BYTES_PER_ELEMENT,
            updateId: attribute._updateId
        };

    }
    private getTrueAttribute(attribute: DataAttribute | InterleavedDataAttribute | InterleavedData): IDataAttribute {
        let attr: IDataAttribute = attribute as IDataAttribute;
        //如果有data的，说明是InterleavedDataAttribute
        if ((attribute as InterleavedDataAttribute).data/*attribute.instanceType == "InterleavedDataAttribute"*/) {
            attr = (attribute as InterleavedDataAttribute).data
        }
        return attr;
    }
    destroy() {
        this.renderer.removeEventListener('onContextChange', this.onContextChange, this);
        this.renderer = null;
    }
}