import Container from "./Container";
import { RENDERER_TYPE, VERSION, osType } from "../const"
// import SystemRenderer from "../renderers/SystemRenderer";
import { Rectangle, Point } from "../math";
import { EventDispatcher } from "../events/EventDispatcher";
import { Event } from "../events/Event";
// import { FloatDisplay } from "./FloatDisplay";
import { DisplayObject } from "./DisplayObject";
import { MouseEvent } from "../events/MouseEvent";

import { WebglRenderer } from "../renderers/WebglRenderer";

import { CanvasRenderer } from "../renderers/CanvasRenderer";
import { SystemRenderer } from "../renderers/SystemRenderer";
import { getEnv, isWebGLSupported } from "../utils";

export class Stage extends Container {

    /**
     * 当前stage所使用的渲染器
     * 渲染器有两种,一种是canvas 一种是webGl
     * @property renderObj
     * @public
     * @since 1.0.0
     * @type {IRender}
     * @default null
     */
    public renderObj: SystemRenderer = null;


    /**
     * 直接获取stage的引用，避免总是从Event.ADD_TO_STAGE 事件中去获取stage引用
     * @property getStage
     * @param {string} stageName
     * @return {any}
     * @since 2.0.0
     */
    public static getStage(stageName: string = "cusEngine"): Stage {
        return Stage._stageList[stageName];
    }

    /**
     * @property _stageList
     * @static
     * @type {Object}
     * @private
     */
    private static _stageList: any = {};

    /**
     * 舞台在设备里截取后的可见区域,有些时候知道可见区域是非常重要的,因为这样你就可以根据舞台的可见区域做自适应了。
     * @property viewRect
     * @public
     * @readonly
     * @since 1.0.0
     * @type {Rectangle}
     * @default {x:0,y:0,width:0,height:0}
     * @readonly
     */
    public viewRect: Rectangle = new Rectangle();

    /**
     * 舞台的尺寸宽,也就是我们常说的设计尺寸
     * @property desWidth
     * @public
     * @since 1.0.0
     * @default 320
     * @type {number}
     * @readonly
     */
    public desWidth: number = 0;
    /**
     * 舞台的尺寸高,也就是我们常说的设计尺寸
     * @property desHeight
     * @public
     * @since 1.0.0
     * @default 240
     * @type {number}
     * @readonly
     */
    public desHeight: number = 0;
    /**
     * 舞台在当前设备中的真实高
     * @property divHeight
     * @public
     * @since 1.0.0
     * @default 320
     * @type {number}
     * @readonly
     */
    public divHeight: number = 0;
    /**
     * 舞台在当前设备中的真实宽
     * @property divWidth
     * @public
     * @since 1.0.0
     * @default 240
     * @readonly
     * @type {number}
     */
    public divWidth: number = 0;
    /**
     * 舞台的背景色
     * 默认就是透明背景
     * 可能设置一个颜色值改变舞台背景
     * @property bgColor
     * @public
     * @since 1.0.0
     * @type {number}
     * @default "";
     */
    private _bgColor: number = 0x000000;

    public get bgColor(): number {
        return this._bgColor;
    }
    /**
     * 设置颜色，即改变渲染器颜色
     */
    public set bgColor(value: number) {
        if (this._bgColor === value) return
        this._bgColor = value;
        this.renderObj.backgroundColor = value;
    }

    /**
     * 上一次鼠标或触碰经过的显示对象列表
     * @property _lastDpList
     * @type {Object}
     * @private
     */
    private _lastDpList: any = {};

    private _dpi: number;

    /**
     *
     */
    private canvas: HTMLCanvasElement;

    /**
     * 淘宝小程序环境canvas的偏移,淘宝环境才用,web环境实时,字段保留,但是不会计算了
     */
    private canvasOffsetTb: { x: number, y: number } = { x: 0, y: 0 };
    private offsetTimeId: any

    get stageWidth() {
        return this.viewRect.width;
    }

    get stageHeight() {
        return this.viewRect.height;
    }
    /**
     * 缩放模式
     */
    private _scaleMode: "fixedWidth" | "fixedHeight";
    /**
     * 舞台是否居中
     */
    private _stageCenter: boolean;
    /**
     * 建立舞台
     * 注意淘宝小程序canvas实际宽高/显示宽高必须是dpr，要么传dpr（参数resolution），要么外部自行设置实际宽高（canvas.width）及显示宽高（样式）
     * @param canvas canvas标签
     * @param desWidth 舞台设计宽，为了直接在舞台加东西不需要考虑缩放
     * @param desHeight 舞台设计高，为了直接在舞台加东西不需要考虑缩放
     * @param divWidth canvas显示宽，或视图宽度，或样式宽度，全屏就是屏幕宽度document.body.clientWidth(web环境)或my.getSystemInfoSync().windowWidth(淘宝小程序)
     * @param divHeight canvas显示高，或视图高度，或样式高度，全屏就是屏幕宽度document.body.clientHeight(web环境)或my.getSystemInfoSync().windowHeight(淘宝小程序)
     * @param renderType 渲染类型,默认canvas
     * @param stageCenter 舞台是否根据设计尺寸居中,默认false(左上角置顶)
     * @param fixedHeight 是否定高,默认false(定宽)
     * @param resolution 分辨率，一般传设备的devicePixelRatio，不传意味着外部自行处理了canvas的实际宽高及显示宽高（注意淘宝小程序实际宽高/显示宽高必须是dpr，要么传dpr，要么外部设置）
     * @param webglOptions 获取webgl上下文时需要的参数，一般用默认的，默认alpha:true,antialias:false,premultipliedAlpha:true,stencil:true,preserveDrawingBuffer:false
     */
    public constructor(
        canvas: HTMLCanvasElement,
        desWidth: number = 750,
        desHeight: number = 1624,
        divWidth: number,
        divHeight: number,
        renderType: RENDERER_TYPE = RENDERER_TYPE.CANVAS,
        stageCenter: boolean = false,
        fixedHeight: boolean = false,
        resolution?: number,
        webglOptions?: WebGLContextAttributes
    ) {
        super();
        this.canvas = canvas;
        let s: Stage = this;
        this._instanceType = "Stage";
        Stage._stageList["canvas"] = s;
        s.stage = this;
        s.name = "stageInstance_" + s.instanceId;
        s.desWidth = desWidth;
        s.desHeight = desHeight;
        s.divWidth = divWidth;
        s.divHeight = divHeight;

        if (!resolution) {//为了兼容以前的活动，以后删了
            //以前的活动外部设置了canvas的实际尺寸和显示尺寸
            console.warn("尝试设置分辨率，内部会根据分辨率设置画布的实际尺寸");
            //这里设置_dpi
            s._dpi = canvas.width / divWidth;
        } else {
            //canvas实际尺寸处理
            s._dpi = resolution;
            var cWidth = divWidth * resolution, cHeight = divHeight * resolution;
            if (getEnv() == "tb") {//淘宝小程序环境有个白边，所以加点像素
                cWidth += resolution;
                cHeight += resolution;
            }
            canvas.width = cWidth;
            canvas.height = cHeight;
        }


        //webgl不支持时的兼容，回退canvas
        if (renderType == RENDERER_TYPE.WEBGL && !isWebGLSupported()) {
            renderType = RENDERER_TYPE.CANVAS;
            console.warn("不支持webgl")
        }

        //打印个版本号
        console.log(
            "\n %c %c %c FYGE " + VERSION + " - \u2730 " + ["Unknown", "WebGL", "Canvas"][renderType] + " \u2730  %c  ",
            "background: #0077FA; padding:5px 0;",
            "background: #0077FA; padding:5px 0;",
            "color: #0095EE; background: #030307; padding:5px 0;",
            "background: #0077FA; padding:5px 0;",
        );

        if (renderType == RENDERER_TYPE.CANVAS) {
            let context = canvas.getContext("2d");
            s.renderObj = new CanvasRenderer(context, canvas.width, canvas.height);
        } else {
            let contextOptions: WebGLContextAttributes = {
                alpha: true,
                antialias: false,//抗锯齿尽量别开，耗性能
                premultipliedAlpha: true,  //一般png图片都不会预乘alpha，所以必为true,除非有些图集工具选择了premultipliedAlpha
                stencil: true,
                // preserveDrawingBuffer: true,//需要toDataUrl时要开，不开的话渲染一次后马上toDataUrl也能截到
                // powerPreference: this.options.powerPreference,
                ...webglOptions
            };
            let context = canvas.getContext("webgl", contextOptions) || canvas.getContext("experimental-webgl", contextOptions) as WebGLRenderingContext;
            s.renderObj = new WebglRenderer(context, canvas.width, canvas.height);
            //监听下上下文丢失的情况
            if (getEnv() == "web") {
                canvas.addEventListener(
                    "webglcontextlost",
                    (s.renderObj as WebglRenderer).handleContextLost,
                    false
                );
                canvas.addEventListener(
                    "webglcontextrestored",
                    (s.renderObj as WebglRenderer).handleContextRestored,
                    false
                );
            }

        }
        //TODO,还有一种方式,传入分辨率和显示宽高,canvas尺寸在里面修改,这样要多传个参数,暂时不改传参,不然版本问题,
        //到时修改这个放到上面,因为渲染器也用到了canvas的尺寸,
        //注意淘宝小程序环境必须先设置canvas的尺寸再getContext,类似CanvasRenderTarget构造函数里的问题
        //分辨率,
        // s._dpi = canvas.width / divWidth;//这里不需要了
        s._stageCenter = stageCenter;
        s._scaleMode = fixedHeight ? "fixedHeight" : "fixedWidth";
        //设置缩放及视窗
        s._setAlign();

        //延时派发舞台初始化事件
        setTimeout(function () {
            //派发事件
            s.dispatchEvent(Event.INIT_STAGE);
        }, 100);

        //淘宝环境额外处理canvas偏移,到时需要修改id传入，再说把，不需要了，淘宝上返回的就是canvas上的显示坐标
        // if (getEnv() == "tb") {//暂时先去掉吧，会出现一些问题，有需要，手动设置canvasOffsetTb的xy，屏幕坐标
        //     s.offsetTimeId = setInterval(() => {
        //         //@ts-ignore
        //         my.createSelectorQuery().select("#canvas").boundingClientRect().exec((r) => {
        //             if (r && r[0]) {
        //                 s.canvasOffsetTb.x = r[0].left || 0;
        //                 s.canvasOffsetTb.y = r[0].top || 0;
        //             }
        //         });
        //     }, 200)
        // }
    }
    /**
     * 暂时不存在外部调用的情况
     */
    private _setAlign() {
        let s = this;
        let canvas = s.canvas;
        //舞台设置居中处理锚点和位置偏移
        if (s._stageCenter) {
            s.anchorX = s.desWidth >> 1;
            s.anchorY = s.desHeight >> 1;
            s.x = (canvas.width - s.desWidth) >> 1;
            s.y = (canvas.height - s.desHeight) >> 1;
        }
        //默认定宽
        var scale = s._scaleMode == "fixedHeight" ? canvas.height / s.desHeight : canvas.width / s.desWidth;
        s.scaleX = scale;
        s.scaleY = scale;
        //视窗计算
        s.viewRect = s.viewRect || new Rectangle();
        s.viewRect.x = (s.desWidth - canvas.width / scale) >> 1;
        s.viewRect.y = (s.desHeight - canvas.height / scale) >> 1;
        s.viewRect.width = s.desWidth - s.viewRect.x * 2;
        s.viewRect.height = s.desHeight - s.viewRect.y * 2;

        //额外逻辑,不设置舞台居中时,x,y为左上角,置0
        if (!s._stageCenter) s.viewRect.x = s.viewRect.y = 0;
    }

    /**
     * 移动端不常用
     * 微信浏览器使用情景:ios返回页面下面出现操作栏
     * window.addEventListener("resize", () => {stage.resize()});
     * 这里会按照原_dpi修改canvas的实际尺寸
     * 一般设备的dpi不会改变,
     * web全屏环境可不传参数,否则自行计算显示尺寸传入
     * @param divWidth
     * @param divHeight
     */
    public resize(divWidth?: number, divHeight?: number) {
        if (getEnv() == "web") {//web环境不传当作全屏处理
            divWidth = divWidth || document.body.clientWidth;
            divHeight = divHeight || document.body.clientHeight;
        }
        if (!divWidth || !divWidth) {
            console.error("传入的显示尺寸不能为空")
            return
        }
        let s = this, d = s._dpi, c = s.canvas;
        //如果一致不修改
        if (divWidth == s.divWidth && divHeight == s.divHeight) return;
        //这里修改canvas尺寸
        if (divWidth != s.divWidth) c.width = divWidth * d;
        if (divHeight != s.divHeight) c.height = divHeight * d;
        //赋值显示宽高
        s.divWidth = divWidth;
        s.divHeight = divHeight;
        //修改缩放显示及视窗
        s._setAlign();
        //渲染器修改尺寸
        s.renderObj.resize(c.width, c.height);
        //派发事件
        s.dispatchEvent(Event.RESIZE);
    }

    /**
     * 这个是鼠标事件的MouseEvent对象池,因为如果用户有监听鼠标事件,如果不建立对象池,那每一秒将会new Fps个数的事件对象,影响性能
     * @property _ml
     * @type {Array}
     * @private
     */
    private _ml: any = [];
    /**
     * 这个是事件中用到的Point对象池,以提高性能
     * @property _mp
     * @type {Array}
     * @private
     */
    private _mp: any = [];

    /**
     * 刷新mouse或者touch事件
     * @method _initMouseEvent
     * @private
     */
    private _initMouseEvent(event: MouseEvent, cp: Point, sp: Point, identifier: number): void {
        event["_pd"] = false;
        event.clientX = cp.x;
        event.clientY = cp.y;
        event.stageX = sp.x;
        event.stageY = sp.y;
        event.identifier = identifier;
    }

    //每一个手指事件的对象池
    /**
     * @property _mouseDownPoint
     * @type {Object}
     * @private
     */
    private _mouseDownPoint: any = {};

    /**
     * 循环刷新页面的函数
     * @method flush
     * @private
     * @return {void}
     */
    public flush(): void {
        this.renderObj.render(this);
    }

    /**
     * html的鼠标或单点触摸对应的引擎事件类型名
     * touchcancel:"onMouseUp"不常用，先不加
     * @property _mouseEventTypes
     * @type {{mousedown: string, mouseup: string, mousemove: string, touchstart: string, touchmove: string, touchend: string}}
     * @private
     */
    private _mouseEventTypes: any = {
        //pc
        mousedown: "onMouseDown",
        mousemove: "onMouseMove",
        mouseup: "onMouseUp",
        //mobile
        touchstart: "onMouseDown",
        touchmove: "onMouseMove",
        touchend: "onMouseUp",
        //tbMini
        touchStart: "onMouseDown",//小程序返回的事件名是驼峰的
        touchMove: "onMouseMove",
        touchEnd: "onMouseUp"
    };
    /**
     * 舞台的鼠标事件是否要阻止默认事件，默认false
     * 对于同时写了监听pc和移动端的，
     */
    public webMouseEventPreventDefault: boolean;
    /**
     * 舞台的鼠标事件是否要阻止事件冒泡，默认false
     */
    public webMouseEventStopPropagation: boolean;
    /**
     * 无多指，无拖动
     * @method onMouseEvent
     * @param e
     * @private
     */
    onMouseEvent(e: any): void {
        //如果是web环境，且设置了阻止默认事件
        if (getEnv() == "web") {
            if (this.webMouseEventPreventDefault) e.preventDefault();
            if (this.webMouseEventStopPropagation) e.stopPropagation();
        }
        let s: Stage = this;
        //检查mouse或touch事件是否有，如果有的话，就触发事件函数
        if (EventDispatcher._totalMEC > 0) {
            let points: any;
            //事件类型

            let item = s._mouseEventTypes[e.type];
            let events: any;
            let event: any;
            //stageMousePoint
            let sp: Point;
            //localPoint;
            let lp: Point;
            //clientPoint
            let cp: Point;
            //事件个数
            let eLen: number;
            let identifier: any;
            if (osType == "pc") {
                e.identifier = 0;
                points = [e];
            } else {
                // points = [e.changedTouches[0]];
                points = e.changedTouches;
            }
            // points = [e.changedTouches[0]];
            //  points = [e.touches[0]];//不能用这个
            let { x: offsetX, y: offsetY } = s.canvasOffsetTb;
            //@ts-ignore
            if (getEnv() == "web") {
                let doc = document.documentElement;
                let box = s.canvas.getBoundingClientRect();
                offsetX = box.left + window.pageXOffset - doc.clientLeft;
                offsetY = box.top + window.pageYOffset - doc.clientTop;
            }


            for (let o = 0; o < points.length; o++) {
                if (!points[o]) continue;
                eLen = 0;
                events = [];
                identifier = "m" + points[o].identifier;
                if (s._mp.length > 0) {
                    cp = s._mp.shift();
                } else {
                    cp = new Point();
                }
                cp.x = ((points[o].pageX || points[o].x || points[o].b) - offsetX) * s._dpi// devicePixelRatio;
                cp.y = ((points[o].pageY || points[o].y || points[o].c) - offsetY) * s._dpi// devicePixelRatio;
                // my.alert({
                //   title: JSON.stringify(points[o])
                // });
                //计算舞台中的点
                sp = s.globalToLocal(cp, DisplayObject._bp);
                //检查是否有鼠标事件
                if (EventDispatcher.getMouseEventCount() > 0) {
                    if (!s._ml[eLen]) {
                        event = new MouseEvent(item);
                        s._ml[eLen] = event;
                    } else {
                        event = s._ml[eLen];
                        event.type = item;
                    }
                    events[events.length] = event;
                    s._initMouseEvent(event, cp, sp, identifier);
                    eLen++;
                }
                if (item == "onMouseDown") {
                    s._mouseDownPoint[identifier] = cp;
                    //清空上次存在的显示列表
                } else if (item == "onMouseUp") {
                    if (s._mouseDownPoint[identifier]) {
                        if (Point.distance(s._mouseDownPoint[identifier], cp) < 20) {
                            //检查是否有添加对应的click事件
                            if (EventDispatcher.getMouseEventCount("onMouseClick") > 0) {
                                if (!s._ml[eLen]) {
                                    event = new MouseEvent("onMouseClick");
                                    s._ml[eLen] = event;
                                } else {
                                    event = s._ml[eLen];
                                    event.type = "onMouseClick";
                                }
                                events[events.length] = event;
                                s._initMouseEvent(event, cp, sp, identifier);
                                eLen++;
                            }
                        }
                    }
                }
                if (eLen > 0) {
                    //有事件开始遍历显示列表
                    //找出最底层的显示对象
                    let d: DisplayObject = s.hitTestPoint(cp, true);
                    // console.log(d)
                    let displayList: Array<DisplayObject> = [];
                    // my.alert({
                    //   title: "55729:" + d.instanceId
                    // });
                    if (d) {
                        //证明有点击到事件,然后从最底层追上来,看看一路是否有人添加过mouse或touch事件,还要考虑mousechildren和阻止事件方法
                        //找出真正的target,因为有些父级可能会mouseChildren=false;
                        while (d) {
                            if (d["mouseChildren"] === false) {
                                //丢掉之前的层级,因为根本没用了
                                displayList.length = 0;
                            }
                            displayList[displayList.length] = d;
                            d = d.parent;
                        }
                    } else {
                        displayList[displayList.length] = s;
                    }
                    let len: number = displayList.length;
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[0];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j]);
                                }
                            }
                        }
                    }
                    //这里一定要反转一下，因为会影响mouseOut mouseOver
                    displayList.reverse();
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type, false) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[eLen - 1];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j], null, false);
                                }
                            }
                        }
                    }
                    //最后要和上一次的遍历者对比下，如果不相同则要触发onMouseOver和onMouseOut
                    if (item != "onMouseDown") {
                        if (EventDispatcher.getMouseEventCount("onMouseOver") > 0 || EventDispatcher.getMouseEventCount("onMouseOut") > 0) {
                            if (s._lastDpList[identifier]) {
                                //从第二个开始，因为第一个对象始终是stage顶级对象
                                let len1 = s._lastDpList[identifier].length;
                                let len2 = displayList.length;
                                len = len1 > len2 ? len1 : len2;
                                let isDiff = false;
                                let overEvent: MouseEvent;
                                let outEvent: MouseEvent;
                                for (let i = 1; i < len; i++) {
                                    if (!isDiff) {
                                        if (s._lastDpList[identifier][i] != displayList[i]) {
                                            //确定哪些有onMouseOver,哪些有onMouseOut
                                            isDiff = true;
                                            if (!s._ml[eLen]) {
                                                overEvent = new MouseEvent("onMouseOver");
                                                s._ml[eLen] = overEvent;
                                            } else {
                                                overEvent = s._ml[eLen];
                                                overEvent.type = "onMouseOver";
                                            }
                                            s._initMouseEvent(overEvent, cp, sp, identifier);
                                            eLen++;
                                            if (!s._ml[eLen]) {
                                                outEvent = new MouseEvent("onMouseOut");
                                                s._ml[eLen] = outEvent;
                                            } else {
                                                outEvent = s._ml[eLen];
                                                outEvent.type = "onMouseOut";
                                            }
                                            s._initMouseEvent(outEvent, cp, sp, identifier);
                                        }
                                    }
                                    if (isDiff) {
                                        if (s._lastDpList[identifier][i]) {
                                            //触发onMouseOut事件
                                            if (!outEvent["_pd"]) {
                                                d = s._lastDpList[identifier][i];
                                                if (d.hasEventListener("onMouseOut")) {
                                                    outEvent.currentTarget = s._lastDpList[identifier][len1 - 1];
                                                    outEvent.target = d;
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    outEvent.localX = lp.x;
                                                    outEvent.localY = lp.y;
                                                    d.dispatchEvent(outEvent);
                                                }
                                            }
                                        }
                                        if (displayList[i]) {
                                            //触发onMouseOver事件
                                            if (!overEvent["_pd"]) {
                                                d = displayList[i];
                                                if (d.hasEventListener("onMouseOver")) {
                                                    overEvent.currentTarget = displayList[len2 - 1];
                                                    overEvent.target = d;
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    overEvent.localX = lp.x;
                                                    overEvent.localY = lp.y;
                                                    d.dispatchEvent(overEvent);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        s._mp[s._mp.length] = cp;
                    }
                    if (item == "onMouseUp") {
                        delete s._mouseDownPoint[identifier];
                        delete s._lastDpList[identifier];
                    } else {
                        s._lastDpList[identifier] = displayList;
                    }
                }
            }
        }
    };
    /**
     * 给舞台绑定canvas标签的鼠标代理事件（web环境下需要显示对象的鼠标事件时调用）
     * 自行调用，内部不主动调用，因为比如淘宝小程序环境方式不一致，需要用onMouseEvent
     */
    addWebMouseEvent() {
        //非网页环境直接return
        if (getEnv() !== "web") return;
        var canvas = this.canvas;
        var mouseEvent = this.onMouseEvent.bind(this);
        if (osType == "pc") {
            canvas.addEventListener("mousedown", mouseEvent, false);
            canvas.addEventListener("mousemove", mouseEvent, false);
            canvas.addEventListener("mouseup", mouseEvent, false);
        } else {
            canvas.addEventListener("touchstart", mouseEvent, false);
            canvas.addEventListener("touchmove", mouseEvent, false);
            canvas.addEventListener("touchend", mouseEvent, false);
        }
    }
    /**
     * 直接修改了，用视窗，
     * 一般用于滤镜等等，对于舞台，超出viewPort的肯定不显示，没必要测量（后续考虑也测量，万一测量出的更小呢）
     */
    public getBounds(): Rectangle {
        return this.viewRect;
    }

    public destroy(): void {
        let s = this;
        // if (getEnv() == "tb") clearInterval(s.offsetTimeId);
        //web环境下，渲染方式webgl模式时，移除webgl上下文丢失事件
        if (getEnv() == "web" && s.renderObj.type == RENDERER_TYPE.WEBGL) {
            s.canvas.removeEventListener(
                "webglcontextlost",
                (s.renderObj as WebglRenderer).handleContextLost,
                false
            );
            s.canvas.removeEventListener(
                "webglcontextrestored",
                (s.renderObj as WebglRenderer).handleContextRestored,
                false
            );
        }
        //然后销毁渲染器
        s.renderObj.destroy();
        s.renderObj = null;
        s.viewRect = null;
        s._lastDpList = null;
        s._ml = null;
        super.destroy();
    }
}
