import BaseTexture from './BaseTexture';
import { SCALE_MODES } from '../const';

/**
 * 将显示对象画在上面的贴图，这是基础纹理，一般用RenderTexture
 * 注意绘制前纹理加载完成
 * For example:
 *
 * ```js
 * let renderer = CanvasRenderer(context,1024, 1024);
 * let baseRenderTexture = new BaseRenderTexture(800, 600);
 * let renderTexture = new RenderTexture(baseRenderTexture);
 * let sprite = Sprite.fromImage("spinObj_01.png");
 *
 * sprite.position.x = 800/2;
 * sprite.position.y = 600/2;
 * sprite.anchorTexture.x = 0.5;
 * sprite.anchorTexture.y = 0.5;
 *
 * renderer.render(sprite,renderTexture);
 * ```
 *
 *
 * @class
 * @extends BaseTexture
 */
export default class BaseRenderTexture extends BaseTexture {
    /**
     * webgl帧缓存列表，键值为渲染器的id
     * 画在他的帧缓存中
     * @private
     * @member {object<number, WebGLTexture>}
     */
    _glRenderTargets: {};
    /**
     * 离屏canvas，多canvas渲染器可共享，所以不需要列表
     * @private
     * @member {CanvasRenderTarget}
     */
    _canvasRenderTarget: any;
    /**
     * 是否可用
     */
    valid: boolean;
    /**
     * @param {number} [width=100] - 宽度
     * @param {number} [height=100] - 高度
     * @param {number} [scaleMode=SCALE_MODES.LINEAR] -  {@link SCALE_MODES}
     */
    constructor(width: number = 100, height: number = 100, scaleMode: number = SCALE_MODES.LINEAR) {
        super(null, scaleMode);

        this.width = Math.ceil(width);
        this.height = Math.ceil(height);

        this.scaleMode = scaleMode;
        this.hasLoaded = true;


        this._glRenderTargets = {};


        this._canvasRenderTarget = null;


        this.valid = false;
    }

    /**
     * 重置尺寸
     * @param {number} width - 宽度
     * @param {number} height - 高度
     */
    resize(width: number, height: number) {
        width = Math.ceil(width);
        height = Math.ceil(height);

        if (width === this.width && height === this.height) {
            return;
        }

        this.valid = (width > 0 && height > 0);

        this.width = width;
        this.height = height;

        if (!this.valid) {
            return;
        }

        this.dispatchEvent('update');
    }

    /**
     * 销毁
     */
    destroy() {
        super.destroy();
    }
}
