import { EventDispatcher } from '../core/EventDispatcher';
import { TOUCH } from '../constants';
import { Quaternion } from '../math/Quaternion';
import { Vector2 } from '../math/Vector2';
import { Vector3 } from '../math/Vector3';
import { Spherical } from '../math/Spherical';

export class OrbitControls {
	enabled = true
	enableDamping = true
	dampingFactor = 0.25
	userZoom = true
	userZoomSpeed = 1.0
	userRotate = true
	userRotateSpeed = 1.0
	autoRotate = false
	autoRotateSpeed = 2.0
	//最小旋转角
	minPolarAngle = 0
	//最大旋转角
	maxPolarAngle = Math.PI
	minDistance = 0
	maxDistance = Infinity
	enablePitch = true
	constructor(camera) {
		this.camera = camera
		//相机目标点
		var center = this.center = new Vector3();

		// this.userPan = true;
		// this.userPanSpeed = 2.0;
		var EPS = 0.000001;
		var PIXELS_PER_ROUND = 1800;

		var rotateStart = new Vector2();
		var rotateEnd = new Vector2();
		var rotateDelta = new Vector2();

		var phiDelta = 0;
		var thetaDelta = 0;
		var scale = 1;

		function getAutoRotationAngle() {
			return 2 * Math.PI / 60 / 60 * this.autoRotateSpeed;
		}

		function getZoomScale() {
			return Math.pow(0.95, this.userZoomSpeed);
		}

		function rotateLeft(angle = getAutoRotationAngle()) {
			thetaDelta -= angle;
		};


		function rotateUp(angle = getAutoRotationAngle()) {
			phiDelta -= angle;
		};

		function zoomIn(zoomScale = getZoomScale()) {
			scale /= zoomScale;
		};
		function zoomOut(zoomScale = getZoomScale()) {
			scale *= zoomScale;
		};

		this.update = () => {
			if (
				Math.abs(thetaDelta) < 0.0001 &&
				Math.abs(phiDelta) < 0.0001 &&
				Math.abs(scale - 1) < 0.0001 &&
				!this.autoRotate
			) {
				return
			}

			var position = this.camera.position;
			var offset = position.clone().sub(center);
			var theta = Math.atan2(offset.x, offset.z);
			var phi = Math.atan2(Math.sqrt(offset.x * offset.x + offset.z * offset.z), offset.y);

			if (this.autoRotate) rotateLeft();

			theta += thetaDelta;
			phi += phiDelta;

			phi = Math.max(this.minPolarAngle, Math.min(this.maxPolarAngle, phi));

			phi = Math.max(EPS, Math.min(Math.PI - EPS, phi));

			var radius = offset.length() * scale;

			// restrict radius to be between desired limits
			radius = Math.max(this.minDistance, Math.min(this.maxDistance, radius));

			offset.x = radius * Math.sin(phi) * Math.sin(theta);
			offset.y = radius * Math.cos(phi);
			offset.z = radius * Math.sin(phi) * Math.cos(theta);

			//相机位置
			position.copy(center).add(offset);

			//相机目标点
			this.camera.lookAt(center.x, center.y, center.z);

			//恢复
			if (this.enableDamping) {
				thetaDelta *= (1 - this.dampingFactor);
				phiDelta *= (1 - this.dampingFactor);
			} else {
				thetaDelta = 0;
				phiDelta = 0;
			}

			scale = 1;
			// if (lastPosition.distanceTo(this.object.position) > 0) {
			//     this.dispatchEvent(changeEvent);
			//     lastPosition.copy(this.object.position);
			// }
		};
		//记录两个手指唯一标识
		let fingles = [];
		//距离
		let deltaDis = 0;
		this.onTouchStart = (e) => {
			if (!this.enabled) return;
			//每个处理
			for (var i = 0; i < e.changedTouches.length; i++) {
				let { x, y, identifier } = e.changedTouches[i];
				fingles.push({ x, y, id: identifier })
				if (fingles.length >= 2) {
					let f1 = fingles[fingles.length - 1];
					let f2 = fingles[fingles.length - 2];
					deltaDis = Math.sqrt(Math.pow(f1.x - f2.x, 2) + Math.pow(f1.y - f2.y, 2));
				}
				rotateStart.set(e.changedTouches[0].x, e.changedTouches[0].y);
			}
		}
		this.onTouchMove = (e) => {
			if (!this.enabled) return
			if (!fingles.length) return;
			for (var i = 0; i < e.changedTouches.length; i++) {
				if (!e.changedTouches[i]) continue;
				let { x, y, identifier } = e.changedTouches[i];
				if (fingles.length == 1) {
					if (!this.userRotate) return;
					if (!rotateStart.x && !rotateStart.y) return;
					rotateEnd.set(x, y);
					rotateDelta.subVectors(rotateEnd, rotateStart);
					rotateLeft(2 * Math.PI * rotateDelta.x / PIXELS_PER_ROUND * this.userRotateSpeed);
					if (this.enablePitch) rotateUp(2 * Math.PI * rotateDelta.y / PIXELS_PER_ROUND * this.userRotateSpeed);
					rotateStart.copy(rotateEnd);
				}
				//多指，取最后两个
				else {
					if (this.userZoom === false) return;
					var oneF = fingles[fingles.length - 1];
					var twoF = fingles[fingles.length - 2];
					if (identifier == oneF.id) {
						oneF.x = x
						oneF.y = y
					} else if (identifier == twoF.id) {
						twoF.x = x
						twoF.y = y
					} else {
						continue
					}
					var cDelta = Math.sqrt(Math.pow(oneF.x - twoF.x, 2) + Math.pow(oneF.y - twoF.y, 2));
					if (cDelta - deltaDis > 0) {
						zoomOut(Math.pow(0.95, cfg.userZoomSpeed / 3));//手势的执行太快了
					} else {
						zoomIn(Math.pow(0.95, cfg.userZoomSpeed / 3));
					}
					deltaDis = cDelta;
				}
			}
			this.update();
		}

		this.onTouchEnd = (e) => {//淘宝有bug，全部移除先
			fingles.length = 0;
		}

	}
}






