import { createCanvas } from "./utils";
import { Sprite } from "./display";
import { Texture } from "./texture";

export class FpsPanel extends Sprite {
  private context: CanvasRenderingContext2D;
  private bgColor: string = '#002002';
  private textColor: string = '#0ff0ff';
  private PR: number = 3;
  private WIDTH = 80 * this.PR;
  private HEIGHT = 48 * this.PR;
  private TEXT_X = 3 * this.PR;
  private TEXT_Y = 2 * this.PR;
  private GRAPH_X = 3 * this.PR;
  private GRAPH_Y = 15 * this.PR;
  private GRAPH_WIDTH = 74 * this.PR;
  private GRAPH_HEIGHT = 30 * this.PR;
  private GRAPH_SIZE = 74;
  private maxValue = 100;
  private min = Infinity;
  private max = 0;
  private items = [];
  /**
   * 帧率面板
   * 后续可以加入每次drawCall，总绘制对象等等
   */
  constructor() {
    super()
    //离屏canvas
    var canvas = createCanvas();
    canvas.width = this.WIDTH;
    canvas.height = this.HEIGHT;
    this.texture = Texture.fromCanvas(canvas)
    this.context = canvas.getContext("2d");
    this.context.font = 'bold ' + (9 * this.PR) + 'px Helvetica,Arial,sans-serif';
    this.context.textBaseline = 'top';
    this.updateText("FPS");
  }
  private lastTime: number = Date.now();
  private frames: number = 0;
  reset() {
    this.lastTime = Date.now();
    this.min = Infinity;
    this.max = 0;
    this.items.length = 0;
    this.frames = 0;
  }
  update() {
    this.frames++;
    var time = Date.now();
    //每秒跑一次
    if (time >= this.lastTime + 1000) {
      var value = (this.frames * 1000) / (time - this.lastTime);
      this.updatePanel(value)
      this.lastTime = time;
      this.frames = 0;
    }
    super.update();
  }
  private updatePanel(value: number) {
    const { items, GRAPH_SIZE, context, GRAPH_X, textColor, GRAPH_Y, PR, GRAPH_HEIGHT, bgColor, maxValue } = this
    items.push(value);
    while (items.length > GRAPH_SIZE) {
      items.shift();
    }
    this.min = Math.min(this.min, value);
    this.max = Math.max(this.max, value);

    this.updateText(Math.round(value) + ' FPS (' + Math.round(this.min) + '-' + Math.round(this.max) + ')');

    for (var i = 0; i < items.length; i++) {
      var startPos = GRAPH_X + (i + GRAPH_SIZE - items.length) * PR;

      context.fillStyle = textColor;
      context.globalAlpha = 1;
      context.fillRect(startPos, GRAPH_Y, PR, GRAPH_HEIGHT);

      context.fillStyle = bgColor;
      context.globalAlpha = 0.9;
      context.fillRect(startPos, GRAPH_Y, PR, Math.round((1 - (items[i] / maxValue)) * GRAPH_HEIGHT));
    }
    this.texture.update();
  }
  private updateText(text: string) {
    const { context, bgColor, textColor, WIDTH, HEIGHT, TEXT_X, TEXT_Y, GRAPH_X, GRAPH_Y, GRAPH_WIDTH, GRAPH_HEIGHT } = this

    context.fillStyle = bgColor;
    context.globalAlpha = 1;
    context.fillRect(0, 0, WIDTH, HEIGHT);

    context.fillStyle = textColor;
    context.fillText(text, TEXT_X, TEXT_Y);


    context.fillRect(GRAPH_X, GRAPH_Y, GRAPH_WIDTH, GRAPH_HEIGHT);

    context.fillStyle = bgColor;
    context.globalAlpha = 0.9;
    context.fillRect(GRAPH_X, GRAPH_Y, GRAPH_WIDTH, GRAPH_HEIGHT);
  }
}

