import { Event } from "../events/Event";
import { Container } from "../display";
import { ScrollView, ScrollEvent } from "./ScrollView";


/**
 * 滚动类的Item基类
 * @class IScrollListItem
 * @public
 * @extends Container
 * @since 2.0.0
 */
export class ScrollListItem extends Container {
    data: any;
    index: number;
    /**
     * 列表滚动时各项item有数据变更时会触发
     * @param data 参数会传入data，也可直接使用this.data
     * @param index data所在数组的索引
     * @protected
     */
    onDataChanged(data?: any, index?: number) {

    };
}

/**
 * 滚动列表
 * @class ScrollList
 * @public
 * @extends ScrollView
 * @since 1.0.9
 */
export class ScrollList extends ScrollView {
    private _data: any[];
    /**
     * 数据列表，
     */
    public get data() {
        return this._data;
    };
    /**
     * list里不允许改变滚动尺寸范围，只能通过数据改变
     */
    get scrollWidth() {
        return this._scrollWidth
    }
    set scrollWidth(v) {
        console.warn("ScrollWidth can not be set manually in ScrollLists!")
    }
    /**
     * list里不允许改变滚动尺寸范围，只能通过数据改变
     */
    get scrollHeight() {
        return this._scrollHeight
    }
    set scrollHeight(v) {
        console.warn("ScrollHeight can not set manually in ScrollLists!")
    }

    private _items: Array<ScrollListItem> = null;
    private _itemW: number;
    private _itemH: number;
    private _itemRow: number;
    private _itemCol: number;
    private _itemCount: number;
    private _itemClass: typeof ScrollListItem;

    private _isInit: number = 0;


    private _cols: number;
    private _lastFirstId: number = -1;

    private _isVertical: boolean;
    /**
     * 构造函数
     * @method ScrollList
     * @param {Class} itemClass 可以做为Item的类，比如ScrollListItem，必须具备onDataChanged方法
     * @param {number} itemWidth item宽
     * @param {number} itemHeight item高
     * @param {number} vW 列表的宽
     * @param {number} vH 列表的高
     * @param {boolean} isVertical 是横向滚动还是纵向滚动 默认是纵向
     * @param {number} cols 分几列，默认是1列
     * @since 2.0.0
     */
    constructor(
        itemClass: typeof ScrollListItem,
        itemWidth: number,
        itemHeight: number,
        vW: number,
        vH: number,
        isVertical: boolean = true,
        cols: number = 1
    ) {
        super(vW, vH, 0, 0);
        let s = this;
        s._instanceType = "ScrollList";
        s._itemW = itemWidth;
        s._itemH = itemHeight;
        s._items = [];
        s._itemClass = itemClass;
        s._itemCount = 0;
        s._cols = cols;
        s._isVertical = isVertical;
        s._updateRowCol();
        //滚动时需要刷新数据
        s.addEventListener(Event.SCROLLING, s.flushData, s);
    }

    /**
     * 设置列表数据
     * @method setData
     * @param {Array} data
     * @param {boolean} isReset 是否重置数据列表。默认为true
     * @since 2.0.0
     */
    public setData(data: Array<any>, isReset: boolean = true): void {
        let s = this;
        data = data || [];
        if (!s._isInit || isReset) {
            s._data = data;
        } else {
            s._data = s._data.concat(data);
        }
        s._isInit = 1;
        s._lastFirstId = -1;
        let maxDistance = Math.ceil(s.data.length / s._cols) * s._itemRow;
        s[s._isVertical ? "_scrollHeight" : "_scrollWidth"] = maxDistance;
        //检查一下位置
        if (!s.checkPosition(s.bounceTime)) {
            //没滚动需要刷新数据
            s.flushData();
        };
    }

    public updateData(data: Array<any>, isReset: boolean = true) {
        console.warn("Method updateData will be abandoned soon,use method setData instead!")
        this.setData(data, isReset);
    }

    private flushData(e?: ScrollEvent) {
        let s: ScrollList = this;
        if (!s._isInit) return;
        let dimension: "x" | "y", antDimension: "x" | "y", speed: number;
        if (s._isVertical) {
            dimension = "y";
            antDimension = "x";
            if (e && e.data) speed = e.data.deltaY;
        } else {
            dimension = "x";
            antDimension = "y";
            if (e && e.data) speed = e.data.deltaX;
        }

        let dis = s._content[dimension];

        let id: number = dis > 0 ? 0 : (Math.abs(Math.floor(dis / s._itemRow)) - 1) * s._cols;
        //如果dis是0的时候会是-1
        id = Math.max(0, id);
        // console.log(id)
        //确定下和之前记录第一项id是否一致，一致就没必要改了
        if (id == s._lastFirstId) return
        s._lastFirstId = id;
        //重新排数组，尽量最少的数据刷新
        if (speed) {
            for (let r = 0; r < s._cols; r++) {
                //向下滚动时
                if (speed < 0) {
                    s._items.unshift(s._items.pop());
                }
                //向上滚动时
                else {
                    s._items.push(s._items.shift());
                }
            }
        }

        for (let i = 0; i < s._itemCount; i++) {
            let item: ScrollListItem = s._items[i];
            if (s._isInit == 1) {
                item.index = -1;
            }
            if (item.index !== id) {
                //id在data数组列表内就执行
                if (id < s.data.length) {
                    item.index = id;
                    item.onDataChanged(item.data = s.data[id], id);
                    item.visible = true;
                    //位置
                    item[dimension] = Math.floor(id / s._cols) * s._itemRow;
                    item[antDimension] = (id % s._cols) * s._itemCol;
                } else {
                    item.data = null;
                    item.index = -1;
                    item.visible = false;
                    //位置需要刷新吗？
                }
            }
            id++;
        }
        s._isInit = 2;
    }

    /**
     * 设置可见区域，可见区域的坐标始终在本地坐标中0,0点位置
     * @method setViewRect
     * @param {number}w 设置可见区域的宽
     * @param {number}h 设置可见区域的高
     * @param {boolean} isVertical 方向
     * @public
     * @since 2.0.0
     */
    public setViewRect(w: number, h: number, isVertical: boolean = true): void {
        super.setViewRect(w, h, false);
        let s = this;
        let same = s._isVertical === isVertical;
        s._isVertical = isVertical;
        //有行列时才需要更新
        if (s._itemRow && s._itemCol) {
            s._updateRowCol();
            //先这样偷懒处理
            if (!same) {
                let oriTime = s.bounceTime;
                s.bounceTime = 0;
                s.setData(s._data, true);
                s.bounceTime = oriTime;
            } else {
                s.flushData();
            }
        }
    }

    private _updateRowCol() {
        let s = this, distance: number;
        if (s._isVertical) {
            s._itemRow = s._itemH;
            s._itemCol = s._itemW;
            distance = s.viewHeight;
            s._scrollWidth = 0;
        } else {
            s._itemRow = s._itemW;
            s._itemCol = s._itemH;
            distance = s.viewWidth;
            s._scrollHeight = 0;
        }
        let newCount: number = (Math.ceil(distance / s._itemRow) + 1) * s._cols;
        if (newCount != s._itemCount) {
            if (newCount > s._itemCount) {
                for (let i = s._itemCount; i < newCount; i++) {
                    let item = new s._itemClass();
                    item.visible = false;
                    item.data = null;
                    s._items.push(item);
                    s._content.addChild(item);
                }
            } else {
                for (let i = 0; i < s._itemCount - newCount; i++) {
                    s._content.removeChild(s._items.pop());
                }
            }
            s._itemCount = newCount;
            s._lastFirstId = -1;
        }
    }
    /**
     * 滚动到指定位置
     * @param distance 滚动距离，一般为正
     * @param time 滚动时间，默认0
     */
    scrollTo(distance: number, time: number = 0) {
        let x = 0, y = 0;
        if (this._isVertical) {
            y = distance;
        } else {
            x = distance;
        }
        super.scrollTo(x, y, time);
    }
    /**
     * 从当前位置开始滚动
     * @param distance 滚动距离，一般为正
     * @param time 滚动时间，默认0
     */
    scrollBy(distance: number, time: number = 0) {
        let x = 0, y = 0;
        if (this._isVertical) {
            y = distance;
        } else {
            x = distance;
        }
        super.scrollBy(x, y, time);
    }

    public destroy(): void {
        let s = this;
        s._items = null;
        s._itemClass = null;
        s._data = null;
        super.destroy();
    }
}
//list不允许操作任何子级
Container._childrenOperationMethods.forEach((v) => {
    Object.defineProperty(ScrollList.prototype, v, {
        value: function (...arg) {
            console.warn("You'd better do not operate children in a ScrollList or its view!");
            return Container.prototype[v].call(this, ...arg)
            //先不重写了
            // console.warn("ScrollLists can not operate children any more!")
        },
        writable: true,
        enumerable: true,
    })
})