import { hex2string, hex2rgb } from '../utils';
import { RENDERER_TYPE, devicePixelRatio } from '../const';
import Container from '../display/Container';
import { EventDispatcher } from '../events/EventDispatcher';
import { DisplayObject } from "../display/DisplayObject";


/**
 * 渲染器抽象类，用于canvas和webgl
 */
export class SystemRenderer extends EventDispatcher {
    /**
     * 渲染类型
     *
     * @member {number}
     * @default RENDERER_TYPE.UNKNOWN
     * @see RENDERER_TYPE
     */
    type: RENDERER_TYPE;

    width: number;
    height: number;
    /**
     * 是否透明
     */
    transparent: boolean;
    /**
     * 背景色，十六进制数值
     * @member {number}
     */
    protected _backgroundColor: number;
    /**
     * 背景色，rgba值
     * @member {number[]}
     */
    protected _backgroundColorRgba: number[];
    /**
     * 背景色，字符串
     */
    protected _backgroundColorString: string;
    /**
     * 临时父级对象，用于更新updateTransform
     */
    protected _tempDisplayObjectParent: Container;
    /**
     * 上一个被渲染的根显示对象
     */
    _lastObjectRendered: DisplayObject;

    constructor() {
        super();

        this.transparent = true;

        this._backgroundColor = 0x000000;

        this._backgroundColorRgba = [0, 0, 0, 0];

        this._backgroundColorString = '#000000';

        this.backgroundColor = this._backgroundColor; // run bg color setter

        this._tempDisplayObjectParent = new Container();

        this._lastObjectRendered = this._tempDisplayObjectParent;
    }

    /**
     * 尺寸重置，暂时根据屏幕分辨率
     * @param {number} screenWidth
     * @param {number} screenHeight
     */
    resize(width: number, height: number) {
        this.width = width;
        this.height = height;
        //下面的不适用了
        // this.htmlElement.width = width * devicePixelRatio;
        // this.htmlElement.height = height * devicePixelRatio;
        // this.htmlElement.style.width = `${width}px`;
        // this.htmlElement.style.height = `${height}px`;
    }

    /**
     * 核心渲染方法，子级重写
     */
    render(displayObject, renderTexture?, transform?) {

    }

    /**
     * 销毁方法
     */
    destroy() {
        this.type = RENDERER_TYPE.UNKNOWN;

        this.transparent = false;

        this._backgroundColor = 0;
        this._backgroundColorRgba = null;
        this._backgroundColorString = null;

        this._tempDisplayObjectParent = null;
        this._lastObjectRendered = null;
    }

    /**
     * 背景色，十六进制颜色
     * @member {number}
     */
    get backgroundColor(): number {
        return this._backgroundColor;
    }

    /**
     * 背景色，十六进制颜色
     */
    set backgroundColor(value: number) {
        if (this._backgroundColor === value) return
        this._backgroundColor = value;
        this._backgroundColorString = hex2string(value);
        hex2rgb(value, this._backgroundColorRgba);
    }
}
