export enum EnvType {
    /**
     * 淘宝小程序，不能头部adapter是因为引擎包并不打包在项目代码里，所以头部都需要，TODO，后续考虑单独为淘宝小程序打引擎包（加入adapter），然后引擎内部就干净了，load里也要肃清
     */
    tb = "tb",
    /**
     * web环境
     */
    web = "web",
    // /**
    //  * 字节helium环境，暂时废弃
    //  * 打包的时候，
    //  * 1、引擎一并打入，并且头部添加helium-adapter，基本和Web保持一致
    //  * 2、客户端先loadScript(helium-adapter.js)，再loadScript(引擎)，最后loadScript(项目代码)
    //  */
    // hel = "hel",
    /**
     * 头条，抖音小游戏
     * 需要先执行ttAdapter.js注入兼容的全局变量
     */
    tt = "tt",
    /**
     * 微信小游戏
     * 需要先执行wxAdapter.js注入兼容的全局变量
     */
    wx = "wx",
}

let env: EnvType = EnvType.web;
//如果是浏览器环境，声明个my为null，为了无声明不报错
if (window) window["my"] = null;//在用webview的小程序环境内，只要在小程序的sdk前加js就无所谓，
export function getEnv(): EnvType {
    return env
}
/**
 * 设置运行环境，一般不需要调用，预留
 * 方法initedByTbCanvas会自行设置env为tb
 * @param e 环境
 */
export function setEnv(e: EnvType) {
    env = e;
}
/**
 * 创建一个离屏的canvas
 */
export function createCanvas(): HTMLCanvasElement {
    //@ts-ignore  先这么改把，以后再改TODO，Texture.WHITE有个自执行，所以在setEnv前就会执行web的链路，以后考虑兼容document
    // return document && document.createElement("canvas") || createTbOffscreenCanvas()//my._createOffscreenCanvas();
    //淘宝小程序环境
    if (getEnv() == "tb") return createTbOffscreenCanvas()//my._createOffscreenCanvas();
    //web环境或其他环境，当作兼容过document的
    /*if (getEnv() == "web" || getEnv() == "hel" || getEnv() == "tt")*/ return document.createElement("canvas");
}
function createTbOffscreenCanvas() {
    //@ts-ignore
    var tbMy = my;
    if (!tbMy) return null;
    //不带_的先试试
    if (tbMy.createOffscreenCanvas) {
        //先试试不加参数是否有返回，再用加参数的，淘宝小部件2.0的好像必须加参数，否则返回空的
        return tbMy.createOffscreenCanvas() || tbMy.createOffscreenCanvas(3, 3);
    }
    //再用带_的
    return tbMy._createOffscreenCanvas() || tbMy._createOffscreenCanvas(3, 3);
}
/**
 * 临时记录的淘宝小程序的主canvas
 */
let tbCanvas;
/**
 * 淘宝小程序项目，拿到canvas先执行这个,修改当前环境
 * @param canvas 
 */
export function initedByTbCanvas(canvas) {
    tbCanvas = canvas;
    setEnv(EnvType.tb);
}
//兼容老版本，TODO废弃
export function initedByCanvas(canvas) {
    console.warn("Function initedByCanvas will be abandoned soon,use function initedByTbCanvas instead")
    initedByTbCanvas(canvas)
}
/**
 * 销毁记录的tbCanvas，一般也没必要执行，
 * 尤其多页面的淘宝小程序，销毁的话，createImage会有问题
 */
export function destroyTbCanvas() {
    tbCanvas = null;
}
//兼容老版本，TODO废弃
export function destroyCanvasContent() {
    console.warn("Function destroyCanvasContent will be abandoned soon,use function destroyTbCanvas instead")
    destroyTbCanvas();
}

/**
 * 返回图片
 * @returns 
 */
export function createImage(): HTMLImageElement {
    if (getEnv() == "tb") {
        if (tbCanvas) return tbCanvas.createImage();
        console.warn("TbMini inited canvas does not exist");
    }
    //当作兼容过Image的
    else {
        return new Image();
    }
}
/**
 * 尽量外部自行调用循环，不用引擎给的，即将废弃
 * @param callback 
 * @returns 
 */
export function requestAnimationFrame(callback) {
    //淘宝小程序环境
    if (getEnv() == "tb") {
        if (tbCanvas) return tbCanvas.requestAnimationFrame(callback);
        console.warn("TbMini inited canvas does not exist");
    }
    //其他环境，当作兼容过的
    else {
        return requestAnimationFrame(callback);
    }
}
export function cancelAnimationFrame(id: number) {
    //淘宝小程序环境
    if (getEnv() == "tb") {
        if (tbCanvas) {
            tbCanvas.cancelAnimationFrame(id);
            return;
        }
        console.warn("TbMini inited canvas does not exist");
    }
    //其他环境，当作兼容过的
    else {
        cancelAnimationFrame(id);
    }

}

let webglSupported: boolean;
/**
 * 判断是否支持webgl
 * @function isWebGLSupported
 * @return {boolean}
 */
export function isWebGLSupported(): boolean {
    //淘宝环境直接返回true，否则找淘宝小程序解决，tt环境待测试，TODO
    if (getEnv() == "tb" || getEnv() == "tt" /*|| getEnv() == "hel"*/ || getEnv() == "wx") return true;
    //已经判断过了，直接返回
    if (webglSupported !== undefined) return webglSupported;
    //进入判断，加上低性能下是否使用webgl标识，pc上因为failIfMajorPerformanceCaveat设为true获取不到webgl一般是驱动问题
    const contextOptions = { stencil: true, failIfMajorPerformanceCaveat: true };
    try {
        if (!window["WebGLRenderingContext"]) {
            return false;
        }
        const canvas = createCanvas()// document.createElement('canvas');
        let gl = canvas.getContext('webgl', contextOptions) || canvas.getContext('experimental-webgl', contextOptions);
        const success = !!(gl && gl["getContextAttributes"]().stencil);
        if (gl) {
            const loseContext = gl["getExtension"]('WEBGL_lose_context');
            if (loseContext) {
                loseContext.loseContext();
            }
        }
        gl = null;
        return webglSupported = success;
    }
    catch (e) {
        return webglSupported = false;
    }
}
//缓存的操作系统类型
let osType: "ios" | "android" | "pc";
/**
 * 获取操作系统类型，"ios" | "android" | "pc";
 * @returns 
 */
export function getOsType(): "ios" | "android" | "pc" {
    //有就直接返回
    if (osType) return osType;
    //淘宝小程序
    if (getEnv() == "tb") {
        //@ts-ignore
        osType = my.getSystemInfoSync().platform.toLowerCase();
    }
    //如果是hel环境，直接取window上的
    // else if (getEnv() == "hel") {
    //     osType = window && window["osType"];
    // }
    else /*if (getEnv() == "web")*/ {
        osType = navigator && navigator.userAgent && (function () {
            let n = navigator.userAgent.toLocaleLowerCase();
            let reg1 = /android/;
            let reg2 = /iphone|ipod|ipad/;
            if (reg1.test(n)) {
                return "android";
            } else if (reg2.test(n)) {
                return "ios"
            } else {
                return "pc";
            }
        })() || null;
        //@ts-ignore 如果没取到，有可能环境还没切换，直接用老方法获取，或者直接给个提示
        if (!osType && my) osType = my.getSystemInfoSync().platform.toLowerCase();
    }
    return osType;
}

let devicePixelRatio: number;
/**
 * 获取屏幕像素比dpr，内部没有用到
 * @returns 
 */
export function getDevicePixelRatio() {
    if (devicePixelRatio) return devicePixelRatio;
    //淘宝小程序
    if (getEnv() == "tb") {
        //@ts-ignore
        devicePixelRatio = my.getSystemInfoSync().pixelRatio;
    }
    //其他正常判断吧
    else /*if (getEnv() == "web" || getEnv() == "hel")*/ {
        devicePixelRatio = window && (window.devicePixelRatio || 1);
        //@ts-ignore如果还没取到，环境切换有问题，用老方法
        if (!devicePixelRatio && my) devicePixelRatio = my.getSystemInfoSync().pixelRatio;
    }
    return devicePixelRatio;
}
