import ObjectRenderer from "../../2d/renderers/webgl/ObjectRenderer";
import { WebglRenderer } from "../../2d/renderers/WebglRenderer";
import { Camera } from "../cameras/Camera";
import { ILightsConfig, ShadowType } from "./Scene3D";
import { Mesh3D } from "./Mesh3D";
import { hex2rgb, rgb2hex, Dict } from "../../2d/utils";
import { Matrix4 } from "../math/Matrix4";
import { BLEND_MODES, SCALE_MODES } from "../../2d/const";
import { getCusShader } from "../shaders/getCusShader";
import { SkinnedMesh3D } from "../bones/SkinnedMesh3D";
import { BaseTexture, Texture, RenderTexture } from "../../2d/texture";
import { Vector3 } from "../math/Vector3";
import { getShadowShader } from "../shaders/ShadowShader";
import TextureMatrix from "../../2d/texture/TextureMatrix";
import { IMaterial, IMaterialParams, RenderSideType } from "../materials/IMaterial";
import { Rectangle } from "../../2d/math";
import { BaseRenderTexture } from "../../2d/texture/BaseRenderTexture";
import { BUFFER_TYPE } from "../../2d/const"
import { DataAttribute } from "../../2d/renderers/webgl/DataAttribute";
import { Shader } from "../../2d/renderers/webgl/Shader";
import { ShaderMaterial, BaseMaterial } from "../materials";
import { IWebglShader } from "../../2d/renderers/managers/ShaderManager";
import { syncUniforms } from "../../2d/renderers/webgl/shader/syncUniforms";

let tempVec3 = new Vector3()
let tempMatrix4 = new Matrix4();
let shadowMap: RenderTexture;
//需要管理渲染的状态 resetToDefault  resetTo3D
export class D3Renderer extends ObjectRenderer {
    /**
     * 赋值的相机，需要里面的worldMatrixInverse和projectionMatrix
     */
    camera: Camera;
    /**
     * 灯光数据，用来初始化着色器和着色器传值
     */
    lightsConfig: ILightsConfig;
    /**
     * 雾化参数
     */
    fog: { color: Float32Array, near: number, far: number /*density: number*/ };
    /**
     * 视窗的方法
     */
    viewport: { x: number, y: number, width: number, height: number };
    /**
     * 阴影类型
     */
    shadowType: ShadowType;
    /**
     * 顶点着色器可传的最大uniform通道
     */
    maxVertexUniforms: number
    floatVertexTextures: boolean;
    private curLightkey: string;
    private maxPrecision: string
    private indexUintExt: any;
    constructor(renderer: WebglRenderer) {
        super(renderer);
    }
    onContextChange() {
        var gl = this.renderer.gl;
        //索引的扩展，置空
        this.indexUintExt = null;
        this.maxVertexUniforms = gl.getParameter(gl.MAX_VERTEX_UNIFORM_VECTORS);
        var maxVertexTextures = gl.getParameter(gl.MAX_VERTEX_TEXTURE_IMAGE_UNITS);
        var vertexTextures = maxVertexTextures > 0;
        var floatFragmentTextures = !!gl.getExtension("OES_texture_float");
        //传入的数据纹理暂时有问题，是UNPACK_ALIGNMENT还是UNPACK_FLIP_Y_WEBGL的问题未知，所以不管先
        this.floatVertexTextures = vertexTextures && floatFragmentTextures;
        //精度计算
        this.maxPrecision = (() => {
            if (gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT).precision > 0 &&
                gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT).precision > 0) {
                return 'highp';
            }
            if (gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT).precision > 0 &&
                gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT).precision > 0) {
                return 'mediump';
            }
            return 'lowp';
        })();
    }
    private storeViewport: Rectangle
    start() {
        this.storeViewport = this.renderer.framebufferManager.viewport;
        //设置3d状态机属性
        // this.renderer.state.resetTo3D();
        //自行设置状态机，原先的有太多问题
        //允许深度检测
        // this.renderer.state.setDepthTest(1);
        //混色模式重新设置
        // this.renderer.state.setBlendMode(BLEND_MODES.NORMAL_NPM);//让着色器里预乘
        //设置逆时针为正面，下面根据材质自行设置
        //this.renderer.state.setFrontFace(0);
        //设置剔除，下面根据材质自行设置
        //this.setCullFace(1);
    }
    stop() {
        this.flush();
        const { framebufferManager, state } = this.renderer;
        //viewport修改回去
        const { x, y, width, height } = this.storeViewport;
        framebufferManager.setViewport(x, y, width, height)
        this.storeViewport = null;
        //重置2d的默认状态
        state.setState(this.renderer.state.defaultState)
    }
    /**
     * 根据是否透明，前后穿插
     */
    private meshes: Mesh3D[] = [];
    //继续需要的数据
    render(obj: Mesh3D) {
        if (!obj.geometry || !obj.material) return;
        var mat = obj.material;
        for (var i = 0; i < this.meshes.length; i++) {
            if (this.meshes[i].material == mat) {//没啥软用
                this.meshes.splice(i, 0, obj)
                return
            }
        }
        //有透明的，从后面加
        if (mat.transparent) {
            this.meshes.push(obj);
        } else {
            this.meshes.unshift(obj);
        }
    }
    flush() {
        //生成一张阴影的深度贴图
        var shadow = this.getShadowMap();
        //同一材质的模型一起处理，同一个着色器，webglRenderer里会判断，不会重复绑定
        var curShader: Shader = null, glShader: IWebglShader;
        var pointLightNum = this.lightsConfig.pointLights.length;
        var dirLightNum = this.lightsConfig.directionalLights.length;
        const { gl, textureManager, shaderManager, bufferManager, vaoManager } = this.renderer;
        const { worldMatrixInverse, projectionMatrix } = this.camera;
        for (var i = 0; i < this.meshes.length; i++) {
            let mesh = this.meshes[i];
            const { geometry, material, _worldMatrix } = mesh;
            //自定义着色器用自行处理uniform;
            if ((material as ShaderMaterial).isShaderMaterial) {
                const { shader, uniformsData } = (material as ShaderMaterial);
                const { uniforms } = glShader = shaderManager.bind(curShader, false);
                if (curShader !== shader) {
                    curShader = shader;
                    //这两个默认得加上
                    if (uniforms["uViewMatrix"])
                        uniforms["uViewMatrix"] = worldMatrixInverse.toArray();
                    if (uniforms["uProjectionMatrix"])
                        uniforms["uProjectionMatrix"] = projectionMatrix.toArray();
                }
                //模型矩阵也默认加上
                if (uniforms["uModelMatrix"])
                    uniforms["uModelMatrix"] = _worldMatrix.toArray();
                //处理所有自定义属性
                syncUniforms(glShader, uniformsData, this.renderer);
            } else {
                let maxBones = this.getMaxBones(mesh);
                let shader = getCusShader(this.maxPrecision, material as BaseMaterial, this.lightsConfig, mesh, maxBones, this.floatVertexTextures, this.fog, shadow);
                const { uniforms } = glShader = shaderManager.bind(curShader, false);
                if (curShader !== shader) {//同一着色器只需要传一次相机参数和光照参数（因为都是场景全局）
                    curShader = shader;
                    //相机
                    uniforms["uViewMatrix"] = worldMatrixInverse.toArray();
                    uniforms["uProjectionMatrix"] = projectionMatrix.toArray();
                    //光照
                    if ((material as BaseMaterial).useLight) {
                        //只能给当前绑定的shader传值，否则location不存在
                        for (var j = 0; j < pointLightNum; j++) {
                            let key = "pointLights[" + j + "]", pl = this.lightsConfig.pointLights[j];
                            uniforms[key].color = pl.color;
                            uniforms[key].position = pl.position;
                            uniforms[key].distance = pl.distance;
                            uniforms[key].decay = pl.decay;
                        }
                        for (var j = 0; j < dirLightNum; j++) {
                            let key = "directionalLights[" + j + "]", dl = this.lightsConfig.directionalLights[j];
                            uniforms[key].color = dl.color;
                            uniforms[key].direction = dl.direction;
                            uniforms[key].shadow = +dl.castShadow;
                            uniforms[key].shadowBias = dl.bias;
                            uniforms[key].shadowRadius = dl.radius;
                            uniforms[key].shadowMapSize = dl.mapSize;
                        }
                        uniforms.uAmbientLightColor = this.lightsConfig.ambientLightColor;
                        //暂时这么处理，只会一个灯光有影子
                        if (uniforms["uDirectionalShadowMatrix"]) {
                            uniforms["uDirectionalShadowMatrix"] = shadow.lightSpaceMatrix;
                            uniforms["uDirectionalShadowMap"] = textureManager.bind(shadow.texture);
                        }
                    }
                }
                let mat = material as BaseMaterial;
                //是否带贴图，贴图考虑下图集，也就是texture的uv是否是01
                if (mat.map && mat.map.valid) {
                    let map = mat.map;
                    //找纹理，绑定纹理，多个纹理也是一样的方法
                    uniforms["uMap"] = textureManager.bind(map);
                    //没加的加一个
                    if (!map.transform) map.transform = new TextureMatrix(map);
                    //更新下
                    map.transform.update();
                    //赋值偏移
                    uniforms["uUvTransform"] = map.transform.mapCoord.toArray(true);
                    //对于透明像素有裁切的
                    const { x0, y0, x2, y2 } = map._uvs;
                    uniforms["uFrameUvs"] = [x0, y0, x2, y2];
                }
                //是否带环境贴图
                if (mat.envMap && mat.envMap.valid) {//环境贴图先不考虑图集的情况，一般一张大jpg
                    //找纹理，绑定纹理，多个纹理也是一样的方法
                    uniforms["uEnvMap"] = textureManager.bind(mat.envMap);;
                    //反射率
                    uniforms["uReflectivity"] = mat.reflectivity;
                    //传相机位置
                    uniforms["uCameraPosition"] = tempVec3.setFromMatrixPosition(this.camera._worldMatrix).toArray();
                }
                //uniform参数修改，好像目前就几个，且固定的，以后得仿three建数组
                uniforms["uMatColor"] = mat.colorArr;
                uniforms["uMatAlpha"] = mat.alpha;
                //带雾化的
                if (mat.useFog && this.fog) {
                    uniforms["uFogColor"] = this.fog.color;
                    // uniforms["uFogDensity"] = this.fog.density;
                    uniforms["uFogNear"] = this.fog.near;
                    uniforms["uFogFar"] = this.fog.far;
                }
                //mesh的模型矩阵
                uniforms["uModelMatrix"] = _worldMatrix.toArray();
                //法线矩阵
                if (uniforms.uNormalMatrix) {
                    var modelViewMatrix = tempMatrix4.multiplyMatrices(this.camera.worldMatrixInverse, mesh._worldMatrix)
                    var normalMatrix = modelViewMatrix.invert().transpose();
                    // var normalMatrix = mesh._worldMatrix.invert().transpose();
                    uniforms["uNormalMatrix"] = normalMatrix.toArray()
                    // this.setShaderUniform(curShader, "uNormalMatrix", normalMatrix.toArray());
                }
                //骨骼需要的uniform
                if (mat.skinning && maxBones > 0 && mesh.instanceType == "SkinnedMesh3D") {
                    let sMesh = mesh as SkinnedMesh3D;
                    uniforms["uBindMatrix"] = sMesh.bindMatrix.toArray();
                    uniforms["uBindMatrixInverse"] = sMesh.bindMatrixInverse.toArray();
                    // sMesh.skeleton.update()
                    var skeleton = sMesh.skeleton;
                    if (skeleton) {
                        var bones = skeleton.bones;
                        if (this.floatVertexTextures) {
                            if (!skeleton.boneTexture) {
                                // layout (1 matrix = 4 pixels)
                                //      RGBA RGBA RGBA RGBA (=> column1, column2, column3, column4)
                                //  with  8x8  pixel texture max   16 bones * 4 pixels =  (8 * 8)
                                //       16x16 pixel texture max   64 bones * 4 pixels = (16 * 16)
                                //       32x32 pixel texture max  256 bones * 4 pixels = (32 * 32)
                                //       64x64 pixel texture max 1024 bones * 4 pixels = (64 * 64)
                                var size = Math.sqrt(bones.length * 4); // 4 pixels needed for 1 matrix
                                size = (function (value) {
                                    return Math.pow(2, Math.ceil(Math.log(value) / Math.LN2));
                                })(size);
                                size = Math.max(size, 4);
                                var boneMatrices = new Float32Array(size * size * 4); // 4 floats per RGBA pixel
                                boneMatrices.set(skeleton.boneMatrices); // copy current values
                                var boneTexture = new BaseTexture({ _data: boneMatrices, width: size, height: size });
                                boneTexture.mipmap = false;
                                boneTexture.scaleMode = SCALE_MODES.NEAREST;
                                boneTexture.premultipliedAlpha = false;
                                skeleton.boneMatrices = boneMatrices;
                                skeleton.boneTexture = boneTexture;
                                skeleton.boneTextureSize = size;
                            }
                            let location: number = textureManager.bind(skeleton.boneTexture);
                            uniforms["uBoneTexture"] = location;
                            uniforms["uBoneTextureSize"] = skeleton.boneTextureSize;
                        } else {
                            uniforms["uBoneMatrices"] = sMesh.skeleton.boneMatrices;
                        }
                    }
                }
            }
            //处理变形数据，确定哪些需要被更新
            updateMorphtargets(mesh, glShader);
            //几何属性，里面也包括了aSkinWeight和aSkinIndex
            const { index, _attributes } = geometry
            for (const name in _attributes) {
                bufferManager.update(_attributes[name], BUFFER_TYPE.ARRAY_BUFFER);
            }
            //索引
            if (index) bufferManager.update(index, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER);
            //vao绑定，为了attr通道指向
            vaoManager.bind(geometry, curShader);
            //部分状态机设置
            this.setStateByMaterial(material);
            //可以画了
            //绘制
            let drawType = this.getDrawType(material);
            if (index) {
                gl.drawElements(drawType, index.count, this.getIndexGlType(index), 0);
            } else {
                gl.drawArrays(drawType, 0, geometry.getAttribute("aPosition").count);
            }
        }
        this.meshes.length = 0
    }

    private setStateByMaterial(material: IMaterial) {
        const { state } = this.renderer;
        if (material.side == RenderSideType.DoubleSide) {
            state.setCullFace(false)//不开启剔除
        } else {
            state.setCullFace(true)
            //设置剔除面
            state.setFrontFace(!!material.side)
        }
        //深度信息
        state.setDepthTest(material.depthTest);
        state.setDepthWrite(material.depthWrite);
        //混色TODO
    }

    private getMaxBones(object) {
        if (object.instanceType !== "SkinnedMesh3D") return 0;
        if (this.floatVertexTextures) return 1024;
        var skeleton = object.skeleton;
        var bones = skeleton.bones;
        var nVertexMatrices = Math.floor((this.maxVertexUniforms - 20) / 4);
        var maxBones = Math.min(nVertexMatrices, bones.length);
        if (maxBones < bones.length) {
            console.warn('Skeleton has ' + bones.length + ' bones. This GPU supports ' + maxBones + '.');
            return 0;
        }
        return maxBones;
    }

    private getShadowMap() {
        const { lightsConfig, meshes, renderer } = this;
        //找直线光，就一个，设置了castShadow的
        var dirLight = (() => {
            if (!lightsConfig || !lightsConfig.directionalLights || !lightsConfig.directionalLights.length) return null
            for (var d of lightsConfig.directionalLights) {
                if (d.castShadow) return d;
            }
            return null
        })();
        if (!dirLight) return null;
        var castMeshes: Mesh3D[] = [], receiveMeshes: Mesh3D[] = [];
        if (meshes && meshes.length) {
            for (let m of meshes) {
                //找设置了投影的物体，且是有光感材质的
                if (m.castShadow && (m.material as BaseMaterial).useLight) castMeshes.push(m);
                //找设置了接收影子的物体
                if (m.receiveShadow && (m.material as BaseMaterial).useLight) receiveMeshes.push(m);
            }
        }
        if (!castMeshes.length || !receiveMeshes.length) return null;
        const { gl, framebufferManager, shaderManager, vaoManager, bufferManager } = renderer;
        //建一个帧缓存
        shadowMap = shadowMap || RenderTexture.create(dirLight.mapSize[0], dirLight.mapSize[1]);

        //然后就可以根据直线光生成光照矩阵的深度贴图了

        const { currentRenderTexture, transform } = framebufferManager;

        //绑定帧缓存
        framebufferManager.bindRenderTexture(shadowMap);
        //清除颜色
        framebufferManager.clear([1, 1, 1, 1]);
        //设置不使用混色
        renderer.state.setBlend(false);
        //获取指定的着色器
        let shader = getShadowShader()
        //绑定着色器
        let shadowShader = shaderManager.bind(shader, false);
        //赋值uniform
        var uniforms = shadowShader.uniforms;

        // var position = dirLight.position, tPosition = dirLight.tPosition;
        // //计算光照空间矩阵
        // shadowCamera = shadowCamera || new OrthographicCamera(- 15, 15, 15, - 15, 1, 50);
        // shadowCamera.position.set(position[0], position[1], position[2]);
        // shadowCamera.lookAt(tPosition[0], tPosition[1], tPosition[2]);
        // shadowCamera.updateWorldMatrix();
        // shadowMatrix = shadowMatrix || new Matrix4;
        // shadowMatrix.set(
        //     0.5, 0.0, 0.0, 0.5,
        //     0.0, 0.5, 0.0, 0.5,
        //     0.0, 0.0, 0.5, 0.5,
        //     0.0, 0.0, 0.0, 1.0
        // );
        // shadowMatrix.multiply(shadowCamera.projectionMatrix);
        // shadowMatrix.multiply(shadowCamera.worldMatrixInverse);
        // uniforms["uLightSpaceMatrix"] = new Matrix4().multiply(shadowCamera.projectionMatrix).multiply(shadowCamera.worldMatrixInverse).toArray();

        //全在scene3d里计算了
        uniforms["uLightSpaceMatrix"] = dirLight.shadowMatrixDepth//dirLight.shadowMatrix;
        //设置剔除
        renderer.state.setCullFace(true)
        //设置剔除面
        renderer.state.setFrontFace(true)

        castMeshes.forEach((m) => {
            const { geometry, material } = m;
            uniforms["uModelMatrix"] = m._worldMatrix.toArray();
            // uniforms["uMatAlpha"] = mat.alpha;
            const { index } = geometry;
            const position = geometry.getAttribute("aPosition");
            //暂时只处理位置数据
            bufferManager.update(position, BUFFER_TYPE.ARRAY_BUFFER);
            if (index) bufferManager.update(index, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER);
            //vao处理
            vaoManager.bind(geometry, shader);
            //绘制
            let drawType = this.getDrawType(material);
            if (index) {
                gl.drawElements(drawType, index.count, this.getIndexGlType(index), 0);
            } else {
                gl.drawArrays(drawType, 0, position.count);
            }
        })

        //纹理绑定回去
        framebufferManager.bindRenderTexture(currentRenderTexture, null, null, transform);
        //混色设置回去
        renderer.state.setBlend(true);
        // renderer.state.setBlendMode(BLEND_MODES.NORMAL_NPM)//先不设置，让着色器里预乘

        //绑定过帧缓存，viewport肯定被改了,该回去
        framebufferManager.setViewport(
            this.viewport.x,
            this.viewport.y,
            this.viewport.width,
            this.viewport.height
        );
        return { texture: shadowMap, lightSpaceMatrix: dirLight.shadowMatrix, shadowType: this.shadowType }
    }
    private getDrawType(mat: IMaterial) {
        var gl = this.renderer.gl;
        if (mat["isPointsMaterial"]) return gl.POINTS;
        if (mat.wireframe) return gl.LINES;
        return gl.TRIANGLES;
    }
    private getIndexGlType(index: DataAttribute) {
        var gl = this.renderer.gl;
        if (index.array instanceof Uint32Array) {
            if (!this.indexUintExt) this.indexUintExt = gl.getExtension("OES_element_index_uint");
            return gl.UNSIGNED_INT
        }
        return gl.UNSIGNED_SHORT
    }
}

//每个几何的权重缓存
const influencesList: Dict<[number, number][]> = {};
//临时用的
let workInfluences: [number, number][];
//给uniform传值用
const morphInfluences = new Float32Array(8);

function updateMorphtargets(
    mesh: Mesh3D,
    glShader: IWebglShader,
) {
    if (!workInfluences) {
        workInfluences = [];
        for (let i = 0; i < 8; i++) {
            workInfluences[i] = [i, 0];
        }
    }
    const { morphTargetInfluences, geometry } = mesh;
    const length = morphTargetInfluences ? 0 : morphTargetInfluences.length;
    //原来几何上的权重，第一个元素标识数据索引，第二个元素标识值，第三个标识现在用的通道
    var influences: [number, number][] = influencesList[geometry.instanceId];
    //如果原先不存在
    if (!influences || influences.length !== length) {
        influences = influences || [];
        for (var i = 0; i < length; i++) {
            influences[i] = [i, 0];
        }
        influencesList[geometry.instanceId] = influences;
    }
    //mesh上的morphTargetInfluences数据写入，一般由时间轴上变更
    for (let i = 0; i < length; i++) {
        const influence = influences[i];
        influence[0] = i;
        influence[1] = morphTargetInfluences[i];
    }
    //权重大的排前面
    influences.sort((a, b) => Math.abs(b[1]) - Math.abs(a[1]));
    //实际使用的workInfluences
    for (let i = 0; i < 8; i++) {
        if (i < length && influences[i][1]) {
            workInfluences[i][0] = influences[i][0];
            workInfluences[i][1] = influences[i][1];
        } else {
            workInfluences[i][0] = Number.MAX_SAFE_INTEGER;
            workInfluences[i][1] = 0;
        }
    }
    //按照索引，小的往前排
    workInfluences.sort((a, b) => a[0] - b[0]);

    const morphTargets = geometry._morphAttributes.position;
    const morphNormals = geometry._morphAttributes.normal;

    let morphInfluencesSum = 0;
    var attrLen = morphNormals ? 4 : 8;
    for (let i = 0; i < 8; i++) {
        const [index, value] = workInfluences[i];
        if (index !== Number.MAX_SAFE_INTEGER && value && i < attrLen) {
            if (morphTargets && geometry.getAttribute('morphTarget' + i) !== morphTargets[index]) {
                geometry.addAttribute('morphTarget' + i, morphTargets[index]);
            }
            if (morphNormals && geometry.getAttribute('morphNormal' + i) !== morphNormals[index]) {
                geometry.addAttribute('morphNormal' + i, morphNormals[index]);
            }
            morphInfluences[i] = value;
            morphInfluencesSum += value;
        } else {
            if (morphTargets && geometry.getAttribute('morphTarget' + i)) {
                geometry.removeAttribute('morphTarget' + i);
            }
            if (morphNormals && geometry.getAttribute('morphNormal' + i)) {
                geometry.removeAttribute('morphNormal' + i);
            }
            morphInfluences[i] = 0;
        }
    }
    //这里感觉有点问题，待测试
    const morphBaseInfluence = geometry.morphTargetsRelative ? 1 : 1 - morphInfluencesSum;
    //属性搞了
    glShader.uniforms["morphTargetBaseInfluence"] = morphBaseInfluence;
    glShader.uniforms["morphTargetInfluences"] = morphInfluences;
}



WebglRenderer.registerPlugin('d3', D3Renderer);






//帧缓存处理
//着色器绑定，uniforms同步
//buffer更新或生成
//vao顶点属性指定通道
