import { BaseTexture } from "./BaseTexture";
import { Rectangle } from "../math";
import { Texture } from "./Texture";
import { loadImage, loadJson } from "../loader";
import { Dict } from "../utils";

/**
 * 纹理在sht图集中的数据格式
 */
export interface ITextureSheetFrameData {
    x: number,  //x,y,w,h为图集上的切图位置数据
    y: number,
    w: number,
    h: number,
    ox: number,  //ox,oy为偏移数据，trim裁剪的数据，在orig上的偏移{x:0,y:0,width:200,height:200}
    oy: number,
    sw: number, //sw,sh为原始宽高
    sh: number,
    ro: boolean, //是否旋转
}
//简化后的格式
var expData = {
    "actOverTitle.png": { "x": 1, "y": 1, "w": 474, "h": 172, "ox": 0, "oy": 0, "sw": 474, "sh": 172, "ro": false },
    "hasActOverLotBtn.png": { "x": 1, "y": 175, "w": 259, "h": 85, "ox": 0, "oy": 0, "sw": 259, "sh": 85, "ro": false }
}

/**
 * 根据基础纹理和图集数据创建纹理图集，会进全局缓存
 * @param {BaseTexture} baseTexture 
 * @param {Dict<ITextureSheetFrameData>} altaData 图集数据，一般是用pack_textures生成的数据
 * @return {Dict<Texture>} 返回贴图集。不常用
 */
export function createTextureSheet(baseTexture: BaseTexture, altaData: Dict<ITextureSheetFrameData>): Dict<Texture> {
    var frames = altaData;
    var frameKeys = Object.keys(frames);
    let frameIndex = 0;
    //要返回的贴图集合
    var textures: Dict<Texture> = {};

    while (frameIndex < frameKeys.length) {
        //名字
        const i = frameKeys[frameIndex];
        //数据
        const data: ITextureSheetFrameData = frames[i];
        //切图上的数据
        let frame: Rectangle = null;
        //裁切的数据
        let trim: Rectangle = null;
        //贴图原始尺寸
        const orig = new Rectangle(
            0,
            0,
            Math.floor(data.sw),
            Math.floor(data.sh)
        );

        //如果旋转过
        if (data.ro) {
            frame = new Rectangle(
                Math.floor(data.x),
                Math.floor(data.y),
                Math.floor(data.h),
                Math.floor(data.w)
            );
        } else {
            frame = new Rectangle(
                Math.floor(data.x),
                Math.floor(data.y),
                Math.floor(data.w),
                Math.floor(data.h)
            );
        }
        //如果是被截掉过透明边界的 ,,加上如果和原始尺寸不一致也是裁切
        if (data.ox || data.oy || data.w != data.sw || data.h != data.sh) {
            //其实就是在orig上切图，偏移
            trim = new Rectangle(
                Math.floor(data.ox),
                Math.floor(data.oy),
                Math.floor(data.w),
                Math.floor(data.h)
            );
        }

        var texture = new Texture(
            baseTexture,
            frame,
            orig,
            trim,
            data.ro ? 2 : 0,
            // data.anchor
        );
        //缓存下
        Texture.addToCache(texture, i);
        textures[i] = texture;
        frameIndex++;
    }

    return textures;
}

/**
 * 加载sht图集
 * @param url 
 * @param onSuccess 
 * @param onFail 
 * @param imgUrl 
 */
export function loadSheet(
    url: string,
    onSuccess?: (textures: Dict<Texture>) => void,
    onFail?: (err: any) => void,
    imgUrl?: string
) {
    //替换后缀改成.png
    imgUrl = imgUrl || url.substring(0, url.lastIndexOf('.')) + '.png';
    //用promise的写法吧，方便
    Promise.all([
        //数据
        new Promise<Dict<ITextureSheetFrameData>>((r) => {
            loadJson(
                url,
                r,
                () => r(null)
            )
        }),
        //图片
        new Promise<BaseTexture>((r) => {
            loadImage(
                imgUrl,
                image => r(BaseTexture.fromImage(image)),
                () => r(null)
            )
        })
    ]).then((results) => {
        if (results[0] && results[1]) {
            var textures = createTextureSheet(results[1], results[0])
            onSuccess && onSuccess(textures)
        } else {
            onFail && onFail("load failed for sht:" + url)
        }
    })
}