import { BUFFER_USAGE } from "../../const";
import { HashObject } from "../../HashObject";

//作为buffer的对应数据，应该具备的属性，可以是普通的，也可以是交错的数据
export interface IDataAttribute {
    /**
     * 类型化数组
     */
    array: any,
    usage: BUFFER_USAGE,
    _updateId: number,
    count: number
}
/**
 * 记录顶点数据对象，对于交错的数据怎么搞比较好，还是考虑都分开
 */
export class DataAttribute extends HashObject implements IDataAttribute {

    public name: string;
    //绑定buffer是使用
    usage: BUFFER_USAGE = BUFFER_USAGE.STATIC_DRAW
    /**
     * 是否需要重新传数据标识，有可能多个地方使用，所以只++，别人和他保持一致
     */
    _updateId: number = 0;

    /**
     * array作为原数据要被传入glBuffer
     * @param array Int8Array,Uint8Array, Int16Array,Uint16Array,Uint32Array,Float32Array等类型化数组
     * @param itemSize 步长，比如位置是3
     * @param normalized 是否标准化 默认false
     */
    constructor(
        array: any,
        public itemSize: number,
        public normalized: boolean = false
    ) {
        super();
        this._instanceType = "DataAttribute";
        if (Array.isArray(array)) {
            throw new TypeError('Array should be a Typed Array');
        }
        this._count = array ? array.length / itemSize : 0;
        this._array = array;
    }
    private _count: number
    /**
     * 实际数据数量，比如顶点个数，array.length/itemSize
     */
    get count() {
        return this._count;
    }
    private _array: any;
    get array() {
        return this._array;
    }
    set array(array: any) {
        if (Array.isArray(array)) {
            throw new TypeError('Array should be a Typed Array');
        }
        this._count = array ? array.length / this.itemSize : 0;
        this._array = array;
        this._updateId++;
    }
    /**
     * 重写覆盖，无法重置长度，所以value长度不能超过原array
     * @param value 普通数组，或类型化数组
     * @param offset 默认0
     */
    set(value: ArrayLike<number>, offset: number = 0) {
        this.array.set(value, offset);
        return this;
    }
    /**
     * 
     * @param index 第几个点
     */
    getX(index: number) {
        return this.array[index * this.itemSize];
    }

    setX(index: number, x: number) {
        this.array[index * this.itemSize] = x;
        return this;
    }

    getY(index: number) {
        return this.array[index * this.itemSize + 1];
    }

    setY(index: number, y: number) {
        this.array[index * this.itemSize + 1] = y;
        return this;
    }

    getZ(index: number) {
        return this.array[index * this.itemSize + 2];
    }

    setZ(index: number, z: number) {
        this.array[index * this.itemSize + 2] = z;
        return this;
    }

    getW(index: number) {
        return this.array[index * this.itemSize + 3];
    }

    setW(index: number, w: number) {
        this.array[index * this.itemSize + 3] = w;
        return this;
    }

    setXY(index: number, x: number, y: number) {
        index *= this.itemSize;
        this.array[index + 0] = x;
        this.array[index + 1] = y;
        return this;
    }

    setXYZ(index: number, x: number, y: number, z: number) {
        index *= this.itemSize;
        this.array[index + 0] = x;
        this.array[index + 1] = y;
        this.array[index + 2] = z;
        return this;
    }

    setXYZW(index: number, x: number, y: number, z: number, w: number) {
        index *= this.itemSize;
        this.array[index + 0] = x;
        this.array[index + 1] = y;
        this.array[index + 2] = z;
        this.array[index + 3] = w;
        return this;
    }
    copy(source: DataAttribute) {
        this.name = source.name;
        this.itemSize = source.itemSize;//先赋值步长，count在array赋值力有计算
        this.array = new source.array.constructor(source.array);
        this.normalized = source.normalized;
        return this;
    }
    clone() {
        return new DataAttribute(null, 0).copy(this);
    }
    destroy() {
        this._array = null;
    }
}