import { HashObject } from "../../2d/HashObject";
import { Texture } from "../../2d/texture";
import { Vector3 } from "../math/Vector3";
import { hex2rgb } from "../../2d/utils";
import { IMaterial, IMaterialParams } from "./IMaterial";


/**
 * 环境贴图融合方式
 */
export enum EnvBlendType {
    ENVMAP_BLENDING_MULTIPLY = "ENVMAP_BLENDING_MULTIPLY",
    ENVMAP_BLENDING_MIX = "ENVMAP_BLENDING_MIX",
    ENVMAP_BLENDING_ADD = "ENVMAP_BLENDING_ADD"
}



/**
 * 材质基类，一些基本属性加上
 */
export class BaseMaterial extends IMaterial {

    /**
     * 十六进制 hex2rgb ,转成0到1的数组
     */
    private _color: number = 0xffffff;
    private _colorArr: Float32Array = new Float32Array([1.0, 1.0, 1.0]);
    get color() {
        return this._color;
    }
    set color(value: number) {
        if (this._color === value) return;
        this._color = value;
        var arr = hex2rgb(value)
        this._colorArr[0] = arr[0];
        this._colorArr[1] = arr[1];
        this._colorArr[2] = arr[2];
    }
    //获取
    get colorArr() {
        return this._colorArr;
    }

    /**
     * 透明度0到1
     */
    alpha: number = 1
    /**
     * 漫反射贴图，一般贴图
     */
    public map: Texture;
    /**
     * 环境贴图
     */
    public envMap: Texture;
    public combine: EnvBlendType = EnvBlendType.ENVMAP_BLENDING_MULTIPLY;//环境贴图融合方式，
    public reflectivity: number = 1;
    // public refractionRatio = 0.98;//先只管反射
    // public envMapIntensity = 1.0
    /**
     * 变形顶点
     */
    morphTargets: boolean = false;
    /**
     * 变形法线
     */
    morphNormals: boolean = false;
    /**
     * 蒙皮
     */
    skinning: boolean = false;
    /**
     * 是否使用顶点颜色
     */
    useVertexColor: boolean = false
    /**
     * 场景的雾化是否对材质有效
     */
    useFog: boolean = true;
    /**
     * 是否光照影响，到时得改下，专门一个pbr材质，不放这里
     */
    useLight: boolean = false;
    constructor(parameters?: IBaseMaterialParams) {
        super();
        this._instanceType = "BaseMaterial";

        //所有属性
        if (parameters) {
            for (var key in parameters) {
                this[key] = parameters[key];
            }
        }
    }
    copy(material: BaseMaterial) {
        super.copy(material);
        [
            "color",
            "alpha",
            "map",
            "envMap",
            "combine",
            "reflectivity",
            // refractionRatio,
            "useVertexColor",
            "morphTargets",
            "morphNormals",
            "skinning",
            "useFog",
            "useLight",
        ].forEach((p) => this[p] = material[p])
        return this;
    }
    clone() {
        return new BaseMaterial().copy(this);
    }
    destroy() {

    }
}

export interface IBaseMaterialParams extends IMaterialParams {
    color?: number,
    alpha?: number,
    map?: Texture,
    envMap?: Texture,
    combine?: EnvBlendType,
    reflectivity?: number,
    // refractionRatio?:number,
    useVertexColor?: boolean,
    morphTargets?: boolean,
    morphNormals?: boolean,
    skinning?: boolean,
    useFog?: boolean,
    useLight?: boolean,
}