import {Texture} from "../texture/Texture";
import Sprite from "../display/Sprite";
import { AnimationNode } from "./AnimationNode";
import { HashObject } from "../HashObject";
import { AnimationClip, IAnimationTrack } from "../AnimationClip";
import { clamp } from "../utils";

/**
 * 帧动画
 * 图片都以素材为中心为原点
 */
export class FrameAni extends AnimationNode {
    protected rawData: Texture[]
    /**
     * 获取当前图片帧
     */
    getCurrentFrame(): number {
        console.warn("method getCurrentFrame will be abandoned soon,use property currentFrame instead")
        return this.currentFrame//this.texturesAll.indexOf(this.showImage.texture)
    }
    /**
     * 总时间，秒计
     */
    get totalTime(): number {
        return this.rawData && this.rawData.length * (1 / this.fps) || 0;
    };
    /**
     * 总帧数
     */
    get totalFrames(): number {
        return this.rawData && this.rawData.length || 0;
    };
    /**
     * 动画显示宽度，取最大的纹理宽度
     */
    get videoWidth(): number {
        if (!this.rawData || !this.rawData.length) return 0;
        //上面已经判断过rawData的长度，所以不多加参数0
        return Math.max(...this.rawData.map(t => t.width))
    };
    /**
     * 动画显示高度，取最大的纹理高度
     */
    get videoHeight(): number {
        if (!this.rawData || !this.rawData.length) return 0;
        return Math.max(...this.rawData.map(t => t.height))
    };
    /**
     * 每秒刷新帧数，默认30;
     */
    public fps = 30;

    get frameRate() {
        console.warn("method frameRate will be abandoned soon,use property fps instead")
        return this._fps
    }
    set frameRate(value: number) {
        console.warn("method frameRate will be abandoned soon,use property fps instead")
        this._fps = value;
    }

    private showImage: Sprite;
    private frameTrack: FrameTrack;
    /**
     * 
     * @param data 所有的纹理资源数组
     */
    constructor(data: Texture[]) {
        super(data);
        this._instanceType = "FrameAni"
    }
    init(data: Texture[]) {
        if (!data || !data.length || data == this.rawData) return;
        //记录源数据
        this.rawData = data;
        //可以复用，只要一个
        if (!this.showImage) {
            this.showImage = this.addChild(new Sprite());
            this.showImage.anchorTexture.set(0.5, 0.5);
        }
        //可以复用，因为只需要一个
        if (!this.frameTrack) {
            this.frameTrack = new FrameTrack(this.showImage, data)
        } else {
            this.frameTrack.textures = data;
        }
        this.frameTrack.resetValue();
        //
        let tracks = [this.frameTrack];
        //合成所有时间轴，总时间按总帧数传，其实可能应该用this.totalFrames-1的，无所谓了，只是最后一帧停留了一帧
        if (!this.animationClip) {
            this.animationClip = new AnimationClip(tracks, this.totalFrames);
        } else {
            this.animationClip.init(tracks, this.totalFrames)
        }
        //数据更新
        this._onRawDataUpdate();
    }
    /**
     * 重置为frame，不播放，即将废弃，用gotoAndStop代替
     */
    reset(frame: number = 0) {
        console.warn("method reset will be abandoned soon,use method gotoAndStop instead")
        this.gotoAndStop(frame);
    }

    /**
     * 重置所有贴图，即将废弃，用init代替
     * @param texturesAll 
     */
    resetTexturesAll(texturesAll: Texture[]) {
        console.warn("method resetTexturesAll will be abandoned soon,use method init instead")
        this.init(texturesAll)
    }
}

class FrameTrack extends HashObject implements IAnimationTrack {
    constructor(
        private sprite: Sprite,
        public textures: Texture[],
    ) {
        super();
        this._instanceType = "FrameTrack";
    }
    /**
     * 这里用的帧数
     * @param time 帧小数
     */
    setValue(time: number) {
        //处理time
        time = Math.round(clamp(time, 0, this.textures.length - 1));
        //找对应纹理
        this.sprite.texture = this.textures[time]
    }
    resetValue() {
        this.setValue(0);
    }
    destroy() {
        this.sprite = null;
        this.textures = null;
    }
}