import { Matrix } from "../math";
import { Texture } from "./Texture";

const tempMat = new Matrix();

/**
 * Class controls uv transform and frame clamp for texture
 * Can be used in Texture "transform" field, or separately, you can use different clamp settings on the same texture.
 * If you want to add support for texture region of certain feature or filter, that's what you're looking for.
 *
 * @see Texture
 * @see mesh.Mesh
 * @see extras.TilingSprite
 * @class
 */
export default class TextureMatrix {
    _texture: Texture;
    mapCoord: Matrix;
    uClampFrame: Float32Array;
    uClampOffset: Float32Array;
    _lastTextureID: number;
    /**
     * Changes frame clamping
     * Works with TilingSprite and Mesh
     * Change to 1.5 if you texture has repeated right and bottom lines, that leads to smoother borders
     *
     * @default 0
     * @member {number}
     */
    clampOffset: number;
    /**
     * Changes frame clamping
     * Works with TilingSprite and Mesh
     * Change to -0.5 to add a pixel to the edge, recommended for transparent trimmed textures in atlas
     *
     * @default 0.5
     * @member {number}
     */
    clampMargin: number;
    /**
     *
     * @param {Texture} texture observed texture
     * @param {number} [clampMargin] Changes frame clamping, 0.5 by default. Use -0.5 for extra border.
     * @constructor
     */
    constructor(texture: Texture, clampMargin: number = 0.5) {
        this._texture = texture;

        this.mapCoord = new Matrix();

        this.uClampFrame = new Float32Array(4);

        this.uClampOffset = new Float32Array(2);

        this._lastTextureID = -1;

        this.clampOffset = 0;

        this.clampMargin = clampMargin;
    }

    /**
     * texture property
     * @member {Texture}
     */
    get texture() {
        return this._texture;
    }

    set texture(value) {
        this._texture = value;
        this._lastTextureID = -1;
    }

    /**
     * Multiplies uvs array to transform
     * @param {Float32Array} uvs mesh uvs
     * @param {Float32Array} [out=uvs] output
     * @returns {Float32Array} output
     */
    multiplyUvs(uvs, out?) {
        if (out === undefined) {
            out = uvs;
        }

        const mat = this.mapCoord;

        for (let i = 0; i < uvs.length; i += 2) {
            const x = uvs[i];
            const y = uvs[i + 1];

            out[i] = (x * mat.a) + (y * mat.c) + mat.tx;
            out[i + 1] = (x * mat.b) + (y * mat.d) + mat.ty;
        }

        return out;
    }

    /**
     * updates matrices if texture was changed
     * @param {boolean} forceUpdate if true, matrices will be updated any case
     * @returns {boolean} whether or not it was updated
     */
    update(forceUpdate?: boolean) {
        const tex = this._texture;

        if (!tex || !tex.valid) {
            return false;
        }

        if (!forceUpdate
            && this._lastTextureID === tex._updateID) {
            return false;
        }

        this._lastTextureID = tex._updateID;

        const uvs = tex._uvs;

        this.mapCoord.set(uvs.x1 - uvs.x0, uvs.y1 - uvs.y0, uvs.x3 - uvs.x0, uvs.y3 - uvs.y0, uvs.x0, uvs.y0);

        const orig = tex.orig;
        const trim = tex.trim;

        if (trim) {
            tempMat.set(orig.width / trim.width, 0, 0, orig.height / trim.height,
                -trim.x / trim.width, -trim.y / trim.height);
            this.mapCoord.append(tempMat);
        }

        const texBase = tex.baseTexture;
        const frame = this.uClampFrame;
        const margin = this.clampMargin;
        const offset = this.clampOffset;

        frame[0] = (tex.frame.x + margin + offset) / texBase.width;
        frame[1] = (tex.frame.y + margin + offset) / texBase.height;
        frame[2] = (tex.frame.x + tex.frame.width - margin + offset) / texBase.width;
        frame[3] = (tex.frame.y + tex.frame.height - margin + offset) / texBase.height;
        this.uClampOffset[0] = offset / texBase.width;
        this.uClampOffset[1] = offset / texBase.height;

        return true;
    }
}
