import { BaseTexture,IImageData } from "./BaseTexture";
import { EventDispatcher } from "../events";
import { Point, Rectangle } from "../math";
import TextureUvs from "./TextureUvs";
import TextureMatrix from "./TextureMatrix";
import { createCanvas, TextureCache } from "../utils";

/**
 * 一张图片或图集的一部分，如果没有frame。默认整张图片
 * eg
 * let texture = Texture.fromImage('assets/image.png');
 * let sprite1 = new Sprite(texture);
 * let sprite2 = new Sprite(texture);
 * 
 * @class
 * @extends EventDispatcher
 */
 export class Texture extends EventDispatcher {
    protected _baseTexture: BaseTexture;
    /**
     * BaseTexture，必有
     */
    get baseTexture() {
        return this._baseTexture;
    }
    /**
     * 标记texture没有frame
     * 表示不是从图集来的固定frame。需要根据BaseTexture的更新而改变frame
     */
    private noFrame: boolean;
    /**
     * 实际绘制矩形框，对于有trimmed的纹理很重要
     * This is the area of the BaseTexture image to actually copy to the Canvas / WebGL when rendering,
     * irrespective of the actual frame size or placement (which can be influenced by trimmed texture atlases)
     */
    protected _frame: Rectangle;
    /**
     * The frame specifies the region of the base texture that this texture uses.
     * 手动修改frame时，而不是set赋值，比如frame.x=111,frame.width=333,需要手动调用updateUvs，不推荐这种方式修改
     * @member {Rectangle}
     */
    get frame() {
        return this._frame;
    }

    set frame(frame) {
        this._frame = frame;

        this.noFrame = false;

        const { x, y, width, height } = frame;
        const xNotFit = x + width > this.baseTexture.width;
        const yNotFit = y + height > this.baseTexture.height;

        if (xNotFit || yNotFit) {
            const relationship = xNotFit && yNotFit ? 'and' : 'or';
            const errorX = `X: ${x} + ${width} = ${x + width} > ${this.baseTexture.width}`;
            const errorY = `Y: ${y} + ${height} = ${y + height} > ${this.baseTexture.height}`;

            throw new Error('Texture Error: frame does not fit inside the base Texture dimensions: '
                + `${errorX} ${relationship} ${errorY}`);
        }

        //标记是否可用
        this.valid = width && height && this.baseTexture.source && this.baseTexture.hasLoaded;

        //如果无裁切透明像素，无旋转，orig和frame完全一致
        if (!this.trim && !this.rotate) this.orig = frame;

        //如果可用，更新uv
        if (this.valid) this.updateUvs();
    }

    /**
     * 贴图旋转及镜像
     */
    private _rotate: number;
    /**
     * Indicates whether the texture is rotated inside the atlas
     * set to 2 to compensate for texture packer rotation
     * set to 6 to compensate for spine packer rotation
     * can be used to rotate or mirror sprites
     * See {@link GroupD8} for explanation
     *
     * @member {number}
     */
    get rotate(): number {
        return this._rotate;
    }
    //先去掉了，TODO
    // set rotate(rotate: number) {
    //     if (this._rotate != rotate) {
    //         this._rotate = rotate;
    //         if (this.valid) this.updateUvs();
    //     }
    // }

    /**
     * 宽高都是纹理真实的宽高，不管trim
     * @member {number}
     */
    get width() {
        return this.orig.width;
    }

    /**
     * 宽高都是纹理真实的宽高，不管trim
     * @member {number}
     */
    get height() {
        return this.orig.height;
    }
    /**
     * 如果矩形边缘有透明像素被裁减后的缩小的区域
     * This is the trimmed area of original texture, before it was put in atlas
     * Please call `updateUvs()` after you change coordinates of `trim` manually.
     */
    trim: Rectangle;
    /**
     * 原始尺寸，放入图集前
     * This is the area of original texture, before it was put in atlas
     */
    orig: Rectangle;
    /**
     * 贴图是否可用，true为可用
     */
    valid: boolean = false;

    /**
     * 对应贴图uv
     * The WebGL UV data cache.
     */
    _uvs: TextureUvs = null;

    /**
     * 贴图的锚点，默认0，0，左上角，范围0到1
     * 设置后会影响使用该纹理的sprite的绘制起始点
     * @default {0,0}
     */
    defaultAnchor: Point;
    /**
     * 更新的id标志
     */
    _updateID: number = 0;

    /**
     * 一般不用，需要时再说
     * Contains data for uvs. May contain clamp settings and some matrices.
     * Its a bit heavy, so by default that object is not created.
     * @member {TextureMatrix}
     * @default null
     */
    transform: TextureMatrix = null;

    /**
     * The ids under which this Texture has been added to the texture cache. This is
     * automatically set as long as Texture.addToCache is used, but may not be set if a
     * Texture is added directly to the TextureCache array.
     *
     * @member {string[]}
     */
    textureCacheIds: string[] = [];

    /**
     * 空纹理
     */
    static EMPTY: Texture;
    /**
     * 白贴图
     */
    private static _WHITE: Texture;
    static get WHITE(): Texture {//用get方式，用到了才创建，引擎在淘宝小部件里自执行my._createOffscreenCanvas()返回的是undefined
        if (!Texture._WHITE) {
            //白图纹理用canvas建
            const canvas = createCanvas();
            canvas.width = 16;
            canvas.height = 16;
            const context = canvas.getContext('2d');
            context.clearRect(0, 0, 16, 16);//淘宝小程序的问题，必须先调用过clearRect，否则有几率绘制无效
            context.fillStyle = 'white';//淘宝小程序待测试
            context.fillRect(0, 0, 16, 16);
            //生成纹理
            const white = new Texture(new BaseTexture(canvas));
            //置空事件方法
            removeAllHandlers(white);
            removeAllHandlers(white.baseTexture);
            //缓存赋值
            Texture._WHITE = white;
        }
        return Texture._WHITE;
    };
    /**
     * @param {BaseTexture} baseTexture - The base texture source to create the texture from
     * @param {Rectangle} [frame] - The rectangle frame of the texture to show
     * @param {Rectangle} [orig] - The area of original texture
     * @param {Rectangle} [trim] - Trimmed rectangle of original texture
     * @param {number} [rotate] - indicates how the texture was rotated by texture packer. See {@link GroupD8}
     * @param {Point} [anchor] - Default anchor point used for sprite placement / rotation
     */
    constructor(
        baseTexture: BaseTexture | Texture,
        frame?: Rectangle,
        orig?: Rectangle,
        trim?: Rectangle,
        rotate?: number,
        anchor?: Point
    ) {
        super();
        this._instanceType = "Texture";
        this.noFrame = false;
        if (!frame) {
            //标记没有固定的frame
            this.noFrame = true;
            //frame初始化个
            frame = new Rectangle(0, 0, 1, 1);
        }

        //如果传入的是Texture，取其base
        if (baseTexture instanceof Texture) baseTexture = baseTexture.baseTexture;

        this._baseTexture = baseTexture;

        this._frame = frame;

        this.trim = trim || null;

        this.orig = orig || frame;// new Rectangle(0, 0, 1, 1);

        this._rotate = Number(rotate || 0);
        //对于canvas形式的判断hasLoaded有问题，导致不能监听update，hasLoaded还用于判断纹理是否可用（canvas宽高为0不可用），所以这里判断吧
        if (baseTexture.hasLoaded || (baseTexture.source && baseTexture.source.getContext)) {
            if (this.noFrame) {
                frame = new Rectangle(0, 0, baseTexture.width, baseTexture.height);
                // if there is no frame we should monitor for any base texture changes..
                baseTexture.addEventListener('update', this.onBaseTextureUpdated, this);
            }
            this.frame = frame;
        }
        else {
            baseTexture.once('loaded', this.onBaseTextureLoaded, this);
        }

        this.defaultAnchor = anchor ? new Point(anchor.x, anchor.y) : new Point(0, 0);
    }

    /**
     * 更新方法，直接调用base的，source尺寸有改变，或内容有改变，基本用于Shape和TextField
     * 在base更新时会触发自己的
     */
    update() {
        this.baseTexture.update();
    }

    // 下面Loaded和Updated两个监听函数基本用于两种情况，不会用于图集中的纹理(除非图集还没加载好)
    // 1、canvas作为贴图需要监听尺寸变化，Shape里以及TextField里
    // 2、单张图片（如网络图片临时加载）需要监听尺寸变化，比如奖品图片，尺寸不一且未知
    /**
     * 
     * @private
     * @param {BaseTexture} baseTexture - The base texture.
     */
    private onBaseTextureLoaded(e: Event/*,baseTexture: BaseTexture*/) {
        var baseTexture = e.target as BaseTexture;
        this._updateID++;

        //执行这个函数意味着noFrame为true
        if (this.noFrame) {
            if (baseTexture.source) {//加载成功了才修改frame和监听
                this.frame = new Rectangle(0, 0, baseTexture.width, baseTexture.height);
                //如果是没传过固定的frame就要监听了
                baseTexture.addEventListener('update', this.onBaseTextureUpdated, this);
            }
        }
        else {
            this.frame = this._frame;
        }
        //加载成功的时候有必要触发update吗？
        this.dispatchEvent("update")

        this.dispatchEvent("loaded")
    }

    /**
     * 
     * @private
     * @param {BaseTexture} baseTexture - The base texture.
     */
    private onBaseTextureUpdated(e: Event/*,baseTexture*/) {
        var baseTexture = e.target as BaseTexture;
        //标志纹理已更新
        this._updateID++;
        //只有尺寸需要
        this._frame.width = baseTexture.width;
        this._frame.height = baseTexture.height;
        this.orig.width = this._frame.width;
        this.orig.height = this._frame.height;
        this.valid = baseTexture.hasLoaded;
        this.dispatchEvent("update")
    }

    /**
     * 销毁
     */
    destroy() {
        if (this.baseTexture) {
            //考虑是否销毁baseTexture，暂时不

            //事件要移除
            this.baseTexture.removeEventListener('update', this.onBaseTextureUpdated, this);
            this.baseTexture.removeEventListener('loaded', this.onBaseTextureLoaded, this);
            this._baseTexture = null;
        }
        this._frame = null;
        this._uvs = null;
        this.trim = null;
        this.orig = null;
        this.valid = false;
        Texture.removeFromCache(this);
        this.textureCacheIds = null;
    }

    /**
     * 克隆
     * @return {Texture} The new texture
     */
    clone(): Texture {
        return new Texture(
            this.baseTexture,
            this.frame,
            this.orig,
            this.trim,
            this.rotate,
            this.defaultAnchor
        );
    }

    /**
     * 更新uv，比如在frame改变或trim改变之后
     */
    updateUvs() {
        if (!this._uvs) this._uvs = new TextureUvs();
        //uvs方法的类型还没改
        this._uvs.set(this._frame, this._baseTexture, this.rotate);
        this._updateID++;
    }



    //辅助方法
    /**
     * 会缓存进全局纹理
     * @param {string} url 
     */
    static fromUrl(url: string) {
        let texture = TextureCache[url];
        if (!texture) {
            texture = new Texture(BaseTexture.fromUrl(url));
            Texture.addToCache(texture, url);
        }
        return texture;
    }

    static fromCanvas(canvas: HTMLCanvasElement, origin: string = 'canvas') {
        return new Texture(BaseTexture.fromCanvas(canvas, origin))
    }

    static fromData(data: IImageData) {
        return new Texture(BaseTexture.fromData(data))
    }

    static fromImage(image: HTMLImageElement) {
        var imageUrl = image.src;
        let texture: Texture = TextureCache[imageUrl];
        if (!texture) {
            texture = new Texture(BaseTexture.fromImage(image));
            if (imageUrl && imageUrl.indexOf('data:') !== 0) Texture.addToCache(texture, imageUrl);
        }
        return texture;
    }
    /**
     * 图片路径，canvas标签，图片标签，图片数据
     * @param anything 
     * @returns 
     */
    static from(anything: string | HTMLCanvasElement | HTMLImageElement | IImageData) {
        //路径
        if (typeof anything === 'string') {
            return Texture.fromUrl(anything);
        }
        else if ((anything as IImageData)._data) {
            return Texture.fromData(anything as IImageData);
        }
        else if ((anything as HTMLCanvasElement).getContext) {
            return Texture.fromCanvas(anything as HTMLCanvasElement);
        } else {
            return Texture.fromImage(anything as HTMLImageElement);
        }
    }

    /**
     * 加入全局纹理缓存，TextureCache[name]调用
     * @static
     * @param {Texture} texture
     * @param {string} id
     */
    static addToCache(texture: Texture, id: string) {
        if (id) {
            if (texture.textureCacheIds.indexOf(id) === -1) {
                texture.textureCacheIds.push(id);
            }
            if (TextureCache[id]) {
                //覆盖
                console.warn(`rewrite cached texture: ${id}`);
            }
            TextureCache[id] = texture;
        }
    }

    /**
     * 从全局缓存中移除
     * @static
     * @param {string|Texture} texture - 纹理的id或纹理自身
     * @return {Texture} 返回移除的纹理
     */
    static removeFromCache(texture: any): Texture {
        if (typeof texture === 'string') {
            const textureFromCache = TextureCache[texture];

            if (textureFromCache) {
                const index = textureFromCache.textureCacheIds.indexOf(texture);

                if (index > -1) {
                    textureFromCache.textureCacheIds.splice(index, 1);
                }

                delete TextureCache[texture];

                return textureFromCache;
            }
        }
        else if (texture && texture.textureCacheIds) {
            for (let i = 0; i < texture.textureCacheIds.length; ++i) {
                // Check that texture matches the one being passed in before deleting it from the cache.
                if (TextureCache[texture.textureCacheIds[i]] === texture) {
                    delete TextureCache[texture.textureCacheIds[i]];
                }
            }

            texture.textureCacheIds.length = 0;

            return texture;
        }

        return null;
    }
}

//将事件置空，空纹理或白纹理不需要响应任何加载或更新事件
function removeAllHandlers(tex) {
    tex.destroy = function _emptyDestroy() { /* empty */ };
    tex.addEventListener = function _emptyOn() { /* empty */ };
    tex.once = function _emptyOnce() { /* empty */ };
    tex.dispatchEvent = function _emptyEmit() { /* empty */ };
    tex.removeEventListener = function _emptyOff() { /* empty */ };
}

/**
 * 空纹理，用于绑定webgl绑定空纹理
 * @static
 * @constant
 */
Texture.EMPTY = new Texture(new BaseTexture());
removeAllHandlers(Texture.EMPTY);
removeAllHandlers(Texture.EMPTY.baseTexture);
