import { BaseRenderTexture } from './BaseRenderTexture';
import { Texture } from './Texture';
import { Rectangle } from '../math';
import { Framebuffer } from '../renderers/webgl/Framebuffer';

/**
 * 将显示对象画在上面的贴图，就是快照
 * 注意绘制前纹理加载完成
 * For example:
 *
 * ```js
 * let renderer = CanvasRenderer(1024, 1024);
 * let renderTexture = RenderTexture.create(800, 600);
 * let sprite = Sprite.fromImage("aaa.png");
 * 
 * sprite.x = 800/2;
 * sprite.y = 600/2;
 * sprite.anchorTexture.set(0.5,0.5);
 * //sprite.setTransform();会重置属性
 *
 * renderer.render(sprite, renderTexture);
 * ```
 *
 * @class
 * @extends Texture
 */
export class RenderTexture extends Texture {
    protected _baseTexture: BaseRenderTexture;
    get baseTexture(): BaseRenderTexture {
        return this._baseTexture
    }
    //filter的TODO/////////
    /**
         * Stores `sourceFrame` when this texture is inside current filter stack.
         *
         * You can read it inside filters.
         *
         * @readonly
         */
    public filterFrame: Rectangle | null;

    /**
     * The key for pooled texture of FilterSystem.
     *
     * @see PIXI.RenderTexturePool
     */
    public filterPoolKey: string | number | null;
    /**
     * @param {BaseRenderTexture} baseRenderTexture
     * @param {Rectangle} [frame]
     */
    constructor(baseRenderTexture: BaseRenderTexture, frame?: Rectangle) {
        super(baseRenderTexture, frame);

        this.valid = true;

        this.updateUvs();
    }
    get framebuffer(): Framebuffer {
        return (this.baseTexture as BaseRenderTexture).framebuffer;
    }
    /**
     * 重置尺寸
     * @param {number} width - 宽度
     * @param {number} height - 高度
     * @param {boolean} doNotResizeBaseTexture - 是否不重置基础纹理的尺寸，默认false，表示也重置
     */
    resize(width: number, height: number, doNotResizeBaseTexture: boolean = false) {
        width = Math.ceil(width);
        height = Math.ceil(height);
        this.valid = (width > 0 && height > 0);

        this._frame.width = this.orig.width = width;
        this._frame.height = this.orig.height = height;

        if (!doNotResizeBaseTexture) {
            this.baseTexture.resize(width, height);
        }

        this.updateUvs();
    }

    /**
     * 创建renderTexture快捷静态方法
     * @param {number} [width=100] - 宽度
     * @param {number} [height=100] - 高度
     * @return {RenderTexture}
     */
    static create(width: number, height: number): RenderTexture {
        return new RenderTexture(new BaseRenderTexture(width, height));
    }
}
