import { Framebuffer } from '../renderers/webgl/Framebuffer';
import { createCanvas } from '../utils';
import { BaseTexture } from './BaseTexture';

/**
 * 将显示对象画在上面的贴图，这是基础纹理，一般用RenderTexture
 * 注意绘制前纹理加载完成
 * For example:
 *
 * ```js
 * let renderer = CanvasRenderer(context,1024, 1024);
 * let baseRenderTexture = new BaseRenderTexture(800, 600);
 * let renderTexture = new RenderTexture(baseRenderTexture);
 * let sprite = Sprite.fromImage("spinObj_01.png");
 *
 * sprite.position.x = 800/2;
 * sprite.position.y = 600/2;
 * sprite.anchorTexture.x = 0.5;
 * sprite.anchorTexture.y = 0.5;
 *
 * renderer.render(sprite,renderTexture);
 * ```
 *
 *
 * @class
 * @extends BaseTexture
 */
export class BaseRenderTexture extends BaseTexture {
    /**
     * canvas模式下使用，可以共用
     */
    _canvasRenderTarget: CanvasRenderTarget;
    /**
     * 也是作为BaseRenderTexture的标记
     */
    public framebuffer: Framebuffer;
    maskStack: Array<any>;
    filterStack: Array<any>;
    /**
     * @param {number} [width=100] - 宽度
     * @param {number} [height=100] - 高度
     */
    constructor(width: number = 100, height: number = 100) {
        super();
        this._width = Math.ceil(width);
        this._height = Math.ceil(height);
        //这个标记下
        this.hasLoaded = true;
        this.mipmap = false;
        this.framebuffer = new Framebuffer(width, height);
        this.framebuffer.colorTexture = this;
        //这个在镇缓存中需要先bind，生成glTexture，然后给frameBuffer
    }

    /**
     * 重置尺寸
     * @param {number} width - 宽度
     * @param {number} height - 高度
     */
    resize(width: number, height: number) {
        width = Math.ceil(width);
        height = Math.ceil(height);

        if (width === this.width && height === this.height) return;
        this._setSize(width, height);
        //
        this.framebuffer.resize(width, height);
        //这里待考虑
        this.dispatchEvent('update');
    }

    /**
     * 销毁
     */
    destroy() {

        super.destroy();
    }
}

/**
 * 其实就是一个离屏canvas，webgl模式不需要建canvas，因为可以用帧缓存
 */
export class CanvasRenderTarget {
    canvas: HTMLCanvasElement;
    context: CanvasRenderingContext2D;
    constructor(width: number, height: number) {
        this.canvas = createCanvas() //document.createElement('canvas');
        // console.log("rd1",this.canvas)
        this.resize(width, height);//要先设置尺寸？
        this.context = this.canvas.getContext('2d');
        // console.log("rd",this.context)
        this.resize(width, height);//为何要多加一次
        this.clear();//淘宝canvas问题，绘制前必须调用过clearRect，否则就有几率绘制不出来
    }

    clear() {
        this.context.setTransform(1, 0, 0, 1, 0, 0);
        this.context.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }

    resize(width: number, height: number): void {
        this.canvas.width = Math.round(width);
        this.canvas.height = Math.round(height);
    }

    destroy() {
        this.context = null;
        this.canvas = null;
    }

    get width(): number {
        return this.canvas.width;
    }

    set width(val: number) {
        this.canvas.width = Math.round(val);
    }

    get height(): number {
        return this.canvas.height;
    }

    set height(val: number) {
        this.canvas.height = Math.round(val);
    }
}
