import { BLEND_MODES } from "../../const";
import { mapWebGLBlendModes } from "../../utils/mapWebGLBlendModes";

const BLEND = 0;
const DEPTH_TEST = 1;
const FRONT_FACE = 2;
const CULL_FACE = 3;
const BLEND_FUNC = 4;
const DEPTH_WRITE = 5;
/**
 * A WebGL state machines
 * 状态机管理类
 * @class
 */
export default class WebGLState {
    private blendModes
    //其实暂时只用到了5个
    private activeState: Uint8Array;

    defaultState: Uint8Array;

    constructor(private gl: WebGLRenderingContext) {
        this.activeState = new Uint8Array(16);
        this.defaultState = new Uint8Array(16);
        //默认的打开混色
        this.defaultState[0] = 1;
        this.blendModes = mapWebGLBlendModes(gl);
    }

    setState(state) {
        this.setBlend(state[BLEND]);
        this.setDepthTest(state[DEPTH_TEST]);
        this.setFrontFace(state[FRONT_FACE]);
        this.setCullFace(state[CULL_FACE]);
        this.setBlendMode(state[BLEND_FUNC]);

    }

    /**
     * 开启或关闭混色
     * @param value 
     * @returns 
     */
    setBlend(value: boolean) {
        let v = value ? 1 : 0;

        if (this.activeState[BLEND] === v) return;

        this.activeState[BLEND] = v;
        this.gl[value ? 'enable' : 'disable'](this.gl.BLEND);
    }

    setBlendMode(value: BLEND_MODES) {
        if (value === this.activeState[BLEND_FUNC]) return;

        this.activeState[BLEND_FUNC] = value;

        const mode = this.blendModes[value];

        if (mode.length === 2) {
            this.gl.blendFunc(mode[0], mode[1]);
        }
        else {
            this.gl.blendFuncSeparate(mode[0], mode[1], mode[2], mode[3]);
        }
    }

    /**
     * 开启或关闭深度检测
     * @param value 
     * @returns 
     */
    setDepthTest(value: boolean) {
        let v = value ? 1 : 0;

        if (this.activeState[DEPTH_TEST] === v) return;

        this.activeState[DEPTH_TEST] = v;
        this.gl[value ? 'enable' : 'disable'](this.gl.DEPTH_TEST);
    }
    /**
     * 是否允许写入深度信息
     * @param v 
     */
    setDepthWrite(value: boolean) {
        let v = value ? 1 : 0;
        if (this.activeState[DEPTH_WRITE] === v) return;
        this.activeState[DEPTH_WRITE] = v;
        this.gl.depthMask(!!value)
    }
    /**
     * 设置是否开启剔除功能
     * @param {boolean} value -
     */
    setCullFace(value: boolean) {
        let v = value ? 1 : 0;

        if (this.activeState[CULL_FACE] === v) return;

        this.activeState[CULL_FACE] = v;
        this.gl[value ? 'enable' : 'disable'](this.gl.CULL_FACE);
    }

    /**
     * 设置gl的正面的绘制顺序，
     * @param {boolean} value - true是顺时针，false是逆时针
     */
    setFrontFace(value: boolean) {
        let v = value ? 1 : 0;

        if (this.activeState[FRONT_FACE] === v) return;

        this.activeState[FRONT_FACE] = v;
        this.gl.frontFace(this.gl[value ? 'CW' : 'CCW']);
    }

    resetToDefault() {
        // set active state so we can force overrides of gl state
        for (let i = 0; i < this.activeState.length; ++i) {
            this.activeState[i] = 32;
        }
        // 翻转Y纹理，gl状态默认不翻转false，这个没必要放这啊
        this.gl.pixelStorei(this.gl.UNPACK_FLIP_Y_WEBGL, false);
        this.setState(this.defaultState);
    }
}
