
import { Rectangle } from "../../math";
import { Geometry } from "./Geometry";
import { InterleavedData, InterleavedDataAttribute } from "./InterleavedDataAttribute";

export class Quad extends Geometry {
    /**
     * 顶点数据
     */
    vertices: Float32Array;
    /**
     * uv数据
     */
    uvs: Float32Array;

    interleavedData: InterleavedData
    constructor() {
        super();
        this.vertices = new Float32Array([
            -1, -1,
            1, -1,
            1, 1,
            -1, 1,
        ]);
        this.uvs = new Float32Array([
            0, 0,
            1, 0,
            1, 1,
            0, 1,
        ]);

        var interleaved = new Float32Array(8 * 2);

        for (let i = 0; i < 4; i++) {
            interleaved[i * 4] = this.vertices[(i * 2)];
            interleaved[(i * 4) + 1] = this.vertices[(i * 2) + 1];
            interleaved[(i * 4) + 2] = this.uvs[i * 2];
            interleaved[(i * 4) + 3] = this.uvs[(i * 2) + 1];
        }

        var interleavedData = this.interleavedData = new InterleavedData(interleaved, 4);

        this.addAttribute("aVertexPosition", new InterleavedDataAttribute(interleavedData, 2, 0, false))
        this.addAttribute("aTextureCoord", new InterleavedDataAttribute(interleavedData, 2, 2, false))
        this.index = createIndicesForQuads(1)
    }
    /**
     * Maps two Rectangle to the quad.
     * @param {Rectangle} targetTextureFrame - the first rectangle
     * @param {Rectangle} destinationFrame - the second rectangle
     * @return {Quad} Returns itself.
     */
    map(targetTextureFrame: Rectangle, destinationFrame: Rectangle): Quad {
        let x = 0; // destinationFrame.x / targetTextureFrame.width;
        let y = 0; // destinationFrame.y / targetTextureFrame.height;

        this.uvs[0] = x;
        this.uvs[1] = y;

        this.uvs[2] = x + (destinationFrame.width / targetTextureFrame.width);
        this.uvs[3] = y;

        this.uvs[4] = x + (destinationFrame.width / targetTextureFrame.width);
        this.uvs[5] = y + (destinationFrame.height / targetTextureFrame.height);

        this.uvs[6] = x;
        this.uvs[7] = y + (destinationFrame.height / targetTextureFrame.height);

        x = destinationFrame.x;
        y = destinationFrame.y;

        this.vertices[0] = x;
        this.vertices[1] = y;

        this.vertices[2] = x + destinationFrame.width;
        this.vertices[3] = y;

        this.vertices[4] = x + destinationFrame.width;
        this.vertices[5] = y + destinationFrame.height;

        this.vertices[6] = x;
        this.vertices[7] = y + destinationFrame.height;

        //数据更新
        let array: Float32Array = this.interleavedData.array;
        for (let i = 0; i < 4; i++) {
            array[i * 4] = this.vertices[(i * 2)];
            array[(i * 4) + 1] = this.vertices[(i * 2) + 1];
            array[(i * 4) + 2] = this.uvs[i * 2];
            array[(i * 4) + 3] = this.uvs[(i * 2) + 1];
        }
        this.interleavedData._updateId++;

        return this;
    }
}


/**
 * 根据图片数量创建索引数组 ，暂时不用了
 * 0....1
 * . .  .
 * .  . .
 * 3....2   
 * @function createIndicesForQuads
 * @private
 * @param {number} size - Number of quads
 * @return {Uint16Array} indices
 */
function createIndicesForQuads(size: number): Uint16Array {

    const totalIndices = size * 6;

    const indices = new Uint16Array(totalIndices);

    for (let i = 0, j = 0; i < totalIndices; i += 6, j += 4) {
        indices[i + 0] = j + 0;
        indices[i + 1] = j + 1;
        indices[i + 2] = j + 2;
        indices[i + 3] = j + 0;
        indices[i + 4] = j + 2;
        indices[i + 5] = j + 3;
    }

    return indices;
}