import { BLEND_MODES, BUFFER_TYPE, BUFFER_USAGE, DRAW_MODES } from "../../const"
import ObjectRenderer from '../webgl/ObjectRenderer';
import { WebglRenderer } from '../WebglRenderer';
import { nextPow2, log2, premultiplyTint, getOsType, Dict, createCanvas } from '../../utils';
import { DisplayObject } from '../../display/DisplayObject';
import { Texture } from '../../texture/Texture';
import { BaseTexture } from '../../texture/BaseTexture';
import { premultiplyBlendMode } from '../../utils/mapPremultipliedBlendModes';
import { createWebglContext } from '../webgl/createWebglContext';
import { Geometry } from "../webgl/Geometry";
import { InterleavedData, InterleavedDataAttribute } from "../webgl/InterleavedDataAttribute";
import { DataAttribute } from "../webgl/DataAttribute";
import { Shader } from "../webgl/Shader";

let TICK = 0;

/**
 * 批处理核心渲染插件
 *
 * @class
 * @extends ObjectRenderer
 */
export class BatchRenderer extends ObjectRenderer {
    /**
     * 每个点
     * aVertexPosition(2), aTextureCoord(2), aColor(1), aTextureId(1) = 6
     *
     * @member {number}
     */
    vertSize: number = 6;
    vertByteSize: number = 6 * 4;
    /**
     * 顶点数量，2000张图*4
     */
    size: number = 2000 * 4;
    /**
     * 当前顶点数量
     */
    currentSize: number = 0;
    /**
     * 当前索引数量
     */
    currentIndexSize: number = 0;
    /**
     * 顶点数据
     */
    vertexArrays: Dict<Float32Array> = {};
    /**
     * 索引数据，Uint16Array足够
     */
    indexArrays: Dict<Uint16Array> = {};
    /**
     * 可以确定只需要了一个着色器
     */
    shader: Shader = null;
    /**
     * 当前累计的绘制对象数量
     */
    currentIndex: number = 0;
    groups: IBatchDrawCall[];
    /**
     * 绘制对象数组
     * 包括图片的和矢量图的
     */
    elements: IBatchElement[] = [];

    MAX_TEXTURES: number;

    geometry: Geometry;

    interleavedData: InterleavedData;

    indexData: DataAttribute;

    constructor(renderer: WebglRenderer) {
        super(renderer);
        this.groups = [];
        //最大数量健好先
        for (let k = 0; k < this.size / 4; k++) {
            this.groups[k] = {
                textures: [],
                blend: 0,
                textureCount: 0,
                start: 0,
                size: 0,
                type: DRAW_MODES.TRIANGLES,
            };
        }

        //几何建好先，一直复用，只有里面数据会修改
        this.geometry = new Geometry();
        //交错数据
        const interleavedData = this.interleavedData = new InterleavedData(
            this.getVertexArrays(8),
            this.vertSize
        );
        interleavedData.usage = BUFFER_USAGE.STREAM_DRAW
        //顶点
        this.geometry
            .addAttribute("aVertexPosition", new InterleavedDataAttribute(interleavedData, 2, 0, false))
            .addAttribute("aTextureCoord", new InterleavedDataAttribute(interleavedData, 2, 2, false))
            .addAttribute("aColor", new InterleavedDataAttribute(interleavedData, 1, 4, false))

        //索引数据
        this.indexData = this.geometry.index = new DataAttribute(this.getIndexArray(12), 1)
        this.indexData.usage = BUFFER_USAGE.STREAM_DRAW
    }

    /**
     * Sets up the renderer context and necessary buffers.
     */
    onContextChange() {
        const gl = this.renderer.gl;

        if (false /*传统方式*/) {
            this.MAX_TEXTURES = 1;
        }
        else {
            // step 1: first check max textures the GPU can handle.
            this.MAX_TEXTURES = Math.min(gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS), 16);

            // step 2: check the maximum number of if statements the shader can have too..
            this.MAX_TEXTURES = checkMaxIfStatementsInShader(this.MAX_TEXTURES, gl);
        }

        //需要按照纹理数量缓存吗？TODO
        this.shader = generateMultiTextureShader(this.MAX_TEXTURES);
        //
        const sampleValues = [];
        for (let i = 0; i < this.MAX_TEXTURES; i++) {
            sampleValues[i] = i;
        }
        //这里直接就绑定修改了，有问题再说
        this.renderer.shaderManager.bind(this.shader).uniforms["uSamplers"] = sampleValues;
        //如果不止一个纹理的
        if (this.MAX_TEXTURES > 1/*this.shader.attributes.aTextureId*/) {
            if (!this.geometry.getAttribute("aTextureId")) {
                this.geometry.addAttribute(
                    "aTextureId",
                    new InterleavedDataAttribute(this.interleavedData, 1, 5, false)
                )
            }
        }
        else if (this.geometry.getAttribute("aTextureId")) {
            this.geometry.removeAttribute("aTextureId")
        }
        //处理下，把buffer都加上先
        let bufferManager = this.renderer.bufferManager;
        //一个就够了，交错数据
        bufferManager.update(this.interleavedData, BUFFER_TYPE.ARRAY_BUFFER)
        //索引
        bufferManager.update(this.indexData, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER);
    }
    /**
     * 处理相应的绑定
     */
    start() {
        //着色器
        this.renderer.shaderManager.bind(this.shader);
        //vao
        this.renderer.vaoManager.bind(this.geometry, this.shader)
    }

    /**
     * 
     * element必须的属性
     * 
     * _texture  里面有.BaseTexture
     * _vertexData  
     * _indices 
     * _worldAlpha
     * _tintRGB
     * _uvs
     * 
     */
    render(element: IBatchElement) {
        if (!element._texture || !element._texture.valid) {
            return;
        }

        if (this.currentSize + (element._vertexData.length / 2) > this.size) {
            this.flush();
        }

        this.elements[this.currentIndex++] = element;

        this.currentSize += element._vertexData.length / 2;
        this.currentIndexSize += element._indices.length;
    }

    flush() {
        if (this.currentSize === 0) {
            return;
        }

        const gl = this.renderer.gl;
        const MAX_TEXTURES = this.MAX_TEXTURES;

        const vertexArray = this.getVertexArrays(this.currentSize);
        const indexArray = this.getIndexArray(this.currentIndexSize);

        const elements = this.elements;
        const groups = this.groups;

        const float32View = vertexArray;
        if (!float32View["uint32View"]) {
            float32View["uint32View"] = new Uint32Array(vertexArray.buffer);
        }
        const uint32View = float32View["uint32View"] as Uint32Array;

        const touch = this.renderer.textureManager.count;

        let index = 0;
        let indexCount = 0;
        let nextTexture;
        let currentTexture;
        let groupCount = 1;

        let textureCount = 0;
        let currentGroup = groups[0];

        //先用第一个的混色模式
        let blendMode = premultiplyBlendMode[elements[0]._texture.baseTexture.premultipliedAlpha ? 1 : 0][elements[0]._blendMode];

        currentGroup.textureCount = 0;
        currentGroup.start = 0;
        currentGroup.blend = blendMode;

        TICK++;

        let i;

        for (i = 0; i < this.currentIndex; ++i) {
            // upload the sprite elements...
            // they have all ready been calculated so we just need to push them into the buffer.

            const sprite = elements[i];

            elements[i] = null;

            nextTexture = sprite._texture.baseTexture;

            const spriteBlendMode = premultiplyBlendMode[nextTexture.premultipliedAlpha ? 1 : 0][sprite._blendMode];

            if (blendMode !== spriteBlendMode) {
                blendMode = spriteBlendMode;

                // force the batch to break!
                currentTexture = null;
                textureCount = MAX_TEXTURES;
                TICK++;
            }

            if (currentTexture !== nextTexture) {
                currentTexture = nextTexture;

                if (nextTexture._enabledId !== TICK) {
                    if (textureCount === MAX_TEXTURES) {
                        TICK++;

                        textureCount = 0;

                        currentGroup.size = indexCount - currentGroup.start;

                        currentGroup = groups[groupCount++];
                        currentGroup.textureCount = 0;
                        currentGroup.blend = blendMode;
                        currentGroup.start = indexCount;
                    }

                    nextTexture._touchedId = touch;
                    nextTexture._enabledId = TICK;
                    nextTexture._id = textureCount;

                    currentGroup.textures[currentGroup.textureCount++] = nextTexture;
                    textureCount++;
                }
            }
            // argb, nextTexture._id, float32View, uint32View, indexBuffer, index, indexCount);
            this.packGeometry(sprite, float32View, uint32View, indexArray, index, indexCount);

            // push a graphics..
            index += (sprite._vertexData.length / 2) * this.vertSize;
            indexCount += sprite._indices.length;
        }

        currentGroup.size = indexCount - currentGroup.start;

        var bufferManager = this.renderer.bufferManager;
        //顶点数据修改
        this.interleavedData.array = vertexArray;
        bufferManager.update(this.interleavedData, BUFFER_TYPE.ARRAY_BUFFER)
        //索引数据修改
        this.indexData.array = indexArray;
        bufferManager.update(this.indexData, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER)

        const textureManager = this.renderer.textureManager;
        for (i = 0; i < groupCount; i++) {
            const group = groups[i];
            const groupTextureCount = group.textureCount;

            for (let j = 0; j < groupTextureCount; j++) {
                if (group.textures[j] != textureManager.boundTextures[j]) {
                    //location已经和sampleId固定了，就算原先boundsTextures有，也要重新绑定位置
                    // console.log("bindTexture")
                    textureManager.bind(group.textures[j], j);
                }
                group.textures[j] = null;
            }
            //混色
            this.renderer.setBlendMode(group.blend)

            gl.drawElements(group.type, group.size, gl.UNSIGNED_SHORT, group.start * 2);
        }

        // reset elements for the next flush
        this.currentIndex = 0;
        this.currentSize = 0;
        this.currentIndexSize = 0;
    }

    packGeometry(element: IBatchElement, float32View: Float32Array, uint32View: Uint32Array, indexBuffer, index: number, indexCount: number) {
        const p = index / this.vertSize;// float32View.length / 6 / 2;
        const uvs = element._uvs;
        const indicies = element._indices;// geometry.getIndex().data;// indicies;
        const vertexData = element._vertexData;
        const textureId = element._texture.baseTexture["_id"];

        var _tintRGB = element._tintRGB == undefined ? 16777215 : element._tintRGB

        const alpha = Math.min(element._worldAlpha, 1.0);
        // we dont call extra function if alpha is 1.0, that's faster
        const argb = alpha < 1.0 && element._texture.baseTexture.premultipliedAlpha ? premultiplyTint(_tintRGB, alpha)
            : _tintRGB + (alpha * 255 << 24);

        // lets not worry about tint! for now..
        for (let i = 0; i < vertexData.length; i += 2) {
            float32View[index++] = vertexData[i];
            float32View[index++] = vertexData[i + 1];
            float32View[index++] = uvs[i];
            float32View[index++] = uvs[i + 1];
            uint32View[index++] = argb;
            float32View[index++] = textureId;
        }

        for (let i = 0; i < indicies.length; i++) {
            indexBuffer[indexCount++] = p + indicies[i];
        }
    }

    /**
     * 获得索引数组
     * @param size 索引数量，
     */
    private getIndexArray(size: number) {
        // 12 indices is enough for 2 quads
        const roundedP2 = nextPow2(Math.ceil(size / 12));
        //用这个作为键值存
        const roundedSizeIndex = log2(roundedP2);
        const roundedSize = roundedP2 * 12;

        let array = this.indexArrays[roundedSizeIndex];

        if (!array) {
            this.indexArrays[roundedSizeIndex] = array = new Uint16Array(roundedSize);
        }

        return array;
    }

    /**
     * 获取相应的顶点数据
     * @param size 
     */
    private getVertexArrays(size: number) {
        // 8 vertices is enough for 2 quads
        const roundedP2 = nextPow2(Math.ceil(size / 8));
        const roundedSizeIndex = log2(roundedP2);
        const roundedSize = roundedP2 * 8;

        //这有啥用
        // if (this._aBuffers.length <= roundedSizeIndex) {
        //     this._iBuffers.length = roundedSizeIndex + 1;
        // }

        let array = this.vertexArrays[roundedSize];

        if (!array) {
            this.vertexArrays[roundedSize] = array = new Float32Array(roundedSize);
        }
        return array;
    }

    destroy() {
        this.renderer.bufferManager.remove(this.interleavedData)
        this.renderer.bufferManager.remove(this.indexData);

        this.renderer.vaoManager.disposeByGeometry(this.geometry);

        if (this.shader) {
            this.shader.destroy();
            this.shader = null;
        }

        super.destroy();
    }
}
//放WebglRenderer那执行，BatchRenderer不往外引，不会执行，
// WebglRenderer.registerPlugin('batch', BatchRenderer);


export interface IBatchElement {
    _blendMode: BLEND_MODES,
    _texture: Texture,
    _worldAlpha: number,
    _vertexData: Float32Array,
    _indices: Uint16Array,
    _tintRGB?: number,
    _uvs: Float32Array,
}

export interface IBatchDrawCall {
    /**
     * 存储基础图片
     */
    textures: BaseTexture[];
    /**
     * 混色模式
     */
    blend: BLEND_MODES;
    /**
     * 纹理总数
     */
    textureCount: number;
    /**
     * 索引起始
     */
    start: number;
    /**
     * 顶点总数
     */
    size: number;
    /**
     * 绘制类型
     */
    type: DRAW_MODES;
}


/**
 * 检查最大纹理数是否合法
 * @param maxIfs 
 * @param gl 
 */
function checkMaxIfStatementsInShader(maxIfs: number, gl: WebGLRenderingContext) {
    const createTempContext = !gl;

    if (createTempContext) {
        const tinyCanvas = createCanvas()
        tinyCanvas.width = 1;
        tinyCanvas.height = 1;
        gl = createWebglContext(tinyCanvas);
    }

    const shader = gl.createShader(gl.FRAGMENT_SHADER);

    let fragTemplate = [
        'precision mediump float;',
        'void main(void){',
        'float test = 0.1;',
        '%forloop%',
        'gl_FragColor = vec4(0.0);',
        '}',
    ].join('\n');

    while (true) {
        const fragmentSrc = fragTemplate.replace(
            /%forloop%/gi,
            ((maxIfs) => {
                let src = '';
                for (let i = 0; i < maxIfs; ++i) {
                    if (i > 0) {
                        src += '\nelse ';
                    }
                    if (i < maxIfs - 1) {
                        src += `if(test == ${i}.0){}`;
                    }
                }
                return src;
            })(maxIfs)
        );

        gl.shaderSource(shader, fragmentSrc);
        gl.compileShader(shader);

        if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
            maxIfs = (maxIfs / 2) | 0;
        }
        else {
            break;
        }
    }
    //如果是临时建的，就丢失
    if (createTempContext) {
        if (gl.getExtension('WEBGL_lose_context')) {
            gl.getExtension('WEBGL_lose_context').loseContext();
        }
    }

    return maxIfs;
}

//
//顶点着色器程序
const VSHADER_SOURCE =
    "precision highp float;" +
    "attribute vec2 aVertexPosition;" +
    "attribute vec2 aTextureCoord;" +
    "attribute vec4 aColor;" +
    "attribute float aTextureId;" +

    "uniform mat3 uProjectionMatrix;" +
    // "uniform mat3 uModelMatrix;" +

    "varying vec2 vTextureCoord;" +
    "varying vec4 vColor;" +
    "varying float vTextureId;" +

    "void main(void){" +
    "gl_Position = vec4((uProjectionMatrix * vec3(aVertexPosition, 1.0)).xy, 0.0, 1.0);" +
    // "gl_Position = vec4((uProjectionMatrix *uModelMatrix* vec3(aVertexPosition, 1.0)).xy, 0.0, 1.0);" +
    "vTextureCoord = aTextureCoord;" +
    "vTextureId = aTextureId;" +
    "vColor = aColor;" +
    "}";
//片元着色器程序
const fragTemplate = [
    'precision mediump float;',
    'varying vec2 vTextureCoord;',
    'varying vec4 vColor;',
    'varying float vTextureId;',
    'uniform sampler2D uSamplers[%count%];',

    'void main(void){',
    'vec4 color;',
    '%forloop%',
    'gl_FragColor = color * vColor;',
    '}',
].join('\n');


/**
 * 创建批处理专用着色器，核心，切换纹理
 * @param maxTextures 最大纹理数
 */
export function generateMultiTextureShader(maxTextures: number) {
    // const vertexSrc = readFileSync(join(__dirname, './texture.vert'), 'utf8');
    let fragmentSrc = fragTemplate;

    fragmentSrc = fragmentSrc.replace(/%count%/gi, maxTextures + "");
    fragmentSrc = fragmentSrc.replace(
        /%forloop%/gi,
        (() => {
            let src = '';
            src += '\n';
            src += '\n';
            for (let i = 0; i < maxTextures; i++) {
                if (i > 0) {
                    src += '\nelse ';
                }
                if (i < maxTextures - 1) {
                    src += `if(vTextureId < ${i}.5)`;
                }
                src += '\n{';
                src += `\n\tcolor = texture2D(uSamplers[${i}], vTextureCoord);`;
                src += '\n}';
            }
            src += '\n';
            src += '\n';
            return src;
        })()
    );
    const shader = new Shader(VSHADER_SOURCE, fragmentSrc, null, ["uSamplers"]);

    return shader;
}

