//buffer和pointer是对应的，attr通道只需要开启或关闭，所以pointer的时候单独执行

import { Dict } from "../../utils";
import { DataAttribute } from "../webgl/DataAttribute";
import { Geometry } from "../webgl/Geometry";
import { InterleavedDataAttribute } from "../webgl/InterleavedDataAttribute";
import { Shader } from "../webgl/Shader";
import { WebglRenderer } from "../WebglRenderer";
import { BufferManager } from "./BufferManager";

//需要根据着色器和几何缓存，都用instanceId，到时继承hashObject
export interface IVaoState {
    geometryId?: number,
    shaderId?: number,

    tempAttributes: number[],
    usedAttributes: number[],
    vao: any,
    attributes: Dict<DataAttribute>,
    attributesNum: number,
    //索引，可能索引可以外部随便给，不一定是用几何内部的
    index: DataAttribute
}
export class VaoManager {
    private vaoExtension
    private maxAttribs: number;
    private defaultState: IVaoState;
    private currentState: IVaoState;
    /**
     * key用几何id和着色器id，仅仅针对有vao扩展时，不然只用一个defaultState
     * 
     */
    private boundVaoStates: Dict<Dict<IVaoState>>
    constructor(public renderer: WebglRenderer) {
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);

    }
    onContextChange() {
        const gl = this.renderer.gl;
        this.vaoExtension = gl.getExtension('OES_vertex_array_object') ||
            gl.getExtension('MOZ_OES_vertex_array_object') ||
            gl.getExtension('WEBKIT_OES_vertex_array_object');
        this.maxAttribs = gl.getParameter(gl.MAX_VERTEX_ATTRIBS);

        this.defaultState = this.createVaoState(null);
        this.currentState = this.defaultState;
        this.boundVaoStates = {};
    }
    /**
     * 处理绑定的纹理通道，这里需要在BufferManager.update之后，确保WebglBuffer都有，也需要先bind着色器后
     * 包括，bindBuffer，pointer，enableAttr
     * @param geometry 
     * @param shader 
     */
    bind(geometry: Geometry, shader: Shader) {
        const gl = this.renderer.gl;
        //是否需要更新顶点通道
        let updateAttributes = false;

        if (this.vaoExtension) {

            let state = this.getBoundState(geometry, shader);

            if (this.currentState !== state) {
                this.currentState = state;
                this.vaoExtension.bindVertexArrayOES(state.vao);
            }
            updateAttributes = this.checkAttributesUpdate(geometry);
            //需要更新的话，重新缓存
            if (updateAttributes) {
                const cache = {};
                const attributes = geometry._attributes;
                let attributesNum = 0;
                for (const key in attributes) {
                    const attribute = attributes[key];
                    cache[key] = attribute;
                    attributesNum++;
                }
                this.currentState.attributes = cache;
                this.currentState.attributesNum = attributesNum;
                this.currentState.index = geometry.index;
            }
        }
        //不支持vao对象后，判断几何和着色器是不是同一个，貌似这样不保险，同一个几何里的attr可以变啊，TODO
        else if (this.currentState.geometryId !== geometry.instanceId ||
            this.currentState.shaderId !== shader.instanceId
        ) {
            this.currentState.geometryId = geometry.instanceId;
            this.currentState.shaderId = shader.instanceId;
            //标记需要更新
            updateAttributes = true;
        }
        let bufferManager: BufferManager = this.renderer.bufferManager;
        let index = geometry.index;
        //这里为啥需要，TODO，暂时感觉是为了这个新的index有glbuffer，现在不需要，three里这个index可以不是geo里的
        // if (index) {
        //     bufferManager.update(index, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER);
        // }
        //不需要更新，retrun
        if (!updateAttributes) return;
        //重置，也许以后有用，到时开放方法，TODO
        const tempAttributes = this.currentState.tempAttributes;
        for (let i = 0; i < tempAttributes.length; i++) {
            tempAttributes[i] = 0;
        }
        //处理各个通道
        const geoAttributes = geometry._attributes;

        const shaderAttributes = this.renderer.shaderManager.get(shader).attributes;

        let lastBuffer = null;
        //用着色器的
        for (const name in shaderAttributes) {
            const { size, location } = shaderAttributes[name];
            //这个是否有必要
            if (location < 0) continue;
            let geoAttribute = geoAttributes[name];
            //instance的情况，TODO
            // if (!geoAttribute) {
            //     if (name === 'instanceMatrix' && object.instanceMatrix) geoAttribute = object.instanceMatrix;
            //     if (name === 'instanceColor' && object.instanceColor) geoAttribute = object.instanceColor;
            // }
            if (!geoAttribute) continue;
            //
            let { normalized, itemSize } = geoAttribute;
            let offset = 0;
            //
            if (geoAttribute.instanceType == "InterleavedDataAttribute") {
                offset = (geoAttribute as InterleavedDataAttribute).offset;
                itemSize = (geoAttribute as InterleavedDataAttribute).data.stride;
            }

            const bufferData = bufferManager.get(geoAttribute);

            // TODO Attribute may not be available on context restore
            if (!bufferData) continue;

            const { buffer, type, bytesPerElement } = bufferData;
            //打开通道，此外instance网格的打开通道方式不一样，其他都一致
            this.enableAttribute(location);
            //绑定buffer，和pointer关联，同一个没必要重复绑定
            if (lastBuffer !== buffer) {
                gl.bindBuffer(gl.ARRAY_BUFFER, buffer);
                lastBuffer = buffer;
            }
            //pointer
            gl.vertexAttribPointer(
                location,
                size,
                type,
                normalized,
                itemSize * bytesPerElement,
                offset * bytesPerElement
            );
            //对于交错数据和instance的，TODO
        }
        //关闭不用的通道
        const usedAttributes = this.currentState.usedAttributes;
        for (let i = 0; i < usedAttributes.length; i++) {
            if (usedAttributes[i] !== tempAttributes[i]) {
                gl.disableVertexAttribArray(i);
                usedAttributes[i] = 0;
            }
        }
        //索引绑定
        if (index) {
            gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferManager.get(index).buffer);
        }

    }
    disposeByGeometry(geometry: Geometry) {
        const shaderMap = this.boundVaoStates[geometry.instanceId];
        if (!shaderMap) return;
        for (const shaderId in shaderMap) {
            const state = shaderMap[shaderId];
            this.vaoExtension.deleteVertexArrayOES(state.vao);
            delete shaderMap[shaderId];
        }
        delete this.boundVaoStates[geometry.instanceId];
    }
    disposeByShader(shader: Shader) {
        for (const geometryId in this.boundVaoStates) {
            const shaderMap = this.boundVaoStates[geometryId];
            const id = shader.instanceId;
            if (!shaderMap[id]) continue;
            const state = shaderMap[id];
            this.vaoExtension.deleteVertexArrayOES(state.vao);
            delete shaderMap[id];
        }

    }
    unbind() {
        if (this.vaoExtension) {
            this.vaoExtension.bindVertexArrayOES(null);
        }
    }
    private createVaoState(vao): IVaoState {
        const tempAttributes = [];
        const usedAttributes = [];

        for (let i = 0; i < this.maxAttribs; i++) {
            tempAttributes[i] = usedAttributes[i] = 0;
        }

        return {
            geometryId: null,
            shaderId: null,

            tempAttributes,
            usedAttributes,
            vao,
            attributes: {},
            attributesNum: 0,
            index: null
        };
    }
    private checkAttributesUpdate(geometry: Geometry) {
        const { attributes, attributesNum, index } = this.currentState;
        const geoAttributes = geometry._attributes;
        let geoAttributesNum = 0;

        for (const key in geoAttributes) {
            const cachedAttribute = attributes[key];
            const geoAttribute = geoAttributes[key];
            //有不一致的返回true
            if (cachedAttribute != geoAttribute) return true;
            geoAttributesNum++;
        }
        //通道数量不一致
        if (attributesNum !== geoAttributesNum) return true;
        //索引不一致
        if (index !== geometry.index) return true;
        return false;
    }
    private enableAttribute(location: number) {
        const { tempAttributes, usedAttributes } = this.currentState;
        tempAttributes[location] = 1;
        //原来是关闭的
        if (!usedAttributes[location]) {
            this.renderer.gl.enableVertexAttribArray(location);
            usedAttributes[location] = 1;
        }
    }
    private getBoundState(geometry: Geometry, shader: Shader) {
        let shaderMap = this.boundVaoStates[geometry.instanceId];
        if (!shaderMap) {
            shaderMap = this.boundVaoStates[geometry.instanceId] = {};
        }

        let state = shaderMap[shader.instanceId];
        if (!state) {
            state = shaderMap[shader.instanceId] = this.createVaoState(
                this.vaoExtension.createVertexArrayOES()
            );
        }
        return state;
    }
    destroy() {
        this.renderer.removeEventListener('onContextChange', this.onContextChange, this);
        this.renderer = null;
    }
}