
import { WebglRenderer } from '../WebglRenderer';
import Graphics from '../../graphics/Graphics';

/**
 * 管理模板缓存，主要用于遮罩
 * @class
 */
export default class StencilManager {
    private stencilMaskStack: Graphics[] = null;
    constructor(private renderer: WebglRenderer) { }

    setMaskStack(stencilMaskStack: Graphics[]) {
        const curStackLen = this.stencilMaskStack ? this.stencilMaskStack.length : 0;
        const newStackLen = stencilMaskStack.length;

        this.stencilMaskStack = stencilMaskStack;

        const gl = this.renderer.gl;

        if (newStackLen !== curStackLen) {
            if (newStackLen === 0) {
                gl.disable(gl.STENCIL_TEST);
            }
            else {
                gl.enable(gl.STENCIL_TEST);
                this._useCurrent();
            }
        }
    }

    push(graphics: Graphics) {
        const gl = this.renderer.gl;
        const prevMaskCount = this.stencilMaskStack.length;

        if (prevMaskCount === 0) {
            this.renderer.framebufferManager.forceStencil();//待测试

            gl.enable(gl.STENCIL_TEST);

            // gl.clearStencil(0);//只要有一个地方执行就行？push或pop
            // gl.clear(gl.STENCIL_BUFFER_BIT);
            // gl.enable(gl.STENCIL_TEST);

            // gl.disable(gl.STENCIL_TEST);
            // gl.clear(gl.STENCIL_BUFFER_BIT);
            // gl.clearStencil(0);
        }

        this.stencilMaskStack.push(graphics);

        // Increment the refference stencil value where the new mask overlaps with the old ones.
        gl.colorMask(false, false, false, false);
        gl.stencilFunc(gl.EQUAL, prevMaskCount, this._getBitwiseMask());
        gl.stencilOp(gl.KEEP, gl.KEEP, gl.INCR);

        graphics.renderable = true;
        var tempAlpha = graphics._worldAlpha;//因为如果alpha为0，不会渲染，
        graphics._worldAlpha = 1;
        graphics.renderWebGL(this.renderer);
        this.renderer.batchManager.flush();
        graphics.renderable = false;
        graphics._worldAlpha = tempAlpha;

        this._useCurrent();
    }

    pop() {

        const gl = this.renderer.gl;
        const graphics = this.stencilMaskStack.pop();

        if (this.stencilMaskStack.length === 0) {
            // the stack is empty!
            gl.disable(gl.STENCIL_TEST);
            gl.clear(gl.STENCIL_BUFFER_BIT);
            gl.clearStencil(0);
        }
        else {
            // Decrement the refference stencil value where the popped mask overlaps with the other ones
            gl.colorMask(false, false, false, false);
            gl.stencilOp(gl.KEEP, gl.KEEP, gl.DECR);

            graphics.renderable = true;
            var tempAlpha = graphics._worldAlpha;//因为如果alpha为0，不会渲染，
            graphics._worldAlpha = 1;
            graphics.renderWebGL(this.renderer);
            this.renderer.batchManager.flush();
            graphics.renderable = false;
            graphics._worldAlpha = tempAlpha;
            this._useCurrent();
        }
    }

    private _useCurrent() {
        const gl = this.renderer.gl;

        gl.colorMask(true, true, true, true);
        gl.stencilFunc(gl.EQUAL, this.stencilMaskStack.length, this._getBitwiseMask());
        gl.stencilOp(gl.KEEP, gl.KEEP, gl.KEEP);
    }

    private _getBitwiseMask() {
        return (1 << this.stencilMaskStack.length) - 1;
    }

    destroy() {
        this.renderer = null;

        this.stencilMaskStack = null;
    }
}
