import { Matrix } from "../../math";
import { Dict } from "../../utils";
import { Shader } from "../webgl/Shader";
import { compileProgram } from "../webgl/shader/compileProgram";
import { extractAttributes, IShaderAttribute } from "../webgl/shader/extractAttributes";
import { extractUniforms, IGlShaderUniformData } from "../webgl/shader/extractUniforms";
import { generateUniformAccessObject } from "../webgl/shader/generateUniformAccessObject";
import { syncUniforms } from "../webgl/shader/syncUniforms";
import { WebglRenderer } from "../WebglRenderer";

export interface IWebglShader {
    program: WebGLProgram;
    attributes: Dict<IShaderAttribute>;
    uniformsData: Dict<IGlShaderUniformData>;
    uniforms: any;
}

//一样和渲染器绑定，没啥多余的功能，考虑是否根据着色器语言进行绑定？还是自行处理GLShader
export class ShaderManager {
    currentShader: Shader;
    private map: WeakMap<Shader, IWebglShader>
    //需要存一下所有的着色器吗？
    constructor(public renderer: WebglRenderer) {
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);
    }
    onContextChange() {
        this.map = new WeakMap();
    }
    get(shader: Shader): IWebglShader {
        return this.map.get(shader);
    }
    /**
     * 只绑定着色器，uniform的传值，自行用返回的shader.uniforms修改，而且暂时都不判断是否修改，直接赋值，后续考虑怎么在值相同的情况下不再传TODO
     * @param shader
     * @param doSyncUniforms 是否同步unifroms，默认false
     */
    bind(shader: Shader, doSyncUniforms: boolean = false): IWebglShader {
        if (this.currentShader == shader) return this.get(shader);
        this.currentShader = shader;
        let glShader = this.get(shader);
        if (!glShader) {
            glShader = this.createGlShader(shader);
            this.map.set(shader, glShader);
        }
        this.renderer.gl.useProgram(glShader.program);
        //一般用不到，TODO
        if (doSyncUniforms && shader.uniformsData) syncUniforms(glShader, shader.uniformsData, this.renderer);
        return glShader;
    }
    /**
     * 外部手动调用，考虑是否需要收集所有shader，到时候集中destroy，TODO
     * @param shader 
     * @returns 
     */
    disposeShader(shader: Shader) {
        let glShader = this.get(shader);
        if (!glShader) return;
        this.renderer.gl.deleteProgram(glShader.program);
        this.map.delete(shader);
    }
    /**
     * 对于2d，需要转换屏幕坐标，左上角为原点
     */
    project2d(projectionMatrix: Matrix) {
        if (!this.currentShader) return;
        let glShader = this.get(this.currentShader);
        let uniformNames = ["projectionMatrix", "uProjectionMatrix"];
        let project2dUniformName = this.currentShader.project2dUniformName;
        if (project2dUniformName) uniformNames.unshift(project2dUniformName);

        uniformNames.forEach((u) => {
            if (glShader.uniforms[u]) {
                glShader.uniforms[u] = projectionMatrix.toArray(true);
            }
        })
    }

    private createGlShader(shader: Shader): IWebglShader {
        const gl = this.renderer.gl;
        const { vertexSrc, fragmentSrc, attributeLocations, removeBracketUnifroms } = shader;
        //编译着色器
        var program = compileProgram(gl, vertexSrc, fragmentSrc, attributeLocations);
        //解析顶点通道
        var attributes = extractAttributes(gl, program);
        //解析uniform通道
        var uniformsData = extractUniforms(gl, program, removeBracketUnifroms);
        //给uniform建立get set映射
        var uniforms = generateUniformAccessObject(gl, uniformsData);

        return {
            program,
            attributes,
            uniformsData,
            uniforms
        }
    }
    destroy() {
        this.renderer = null;
    }
}