import { Matrix, Rectangle } from "../../math";
import { RenderTexture } from "../../texture";
import { BaseRenderTexture } from "../../texture/BaseRenderTexture";
import { Framebuffer } from "../webgl/Framebuffer";
import { WebglRenderer } from "../WebglRenderer";
import { TextureManager } from "./TextureManager";

export interface IGLFrameBuffer {
    framebuffer: WebGLFramebuffer;
    stencil?: WebGLRenderbuffer;
    dirtyId: number;
    mipLevel: number;
}
export class FramebufferManager {
    public readonly _managedFramebuffers: Array<Framebuffer> = [];
    public current: Framebuffer;
    public viewport: Rectangle = new Rectangle();
    private map: WeakMap<Framebuffer, IGLFrameBuffer>;

    /////renderTexture的放这里;
    currentRenderTexture: RenderTexture;
    public defaultMaskStack: any[] = [];
    public readonly sourceFrame: Rectangle = new Rectangle();
    public readonly destinationFrame: Rectangle = new Rectangle();
    transform: Matrix;

    projectionMatrix: Matrix = new Matrix();
    constructor(public renderer: WebglRenderer) {
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);
    }
    onContextChange() {
        this.map = new WeakMap();
        this.current = null;
        this.viewport.clear();
        this.disposeAll(true);
        //有些扩展的，用到时再说了TODO
    }
    bindRenderTexture(
        renderTexture?: RenderTexture,
        sourceFrame?: Rectangle,
        destinationFrame?: Rectangle,
        transform?: Matrix
    ): void {
        const renderer = this.renderer;

        this.currentRenderTexture = renderTexture;

        let baseTexture: BaseRenderTexture;
        let framebuffer: Framebuffer;
        //有的时候
        if (renderTexture) {
            baseTexture = renderTexture.baseTexture as BaseRenderTexture;
            if (!sourceFrame) {
                sourceFrame = new Rectangle(
                    0, 0,
                    renderTexture.frame.width,
                    renderTexture.frame.height
                );
            }

            if (!destinationFrame) {
                destinationFrame = new Rectangle(
                    renderTexture.frame.x,
                    renderTexture.frame.y,
                    sourceFrame.width,
                    sourceFrame.height
                );
            }
            framebuffer = baseTexture.framebuffer;
        }
        else {
            if (!sourceFrame) {
                sourceFrame = new Rectangle(
                    0, 0,
                    renderer.width,
                    renderer.height
                );
            }

            if (!destinationFrame) {
                destinationFrame = new Rectangle();
                destinationFrame.width = sourceFrame.width;
                destinationFrame.height = sourceFrame.height;
            }
        }

        const viewportFrame = new Rectangle();

        viewportFrame.x = destinationFrame.x;
        viewportFrame.y = destinationFrame.y;
        viewportFrame.width = destinationFrame.width;
        viewportFrame.height = destinationFrame.height;
        //如果root，处理下y，为啥区分root
        if (!renderTexture) {
            viewportFrame.y = renderer.height - (viewportFrame.y + viewportFrame.height);
        }
        //这里会直接viewport
        this.bind(framebuffer, viewportFrame);
        //计算project,考虑是否可以缓存下
        // this.renderer.projection.update(destinationFrame, sourceFrame, resolution, !framebuffer);
        const pm = this.projectionMatrix;
        pm.identity();
        const sign = framebuffer ? 1 : -1;//这里迟早改下，TODO
        pm.a = (1 / sourceFrame.width * 2);
        pm.d = sign * (1 / sourceFrame.height * 2);
        pm.tx = -1 - (sourceFrame.x * pm.a);
        pm.ty = -sign - (sourceFrame.y * pm.d);
        //有偏移的处理下
        if (transform) pm.append(transform);
        this.transform = transform || null;
        //相应的激活的着色器的projection也需要同样处理，，其他时候自己取project赋值
        this.renderer.shaderManager.project2d(this.projectionMatrix);
        //TODO
        // if (renderTexture) {
        //     this.renderer.maskManager.setMaskStack(baseTexture.maskStack);
        // }
        // else {
        //     this.renderer.maskManager.setMaskStack(this.defaultMaskStack);
        // }
        //暂时这么处理，只有模版缓存，setMaskStack还需要加逻辑判断，TODO
        this.renderer.stencilManager.setMaskStack(renderTexture ? baseTexture.maskStack : this.defaultMaskStack)

        this.sourceFrame.copy(sourceFrame);
        this.destinationFrame.copy(destinationFrame);
    }
    reset() {
        this.bindRenderTexture(null);
    }
    private bind(framebuffer?: Framebuffer, frame?: Rectangle, mipLevel = 0): void {
        const gl = this.renderer.gl;
        //作为主窗口root
        if (!framebuffer) {
            //有绑定过的移除
            if (this.current) {
                this.current = null;
                gl.bindFramebuffer(gl.FRAMEBUFFER, null);
            }
            if (frame) {
                this.setViewport(frame.x, frame.y, frame.width, frame.height);
            }
            else {
                this.setViewport(0, 0, this.renderer.width, this.renderer.height);
            }
            return;
        }
        let fbo = this.map.get(framebuffer)
        if (!fbo) {
            fbo = {
                dirtyId: -1,
                mipLevel: 0,
                framebuffer: gl.createFramebuffer()
            };
            this.map.set(framebuffer, fbo);
            //进管理
            this._managedFramebuffers.push(framebuffer);
        }
        if (this.current !== framebuffer) {
            this.current = framebuffer;
            gl.bindFramebuffer(gl.FRAMEBUFFER, fbo.framebuffer);
        }
        // make sure all textures are unbound..
        if (fbo.mipLevel !== mipLevel) {
            framebuffer._dirtyId++;
            fbo.mipLevel = mipLevel;
        }
        //TODO
        const textureManager: any = this.renderer.textureManager;
        // now check for updates...
        if (fbo.dirtyId !== framebuffer._dirtyId) {
            fbo.dirtyId = framebuffer._dirtyId;
            const colorTexture = framebuffer.colorTexture;
            if (colorTexture) {
                (textureManager as TextureManager).bind(colorTexture, 0);
                gl.framebufferTexture2D(
                    gl.FRAMEBUFFER,
                    gl.COLOR_ATTACHMENT0 + 0,
                    gl.TEXTURE_2D, // texture.target,
                    (textureManager as TextureManager).get(colorTexture),
                    mipLevel);
            }
            // if (framebuffer.depthTexture) {
            //     const writeDepthTexture = this.writeDepthTexture;

            //     if (writeDepthTexture) {
            //         const depthTexture = framebuffer.depthTexture;

            //         this.renderer.texture.bind(depthTexture, 0);

            //         gl.framebufferTexture2D(gl.FRAMEBUFFER,
            //             gl.DEPTH_ATTACHMENT,
            //             gl.TEXTURE_2D,
            //             depthTexture._glTextures[this.CONTEXT_UID].texture,
            //             mipLevel);
            //     }
            // }
            //和上面的是冲突的，上面的不用，就用下面的
            // if ((framebuffer.stencil || framebuffer.depth) && !(framebuffer.depthTexture && this.writeDepthTexture)) {
            if (framebuffer.stencil) {//帧缓存里需要模版缓存时，比如滤镜里的显示对象有遮罩
                fbo.stencil = fbo.stencil || gl.createRenderbuffer();
                gl.bindRenderbuffer(gl.RENDERBUFFER, fbo.stencil);
                gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_STENCIL, framebuffer.width, framebuffer.height);
                gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, fbo.stencil);
            }
            //否则删除
            else if (fbo.stencil) {
                gl.deleteRenderbuffer(fbo.stencil);
                fbo.stencil = null;
            }
        }

        //所有的纹理解除绑定
        if (framebuffer.colorTexture) (textureManager as TextureManager).unbind(framebuffer.colorTexture);

        // if (framebuffer.depthTexture) {
        //     (textureManager as TextureManager).unbind(framebuffer.depthTexture);
        // }

        if (frame) {
            const mipWidth = (frame.width >> mipLevel);
            const mipHeight = (frame.height >> mipLevel);

            const scale = mipWidth / frame.width;

            this.setViewport(
                frame.x * scale,
                frame.y * scale,
                mipWidth,
                mipHeight
            );
        }
        else {
            const mipWidth = (framebuffer.width >> mipLevel);
            const mipHeight = (framebuffer.height >> mipLevel);

            this.setViewport(0, 0, mipWidth, mipHeight);
        }

    }

    /**
     * webgl窗口，注意是左下角开始
     * @param {Number} x 左下角x
     * @param {Number} y 左下角y
     * @param {Number} width 
     * @param {Number} height 
     */
    setViewport(x: number, y: number, width: number, height: number): void {
        const v = this.viewport;
        //处理下，整数
        x = Math.round(x);
        y = Math.round(y);
        width = Math.round(width);
        height = Math.round(height);

        if (v.width !== width || v.height !== height || v.x !== x || v.y !== y) {
            v.x = x;
            v.y = y;
            v.width = width;
            v.height = height;
            this.renderer.gl.viewport(x, y, width, height);
        }
    }
    /**
     * 对于帧缓存内的遮罩，必须开启
     * @returns 
     */
    forceStencil(): void {
        const framebuffer = this.current;
        //是root
        if (!framebuffer) return;

        const fbo = this.map.get(framebuffer);
        //没有或已经加了
        if (!fbo || fbo.stencil) return;
        framebuffer.stencil = true;

        const w = framebuffer.width;
        const h = framebuffer.height;
        const gl = this.renderer.gl;
        const stencil = gl.createRenderbuffer();
        gl.bindRenderbuffer(gl.RENDERBUFFER, stencil);
        gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_STENCIL, w, h);
        fbo.stencil = stencil;
        gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, stencil);
    }
    private clearColor: [number, number, number, number] = [0, 0, 0, 0]
    clear(clearColor?: [number, number, number, number]): void {
        clearColor = clearColor || this.clearColor
        const gl = this.renderer.gl;
        const destinationFrame = this.destinationFrame;
        const baseFrame = this.currentRenderTexture ? this.currentRenderTexture.baseTexture : this.renderer;
        const clearMask = destinationFrame.width !== baseFrame.width || destinationFrame.height !== baseFrame.height;

        if (clearMask) {
            let { x, y, width, height } = this.viewport;

            x = Math.round(x);
            y = Math.round(y);
            width = Math.round(width);
            height = Math.round(height);

            // TODO: ScissorSystem should cache whether the scissor test is enabled or not.
            this.renderer.gl.enable(this.renderer.gl.SCISSOR_TEST);
            this.renderer.gl.scissor(x, y, width, height);
        }

        gl.clearColor(...clearColor);
        //默认，以后有需要再改
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

        if (clearMask) {
            gl.disable(gl.SCISSOR_TEST);
            //这里原先考虑其他有裁切的地方，需要恢复的意思，目前没有，直接处理,TODO到时得用起来
            // this.renderer.scissor.pop();
        }


    }

    /**
     * 
     * @param framebuffer 
     * @param contextLost 
     * @returns 
     */
    dispose(framebuffer: Framebuffer, contextLost?: boolean): void {
        const fbo = this.map.get(framebuffer);
        const gl = this.renderer.gl;
        if (!fbo) return;

        this.map.delete(framebuffer)

        const index = this._managedFramebuffers.indexOf(framebuffer);
        if (index >= 0) {
            this._managedFramebuffers.splice(index, 1);
        }
        //不是上下文丢失的情况需要相应删除Framebuffer
        if (!contextLost) {
            gl.deleteFramebuffer(fbo.framebuffer);
            if (fbo.stencil) {
                gl.deleteRenderbuffer(fbo.stencil);
            }
        }
    }

    disposeAll(contextLost?: boolean): void {
        for (let i = 0; i < this._managedFramebuffers.length; i++) {
            this.dispose(this._managedFramebuffers[i], contextLost);
        }
        this._managedFramebuffers.length = 0;
    }
    destroy() {
        this.renderer = null;
    }
}