import { Rectangle, Matrix, Point } from "../../math";
import { Filter } from "../../filter";
import { WebglRenderer } from "../WebglRenderer";
import { Quad } from "../webgl/Quad";
import { DisplayObject } from "../../display/DisplayObject";
import { Dict, nextPow2 } from "../../utils";
import { RenderTexture } from "../../texture";
import { UniformType } from "../webgl/shader/UniformType";
import { BaseRenderTexture } from "../../texture/BaseRenderTexture";
import { Sprite } from "../../display";
import { syncUniforms } from "../webgl/shader/syncUniforms";
import { BUFFER_TYPE } from "../../const";

export class FilterState {
    renderTexture: RenderTexture = null;
    //z作用的显示对象
    target: { filterArea: Rectangle, getBounds(skipUpdate?: boolean): Rectangle; } = null;
    resolution: number = 1;
    sourceFrame: Rectangle = new Rectangle();
    destinationFrame: Rectangle = new Rectangle();
    filters: Filter[] = [];

    /** Original render-target source frame. */
    bindingSourceFrame: Rectangle = new Rectangle();
    /** Original render-target destination frame. */
    bindingDestinationFrame: Rectangle = new Rectangle();
    //为了还原原先renderer的数据
    transform: Matrix = null;
    constructor() { }
    clear() {
        this.filters = null;
        this.target = null;
        this.renderTexture = null;
    }
}

const screenKey = 'screen';

const tempRect = new Rectangle();

const tempPoints = [new Point(), new Point(), new Point(), new Point()];
const tempMatrix = new Matrix();

export class FilterManager {

    quad: Quad = new Quad();
    pool: Dict<RenderTexture[]> = {};
    filterStack: FilterState[] = [];
    statePool: FilterState[] = [];
    _screenWidth: number;
    _screenHeight: number;

    private activeState: FilterState;
    /**
     * @param {WebGLRenderer} renderer - The renderer this manager works for.
     */
    constructor(private renderer: WebglRenderer) {

        this._screenWidth = renderer.width;
        this._screenHeight = renderer.height;
        this.renderer.addEventListener('onContextChange', this.onContextChange, this);
        this.renderer.addEventListener('onPreRender', this.onPreRender, this);
    }
    onContextChange() {
        this.emptyPool();
    }
    /**
     * 压栈
     * @param {DisplayObject} target 
     * @param {Filter[]} filters
     */
    pushFilter(target: DisplayObject, filters: Filter[]) {

        const renderer = this.renderer;
        const filterStack = this.filterStack;
        //取一个状态
        const state = this.statePool.pop() || new FilterState();
        const framebufferManager = renderer.framebufferManager;

        let resolution = filters[0].resolution,
            padding = filters[0].padding,
            autoFit = filters[0].autoFit;

        for (let i = 1; i < filters.length; i++) {
            const filter = filters[i];
            resolution = Math.min(resolution, filter.resolution);
            padding += filter.padding;
            autoFit = autoFit && filter.autoFit;
        }
        //??
        if (filterStack.length === 1) {
            filterStack[0].renderTexture = framebufferManager.currentRenderTexture;
        }

        filterStack.push(state);

        state.resolution = resolution;

        state.target = target;
        state.sourceFrame.copy(target.filterArea || target.getBounds(true));

        state.sourceFrame.pad(padding);

        if (autoFit) {
            const sourceFrameProjected = tempRect.copy(framebufferManager.sourceFrame);

            if (framebufferManager.transform) {
                this.transformAABB(
                    tempMatrix.copy(framebufferManager.transform).invert(),
                    sourceFrameProjected
                );
            }
            state.sourceFrame.fit(sourceFrameProjected);
        }
        // Round sourceFrame in screen space based on render-texture.
        this.roundFrame(
            state.sourceFrame,
            1,// renderTextureSystem.current ? renderTextureSystem.current.resolution : renderer.resolution,
            framebufferManager.sourceFrame,
            framebufferManager.destinationFrame,
            framebufferManager.transform,
        );

        state.renderTexture = this.getRenderTexture(
            state.sourceFrame.width,
            state.sourceFrame.height,
            resolution
        );
        state.filters = filters;

        state.destinationFrame.width = state.renderTexture.width;
        state.destinationFrame.height = state.renderTexture.height;

        const destinationFrame = tempRect;

        destinationFrame.x = 0;
        destinationFrame.y = 0;
        destinationFrame.width = state.sourceFrame.width;
        destinationFrame.height = state.sourceFrame.height;

        state.renderTexture.filterFrame = state.sourceFrame;
        state.bindingSourceFrame.copy(framebufferManager.sourceFrame);
        state.bindingDestinationFrame.copy(framebufferManager.destinationFrame);

        state.transform = framebufferManager.transform;
        framebufferManager.transform = null;
        //renderTexture清空一下
        framebufferManager.bindRenderTexture(state.renderTexture, state.sourceFrame, destinationFrame);
        framebufferManager.clear([0, 0, 0, 0]);
    }

    /**
     * 栈出，并应用
     */
    popFilter() {
        const filterStack = this.filterStack;
        const state = filterStack.pop();
        const filters = state.filters;

        this.activeState = state;
        //最后一项即上一个
        const lastState = filterStack[filterStack.length - 1];

        //只有一个滤镜的
        if (filters.length === 1) {
            filters[0].apply(this, state.renderTexture, lastState.renderTexture, false, state);
            this.recoverRenderTexture(state.renderTexture);
        }
        else {
            let flip = state.renderTexture;
            let flop = this.getRenderTexture(
                flip.width,
                flip.height,
                state.resolution
            );

            flop.filterFrame = flip.filterFrame;

            let i = 0;

            for (i = 0; i < filters.length - 1; ++i) {
                filters[i].apply(this, flip, flop, true, state);
                const t = flip;
                flip = flop;
                flop = t;
            }

            filters[i].apply(this, flip, lastState.renderTexture, false, state);

            this.recoverRenderTexture(flip);
            this.recoverRenderTexture(flop);
        }
        //回收
        state.clear();
        this.statePool.push(state);
    }

    /**
     * 给filter调用的
     */
    applyFilter(filter: Filter, input: RenderTexture, output: RenderTexture, clear?: boolean) {
        const renderer = this.renderer;
        const { gl, framebufferManager, vaoManager, bufferManager } = renderer;
        const state = this.activeState;
        //先把着色器绑了，后面bindRenderTexture里会处理projection，考虑怎么搞优雅点
        let shader = renderer.shaderManager.bind(filter);
        //绑定output纹理
        if (output === this.filterStack[this.filterStack.length - 1].renderTexture) {
            // Restore projection transform if rendering into the output render-target.
            framebufferManager.transform = state.transform;
        }
        else {
            // Prevent projection within filtering pipeline.
            framebufferManager.transform = null;
        }

        if (output && output.filterFrame) {
            const destinationFrame = tempRect;
            destinationFrame.x = 0;
            destinationFrame.y = 0;
            destinationFrame.width = output.filterFrame.width;
            destinationFrame.height = output.filterFrame.height;
            framebufferManager.bindRenderTexture(output, output.filterFrame, destinationFrame);
        }
        else if (output !== this.filterStack[this.filterStack.length - 1].renderTexture) {
            framebufferManager.bindRenderTexture(output);
        }
        else {
            // Restore binding for output render-target.
            framebufferManager.bindRenderTexture(
                output,
                state.bindingSourceFrame,
                state.bindingDestinationFrame
            );
        }
        //是否清空
        if (clear) {
            framebufferManager.clear([0, 0, 0, 0]);
        }
        //通用的unifrom属性同步下
        if (filter.uniformsData.filterArea) {
            //为啥这样的顺序，TODO
            filter.uniformsData.filterArea.value = [
                state.destinationFrame.width,
                state.destinationFrame.height,
                state.sourceFrame.x,
                state.sourceFrame.y,
            ]
        }
        if (filter.uniformsData.filterClamp) {
            filter.uniformsData.filterClamp.value = [
                0, 0,
                (state.sourceFrame.width - 1) / state.destinationFrame.width,
                (state.sourceFrame.height - 1) / state.destinationFrame.height,
            ]
        }
        //使用的纹理
        filter.uniformsData.uSampler.value = input;
        //projection
        if (filter.uniformsData.projectionMatrix) {
            filter.uniformsData.projectionMatrix.value = framebufferManager.projectionMatrix;
        }
        //uniform属性同步下，自定义属性在各自的apply里执行
        syncUniforms(shader, filter.uniformsData, renderer,0)
        //顶点和uv计算，其实可以放到着色器里，以后再说 TODO
        this.quad.map(state.destinationFrame, state.sourceFrame);
        //传数据
        bufferManager.update(this.quad.interleavedData, BUFFER_TYPE.ARRAY_BUFFER);
        //索引
        bufferManager.update(this.quad.index, BUFFER_TYPE.ELEMENT_ARRAY_BUFFER);
        //vao绑定
        vaoManager.bind(this.quad, filter)
        //绘制
        gl.drawElements(gl.TRIANGLES, 6, gl.UNSIGNED_SHORT, 0);
    }



    /**
     * This will map the filter coord so that a texture can be used based on the transform of a sprite
     *
     * @param {Matrix} outputMatrix - The matrix to output to.
     * @param {Sprite} sprite - The sprite to map to.
     * @return {Matrix} The mapped matrix.
     */
    calculateSpriteMatrix(outputMatrix: Matrix, sprite: Sprite, state?: FilterState): Matrix {
        const { sourceFrame, destinationFrame } = this.activeState || state;
        const { orig } = sprite._texture;
        const mappedMatrix = outputMatrix.set(destinationFrame.width, 0, 0,
            destinationFrame.height, sourceFrame.x, sourceFrame.y);
        const worldTransform = Matrix.TEMP_MATRIX.copy(sprite.worldMatrix)

        worldTransform.invert();
        mappedMatrix.prepend(worldTransform);
        mappedMatrix.scale(1.0 / orig.width, 1.0 / orig.height);
        mappedMatrix.translate(sprite.anchorTexture.x, sprite.anchorTexture.y);

        return mappedMatrix;
    }
    /**
     * Empties the texture pool.
     *
     */
    emptyPool() {
        for (const i in this.pool) {
            const textures = this.pool[i];

            if (textures) {
                for (let j = 0; j < textures.length; j++) {
                    textures[j].baseTexture.destroy()
                    textures[j].destroy();
                }
            }
        }
        this.pool = {};
    }

    /**
     * Frees a render target back into the pool.
     *
     * @param {RenderTarget} renderTarget - The renderTarget to free
     */
    freePotRenderTarget(renderTarget) {
        this.pool[renderTarget.filterPoolKey].push(renderTarget);
    }

    /**
     * 渲染前检查
     */
    onPreRender() {
        if (this._screenWidth !== this.renderer.width
            || this._screenHeight !== this.renderer.height
        ) {
            this._screenWidth = this.renderer.width;
            this._screenHeight = this.renderer.height;
            const textures = this.pool[screenKey];
            //原先存的都没用了
            if (textures) {
                for (let j = 0; j < textures.length; j++) {
                    textures[j].baseTexture.destroy()
                    textures[j].destroy();
                }
            }
            this.pool[screenKey] = [];
        }
    }
    destroy() {
        const renderer = this.renderer;
        renderer.removeEventListener('onPreRender', this.onPreRender, this);
        renderer.removeEventListener('onContextChange', this.onContextChange, this);
        this.renderer = null;
        this.pool = null
    }
    private getRenderTexture(minWidth: number, minHeight: number, resolution: number = 1) {
        let key: any = screenKey;

        minWidth *= resolution;
        minHeight *= resolution;

        if (minWidth !== this._screenWidth
            || minHeight !== this._screenHeight
        ) {
            minWidth = nextPow2(minWidth);
            minHeight = nextPow2(minHeight);
            // key = ((minWidth & 0xFFFF) << 16) | (minHeight & 0xFFFF);
            key = (((minWidth & 0xFFFF) << 16) | (minHeight & 0xFFFF)) >>> 0;
        }

        if (!this.pool[key]) this.pool[key] = [];

        let renderTexture = this.pool[key].pop();


        if (!renderTexture) {
            renderTexture = new RenderTexture(new BaseRenderTexture(
                minWidth,
                minHeight
            ))
        }
        //为了回收
        renderTexture.filterPoolKey = key;
        //分辨率不知道作用
        // renderTexture.setResolution(resolution);

        return renderTexture;
    }
    private recoverRenderTexture(renderTexture: RenderTexture) {
        renderTexture.filterFrame = null;
        this.pool[renderTexture.filterPoolKey].push(renderTexture);
    }
    /**
     * @param matrix - first param
     * @param rect - second param
     */
    private transformAABB(matrix: Matrix, rect: Rectangle): void {
        const lt = tempPoints[0];
        const lb = tempPoints[1];
        const rt = tempPoints[2];
        const rb = tempPoints[3];

        lt.set(rect.left, rect.top);
        lb.set(rect.left, rect.bottom);
        rt.set(rect.right, rect.top);
        rb.set(rect.right, rect.bottom);

        matrix.transformPoint(lt.x, lt.y, lt);
        matrix.transformPoint(lb.x, lb.y, lb);
        matrix.transformPoint(rt.x, rt.y, rt);
        matrix.transformPoint(rb.x, rb.y, rb);

        const x0 = Math.min(lt.x, lb.x, rt.x, rb.x);
        const y0 = Math.min(lt.y, lb.y, rt.y, rb.y);
        const x1 = Math.max(lt.x, lb.x, rt.x, rb.x);
        const y1 = Math.max(lt.y, lb.y, rt.y, rb.y);

        rect.x = x0;
        rect.y = y0;
        rect.width = x1 - x0;
        rect.height = y1 - y0;
    }
    private roundFrame(
        frame: Rectangle,
        resolution: number,
        bindingSourceFrame: Rectangle,
        bindingDestinationFrame: Rectangle,
        transform?: Matrix
    ) {
        if (transform) {
            const { a, b, c, d } = transform;

            // Skip if skew/rotation present in matrix, except for multiple of 90° rotation. If rotation
            // is a multiple of 90°, then either pair of (b,c) or (a,d) will be (0,0).
            if ((Math.abs(b) > 1e-4 || Math.abs(c) > 1e-4)
                && (Math.abs(a) > 1e-4 || Math.abs(d) > 1e-4)) {
                return;
            }
        }

        transform = transform ? tempMatrix.copy(transform) : tempMatrix.identity();

        // Get forward transform from world space to screen space
        transform
            .translate(-bindingSourceFrame.x, -bindingSourceFrame.y)
            .scale(
                bindingDestinationFrame.width / bindingSourceFrame.width,
                bindingDestinationFrame.height / bindingSourceFrame.height)
            .translate(bindingDestinationFrame.x, bindingDestinationFrame.y);

        // Convert frame to screen space
        this.transformAABB(transform, frame);

        // Round frame in screen space TODO
        // frame.ceil(resolution);

        // Project back into world space.
        this.transformAABB(transform.invert(), frame);
    }
}
