import { BatchRenderer } from "./plugins/BatchRenderer";
import { SystemRenderer } from "./SystemRenderer";
import { RENDERER_TYPE, BLEND_MODES } from "../const";
import { TextureManager } from "./managers/TextureManager";
import { SCALE_MODES, /*devicePixelRatio*/ } from "../const";
// import RenderTexture from "../texture/RenderTexture";
import { Matrix, Transform } from "../math";
import WebGLState from "./webgl/WebGLState";
import BatchManager from "./managers/BatchManager";
import MaskManager from "./managers/MaskManager";
import StencilManager from "./managers/StencilManager";
import { DisplayObject } from "../display/DisplayObject";
import { hex2string, hex2rgb } from "../utils";
import { FilterManager } from "./managers/FilterManager";
import { ShaderManager } from "./managers/ShaderManager";
import { BufferManager } from "./managers/BufferManager";
import { FramebufferManager } from "./managers/FramebufferManager";
import { VaoManager } from "./managers/VaoManager";
import { RenderTexture } from "../texture";

let CONTEXT_UID = 0;

//渲染方式基本用了pixi的方式
export class WebglRenderer extends SystemRenderer {
    /**
     * 所有插件列表，目前只有batch
     */
    plugins = {};
    /**
     * webgl上下文
     */
    gl: WebGLRenderingContext
    /**
     * 渲染器唯一标识
     */
    CONTEXT_UID: number;
    /**
     * 遮罩管理
     */
    maskManager: MaskManager;
    /**
     * 模板管理
     */
    stencilManager: StencilManager;
    /**
     * 滤镜管理器
     */
    filterManager: FilterManager;
    /**
     * 批处理管理
     */
    batchManager: BatchManager
    /**
     * 纹理管理
     */
    textureManager: TextureManager;

    shaderManager: ShaderManager;
    bufferManager: BufferManager;
    framebufferManager: FramebufferManager;
    vaoManager: VaoManager
    /**
     * 状态机，暂时不需要切换，以后有3d再说
     */
    state: WebGLState;

    renderingToScreen: boolean = true;
    /**
     * 只传入上下文和尺寸，canvas标签在stage上处理
     * @param gl 
     * @param width 
     * @param height 
     */
    constructor(gl: WebGLRenderingContext, width: number, height: number) {
        super();
        this._instanceType = "WebglRenderer";

        this.type = RENDERER_TYPE.WEBGL;
        //修改下函数this指向，stage里直接用
        this.handleContextLost = this.handleContextLost.bind(this);
        this.handleContextRestored = this.handleContextRestored.bind(this);

        this._backgroundColorRgba[3] = this.transparent ? 0 : 1;

        // this.gl = createContext(this.htmlElement, contextOptions);
        this.gl = gl

        this.CONTEXT_UID = CONTEXT_UID++;

        this.maskManager = new MaskManager(this);
        this.stencilManager = new StencilManager(this);
        this.batchManager = new BatchManager(this)
        this.textureManager = new TextureManager(this);
        this.filterManager = new FilterManager(this);
        this.shaderManager = new ShaderManager(this);
        this.bufferManager = new BufferManager(this);
        this.framebufferManager = new FramebufferManager(this);
        this.vaoManager = new VaoManager(this);
        //初始化插件
        this.initPlugins(WebglRenderer.__plugins)

        this.state = new WebGLState(this.gl);

        this._initContext();

        this.resize(width, height);
    }

    private _initContext() {
        const gl = this.gl;
        //恢复丢失的上下文
        if (gl.isContextLost() && gl.getExtension('WEBGL_lose_context')) {
            gl.getExtension('WEBGL_lose_context').restoreContext();
        }
        this.state.resetToDefault();

        this.dispatchEvent('onContextChange');
    }

    render(displayObject: DisplayObject, renderTexture?: RenderTexture, transform?: Matrix) {
        //触发onPreRender
        this.dispatchEvent('onPreRender');
        //是否渲染到屏幕root
        this.renderingToScreen = !renderTexture;
        //如果上下文丢失
        if (!this.gl || this.gl.isContextLost()) return;

        //update更新属性
        displayObject.update()

        //更新矩阵
        const cacheParent = displayObject.parent;
        displayObject.parent = this._tempDisplayObjectParent;
        displayObject.updateTransform();
        displayObject.parent = cacheParent;

        //绑定帧缓存
        this.framebufferManager.bindRenderTexture(renderTexture, null, null, transform);

        //当前插件start
        this.batchManager.currentRenderer.start();

        //渲染对象清空画布
        this.framebufferManager.clear();

        //开始渲染所有的显示对象
        displayObject.renderWebGL(this);

        //一次性刷一次
        this.batchManager.currentRenderer.flush();

        //触发onPostRender
        this.dispatchEvent('onPostRender');
    }

    clear() {
        //重置到root
        this.framebufferManager.reset();
        this.framebufferManager.clear();
    }

    setBlendMode(blendMode: BLEND_MODES) {
        this.state.setBlendMode(blendMode);
    }

    /**
     * 传入真实的canvas尺寸
     * @param screenWidth 
     * @param screenHeight 
     */
    resize(screenWidth: number, screenHeight: number) {
        super.resize(screenWidth, screenHeight);
        //里面会根据renderer尺寸计算
        this.framebufferManager.bindRenderTexture();
    }

    /**
     * 渲染器销毁方法
     */
    destroy() {
        //插件销毁
        this.destroyPlugins();
        // remove listeners，这部分逻辑放舞台里，注意舞台先移除监听，再销毁渲染器
        // this.htmlElement.removeEventListener('webglcontextlost', this.handleContextLost);
        // this.htmlElement.removeEventListener('webglcontextrestored', this.handleContextRestored);
        //纹理管理器销毁
        this.textureManager.destroy();

        //父级销毁方法
        super.destroy();

        // 销毁所有管理器
        this.maskManager.destroy();
        this.stencilManager.destroy();
        this.filterManager.destroy();

        this.maskManager = null;
        this.filterManager = null;
        this.textureManager = null;

        this.handleContextLost = null;
        this.handleContextRestored = null;

        //淘宝小程序内会有问题，这样执行，暂时先注释，如果web环境出问题再修改
        // this.gl.useProgram(null);

        // if (this.gl.getExtension('WEBGL_lose_context')) {
        //   this.gl.getExtension('WEBGL_lose_context').loseContext();
        // }

        // this.gl = null;
    }



    /**
     * webgl上下文恢复时
     * @private
     */
    handleContextRestored() {
        // this.filterManager.destroy(true);//在_initContext的oncontextchange里处理了
        this._initContext();
        this.resize(this.width, this.height);//放在这吧,改动最小
    }

    /**
     * webgl上下文丢失时
     * @param {WebGLContextEvent} event - 事件
     */
    handleContextLost(event) {
        event.preventDefault();
    }

    /**
     * 初始化插件
     * @protected
     * @param {object} staticMap - 静态属性里存的插件列表
     */
    initPlugins(staticMap) {
        for (const o in staticMap) {
            this.plugins[o] = new (staticMap[o])(this);
        }
    }
    /**
     * 销毁插件
     */
    destroyPlugins() {
        for (const o in this.plugins) {
            this.plugins[o].destroy();
            this.plugins[o] = null;
        }
        this.plugins = null;
    }
    /**
     * 插件列表
     */
    static __plugins;
    /**
     * 添加插件，用于initPlugins时生成所有插件实例，webgl渲染器估计插件可扩展多，所以加静态方法
     * @method
     * @param {string} pluginName - 插件名，用于区分键值
     * @param {Function} ctor - 插件类
     */
    static registerPlugin(pluginName, ctor) {
        WebglRenderer.__plugins = WebglRenderer.__plugins || {};
        WebglRenderer.__plugins[pluginName] = ctor;
    }
}

//注册2d批处理插件，不放这里BatchRenderer文件里，因为不往外引，不会执行，
WebglRenderer.registerPlugin('batch', BatchRenderer);

