import { createImage, getEnv } from "../utils";

/**
 * 加载图片
 * @param url 图片路径
 * @param onSuccess 
 * @param onFail 
 * @param crossOrigin 默认anonymous
 */
export function loadImage(
    url: string,
    onSuccess: (img: HTMLImageElement) => void,
    onFail?: (err?: any) => void,
    crossOrigin: string = "anonymous",
) {
    let img = createImage();
    img.onload = function () {
        onSuccess(img);
    }
    img.onerror = function () {
        onFail && onFail();
    }
    //坑爹中文，decodeURI一次是为了以防链接被encode过一次
    url = encodeURI(decodeURI(url))
    //除了base64的路径，其他都设置了跨域，其实有部分服务器不允许跨域，不能设置anonymous
    if (url.indexOf('data:') !== 0) img.crossOrigin = crossOrigin;
    img.src = url;
}

/**
 * 加载json
 * @param {string} url 
 * @param onSuccess 
 * @param onFail 
 */
export function loadJson(
    url: string,
    onSuccess: (res: any) => void,
    onFail?: (err?: any) => void,
) {
    loadRaw(url, onSuccess, onFail, ResponseDataType.json)
}

/**
 * 加载二进制数据
 * @param {string} url  
 * @param onSuccess 
 * @param onFail 
 */
export function loadArrayBuffer(
    url: string,
    onSuccess: (res: any) => void,
    onFail?: (err?: any) => void,
) {
    loadRaw(url, onSuccess, onFail, ResponseDataType.arraybuffer)
}

/**
 * 加载文本
 * @param {string} url
 * @param onSuccess 
 * @param onFail 
 */
export function loadText(
    url: string,
    onSuccess: (res: string) => void,
    onFail?: (err?: any) => void,
) {
    loadRaw(url, onSuccess, onFail, ResponseDataType.text)
}

/**
 * 加载原始数据，需要传type，否则按照文本加载
 * @param url 
 * @param onSuccess 
 * @param onFail 
 * @param type 返回数据类型，默认text
 */
export function loadRaw(
    url: string,
    onSuccess: (res: any) => void,
    onFail?: (err?: any) => void,
    type: ResponseDataType = ResponseDataType.text,
) {
    (getEnv() == "tb" ? tbLoad : loadRawWeb)(
        url,
        onSuccess,
        onFail,
        type
    )
}

/**
 * 返回数据类型
 */
export enum ResponseDataType {
    text = "text",
    json = "json",
    arraybuffer = "arraybuffer",
}

/**
 * web环境原生加载方式
 * @param callback 
 * @param url 
 * @param type 
 */
function loadRawWeb(
    url: string,
    onSuccess: (res: any) => void,
    onFail?: (err: any) => void,
    type: ResponseDataType = ResponseDataType.text
) {
    //每次都要new
    let _req;
    if (window["XMLHttpRequest"]) {
        _req = new XMLHttpRequest();
    } else if (window["ActiveXObject"]) {
        _req = new window["ActiveXObject"]();
    } else {
        onFail && onFail("your browser doesnt support XMLHttpRequest!");
        return
    }
    if (_req != null) {
        _req.open("GET", url, true);
        _req.responseType = type;
        _req.send();
        _req.onreadystatechange = () => {
            // if (_req.readyState == 4 && _req.status == 200) {
            //     callback(true, _req.response)
            // }
            if (_req.readyState == 4) {
                var status = _req.status;
                if ((status >= 200 && status < 300) ||//2XX表示有效响应
                    status == 304//304意味着是从缓存读取
                ) {
                    onSuccess(_req.response)
                } else {
                    onFail && onFail("request.status:" + status);
                }
            }
        };
        // _req.onerror = (reason): void => {
        //     callback(false, reason)
        // }
    }
}
/////////淘宝小程序相关
/**
 * 获取临时路径
 * @param url cloud路径
 * @param callback 
 */
function getTbTempUrl(
    url: string,
    onSuccess: (src: string) => void,
    onFail?: (err: any) => void
) {
    //@ts-ignore
    const { cloud } = getApp();
    //获取临时地址
    cloud.file.getTempFileURL({ fileId: [url] }).then((urls) => {
        onSuccess(urls[0].url.replace('-internal', ''))
    }).catch((err) => {
        //淘宝上的问题，直接打印吧
        // console.error(err)
        onFail && onFail(err);
    })
}


function downloadReadFile(
    url: string,
    onSuccess: (res: any) => void,
    onFail?: (err: any) => void,
    type: ResponseDataType = ResponseDataType.text//"utf8" | "ArrayBuffer" = "ArrayBuffer"
) {
    //@ts-ignore
    let tbMy = my;
    tbMy.downloadFile({
        url: url,
        success(res) {
            var i = res.apFilePath;//临时地址是否有必要缓存下，如果读不到再考虑下载。
            tbMy.getFileSystemManager().readFile({//"utf8" | "ArrayBuffer"是文档给的类型，但是encoding不传才是ArrayBuffer
                filePath: i,
                encoding: type === ResponseDataType.arraybuffer ? "" : "utf8",//不加表示加载的是ArrayBuffer
                success: function (r) {
                    //是json的需要反序列化一下
                    onSuccess(type == ResponseDataType.json ? JSON.parse(r.data) : r.data)//注意是r.data
                    // actions.load_viaProto(r.data, cb, failure);
                },
                fail: function (res) {
                    onFail && onFail(res);
                }
            })
        },
        fail(res) {
            onFail && onFail(res);
        },
    });
}

function tbLoad(
    url: string,
    onSuccess: (res?: any) => void,
    onFail?: (err: any) => void,
    type: ResponseDataType = ResponseDataType.text//"utf8" | "ArrayBuffer" = "ArrayBuffer"
) {
    //类似这种地址"cloud://A8673B47AAA58993A24A6718E203B967//dice.svga"
    if (url.indexOf("cloud://") == 0) {
        getTbTempUrl(
            url,
            (src) => {
                downloadReadFile(src, onSuccess, onFail, type)
            },
            onFail
        )
    } else {
        downloadReadFile(url, onSuccess, onFail, type)
    }
}