import Container from "./Container";
import { RENDERER_TYPE, VERSION/*, osType*/ } from "../const"
// import SystemRenderer from "../renderers/SystemRenderer";
import { Rectangle, Point } from "../math";
import { EventDispatcher } from "../events/EventDispatcher";
import { Event } from "../events/Event";
// import { FloatDisplay } from "./FloatDisplay";
import { DisplayObject } from "./DisplayObject";
import { MouseEvent } from "../events/MouseEvent";

import { WebglRenderer } from "../renderers/WebglRenderer";

import { CanvasRenderer } from "../renderers/CanvasRenderer";
import { SystemRenderer } from "../renderers/SystemRenderer";
import { getEnv, getOsType, isWebGLSupported } from "../utils";
import { createWebglContext } from "../renderers/webgl/createWebglContext";

export class Stage extends Container {

    /**
     * stage使用的渲染器
     * @property renderObj
     * @public
     * @since 1.0.0
     * @type {SystemRenderer}
     * @default null
     */
    public renderObj: SystemRenderer = null;

    /**
     * 相对于stage的可见区域，考虑不对外开放，但是有些地方又有用，比如某点是否在舞台可见区域内viewRect.isPointIn(gp)
     * @property viewRect
     * @public
     * @since 1.0.0
     * @type {Rectangle}
     * @default 
     * @readonly
     */
    public viewRect: Rectangle = new Rectangle();

    /**
     * 舞台设计尺寸，一般为设计稿内容尺寸
     * @property desWidth
     * @public
     * @since 1.0.0
     * @default 0
     * @type {number}
     * @readonly
     */
    public desWidth: number = 0;
    /**
     * 舞台设计尺寸，一般为设计稿内容尺寸
     * @property desHeight
     * @public
     * @since 1.0.0
     * @default 0
     * @type {number}
     * @readonly
     */
    public desHeight: number = 0;
    /**
     * 舞台在设备中的显示尺寸
     * @property divHeight
     * @public
     * @since 1.0.0
     * @default 0
     * @type {number}
     * @readonly
     */
    public divHeight: number = 0;
    /**
     * 舞台在设备中的显示尺寸
     * @property divWidth
     * @public
     * @since 1.0.0
     * @default 0
     * @readonly
     * @type {number}
     */
    public divWidth: number = 0;

    private _bgColor: number = 0x000000;
    /**
     * 舞台的背景色，暂时无效，渲染器默认透明背景，且webgl模式下clearColor不会跟着修改，TODO 以后修改
     * @property bgColor
     * @public
     * @since 1.0.0
     * @type {number}
     * @default 0;
     */
    public get bgColor(): number {
        return this._bgColor;
    }
    public set bgColor(value: number) {
        if (this._bgColor === value) return
        this._bgColor = value;
        this.renderObj.backgroundColor = value;
    }

    /**
     * 上一次鼠标或触碰经过的显示对象列表，为了over和out事件
     * @property _lastDpList
     * @type {Object}
     * @private
     */
    private _lastDpList: any = {};

    /**
     * 计算画布实际尺寸/显示尺寸所得或构造函数传入设置画布实际尺寸
     */
    private _dpi: number;

    /**
     * 记录下画布对象
     */
    private canvas: HTMLCanvasElement;

    /**
     * 淘宝小程序环境canvas的偏移,淘宝环境才用,web环境实时,字段保留,但是不会计算了
     */
    private canvasOffsetTb: { x: number, y: number } = { x: 0, y: 0 };
    // private offsetTimeId: any

    get stageWidth() {
        return this.viewRect.width;
    }

    get stageHeight() {
        return this.viewRect.height;
    }

    private _autoSteering: boolean = false;
    /**
     * 是否自动转向，一般对于需要手机横屏自动更新内容朝向的设置为true，自动对齐设计尺寸和屏幕显示尺寸
     * 默认false
     * @property autoSteering
     * @public
     * @type {boolean}
     * @default false
     */
    get autoSteering() {
        return this._autoSteering;
    }
    set autoSteering(value: boolean) {
        if (this._autoSteering == value) return;
        this._autoSteering = value;
        //没啥性能问题，设置了就执行一次
        this._setAlign();
    }

    private _scaleMode: "fixedWidth" | "fixedHeight";
    /**
     * 缩放模式，暂时只有 "fixedWidth" | "fixedHeight"，
     * 以后有需要再加，同时修改构造函数传参fixedHeight
     */
    get scaleMode(): "fixedWidth" | "fixedHeight" {
        return this._scaleMode
    }
    set scaleMode(value: "fixedWidth" | "fixedHeight") {
        if (this._scaleMode == value) return;
        this._scaleMode = value;
        //没啥性能问题，设置了就执行一次
        this._setAlign();
    }
    /**
     * 舞台是否居中，
     * 不居中则以左上角为00点，
     * 否则按照scaleMode均匀裁切上下或左右，用viewRect.xy获取实际裁剪数值
     * 暂时外部没有需求，不开放
     */
    private _stageCenter: boolean;
    /**
     * 建立舞台
     * 注意淘宝小程序canvas实际宽高/显示宽高必须是dpr，要么传dpr（参数resolution），要么外部自行设置实际宽高（canvas.width）及显示宽高（样式）
     * @param canvas canvas标签
     * @param desWidth 舞台设计宽，为了直接在舞台加东西不需要考虑缩放
     * @param desHeight 舞台设计高，为了直接在舞台加东西不需要考虑缩放
     * @param divWidth canvas显示宽，或视图宽度，或样式宽度，全屏就是屏幕宽度document.body.clientWidth(web环境)或my.getSystemInfoSync().windowWidth(淘宝小程序)
     * @param divHeight canvas显示高，或视图高度，或样式高度，全屏就是屏幕宽度document.body.clientHeight(web环境)或my.getSystemInfoSync().windowHeight(淘宝小程序)
     * @param renderType 渲染类型,默认canvas
     * @param stageCenter 舞台是否根据设计尺寸居中,默认false(左上角置顶)
     * @param fixedHeight 是否定高,默认false(定宽)
     * @param resolution 分辨率，一般传设备的devicePixelRatio，不传意味着外部自行处理了canvas的实际宽高及显示宽高（注意淘宝小程序实际宽高/显示宽高必须是dpr，要么传dpr，要么外部设置）
     * @param webglOptions 获取webgl上下文时需要的参数，一般用默认的，默认alpha:true,antialias:false,premultipliedAlpha:true,stencil:true,preserveDrawingBuffer:false
     */
    public constructor(
        canvas: HTMLCanvasElement,
        desWidth: number = 750,
        desHeight: number = 1624,
        divWidth: number,
        divHeight: number,
        renderType: RENDERER_TYPE = RENDERER_TYPE.CANVAS,
        stageCenter: boolean = false,
        fixedHeight: boolean = false,
        resolution?: number,
        webglOptions?: WebGLContextAttributes
    ) {
        super();
        this.canvas = canvas;
        let s: Stage = this;
        this._instanceType = "Stage";
        s.stage = this;
        s.name = "stageInstance_" + s.instanceId;
        s.desWidth = desWidth;
        s.desHeight = desHeight;
        s.divWidth = divWidth;
        s.divHeight = divHeight;

        if (!resolution) {//为了兼容以前的活动，以后删了
            //以前的活动外部设置了canvas的实际尺寸和显示尺寸
            console.warn("try to send param resolution, canvas‘s width and height will be setted up based on resolution");
            //这里设置_dpi
            s._dpi = canvas.width / divWidth;
        } else {
            //canvas实际尺寸处理
            s._dpi = resolution;
            var cWidth = divWidth * resolution, cHeight = divHeight * resolution;
            if (getEnv() == "tb") {//淘宝小程序环境有个白边，所以加点像素
                cWidth += resolution;
                cHeight += resolution;
            }
            canvas.width = cWidth;
            canvas.height = cHeight;
        }


        // console.log("%c ", "background: url(http://5b0988e595225.cdn.sohucs.com/images/20180315/d41842ad9b5443d3854a480ea49f3b09.gif) no-repeat center;padding-left:80px;padding-bottom: 80px;border-radius:50%;")
        //打印个版本号
        console.log(
            "%cfyge version:" + VERSION,
            "text-shadow: 0 1px 0 #ccc,0 2px 0 #c9c9c9,0 3px 0 #bbb,0 4px 0 #b9b9b9,0 5px 0 #aaa,0 6px 1px rgba(0,0,0,.1),0 0 5px rgba(0,0,0,.1),0 1px 3px rgba(0,0,0,.3),0 3px 5px rgba(0,0,0,.2),0 5px 10px rgba(0,0,0,.25),0 10px 10px rgba(0,0,0,.2),0 20px 20px rgba(0,0,0,.15);font-size:3em"
        )
        //webgl不支持时的兼容，回退canvas
        if (renderType == RENDERER_TYPE.WEBGL && !isWebGLSupported()) {
            renderType = RENDERER_TYPE.CANVAS;
            console.warn("your device doesn‘t support webgl,will use canvas2d instead")
        }
        if (renderType == RENDERER_TYPE.CANVAS) {
            let context = canvas.getContext("2d");
            s.renderObj = new CanvasRenderer(context, canvas.width, canvas.height);
        } else {
            let contextOptions: WebGLContextAttributes = {
                alpha: true,
                antialias: false,//抗锯齿尽量别开，耗性能
                premultipliedAlpha: true,  //一般png图片都不会预乘alpha，所以必为true,除非有些图集工具选择了premultipliedAlpha
                stencil: true,
                // preserveDrawingBuffer: true,//需要toDataUrl时要开，不开的话渲染一次后马上toDataUrl也能截到
                // powerPreference: this.options.powerPreference,
                ...webglOptions
            };
            let context = createWebglContext(canvas, contextOptions);
            s.renderObj = new WebglRenderer(context, canvas.width, canvas.height);
            //监听下上下文丢失的情况
            if (getEnv() == "web") {
                canvas.addEventListener(
                    'webglcontextlost',
                    (s.renderObj as WebglRenderer).handleContextLost,
                    false
                );
                canvas.addEventListener(
                    'webglcontextrestored',
                    (s.renderObj as WebglRenderer).handleContextRestored,
                    false
                );
            }

        }
        //TODO,还有一种方式,传入分辨率和显示宽高,canvas尺寸在里面修改,这样要多传个参数,暂时不改传参,不然版本问题,
        //到时修改这个放到上面,因为渲染器也用到了canvas的尺寸,
        //注意淘宝小程序环境必须先设置canvas的尺寸再getContext,类似CanvasRenderTarget构造函数里的问题
        //分辨率,
        // s._dpi = canvas.width / divWidth;//这里不需要了
        s._stageCenter = stageCenter;
        s._scaleMode = fixedHeight ? "fixedHeight" : "fixedWidth";
        //设置缩放及视窗
        s._setAlign();

        //延时派发舞台初始化事件
        setTimeout(function () {
            //派发事件
            s.dispatchEvent(Event.INIT_STAGE);
        }, 100);

        //淘宝环境额外处理canvas偏移,到时需要修改id传入，再说把，不需要了，淘宝上返回的就是canvas上的显示坐标
        // if (getEnv() == "tb") {//暂时先去掉吧，会出现一些问题，有需要，手动设置canvasOffsetTb的xy，屏幕坐标
        //     s.offsetTimeId = setInterval(() => {
        //         //@ts-ignore
        //         my.createSelectorQuery().select('#canvas').boundingClientRect().exec((r) => {
        //             if (r && r[0]) {
        //                 s.canvasOffsetTb.x = r[0].left || 0;
        //                 s.canvasOffsetTb.y = r[0].top || 0;
        //             }
        //         });
        //     }, 200)
        // }
    }
    /**
     * 处理适配，暂时不存在外部调用的情况
     */
    private _setAlign() {
        let s = this;
        let canvas = s.canvas;
        //实际尺寸和设计内容尺寸
        let divW = canvas.width,
            divH = canvas.height,
            desW = s.desWidth,
            desH = s.desHeight;
        //舞台设置居中处理锚点和位置偏移
        if (s._stageCenter) {
            s.anchorX = desW >> 1;
            s.anchorY = desH >> 1;
            s.x = (divW - desW) >> 1;
            s.y = (divH - desH) >> 1;
        }

        //设备是否为竖屏
        let isDivH = divH >= divW;
        //内容是否为竖屏内容
        let isDesH = desH >= desW;
        //处理下,显示尺寸宽高颠倒下
        if (s._autoSteering && isDesH != isDivH) {
            let d = divH;
            divH = divW;
            divW = d;
        }
        //默认定宽
        var scale = s._scaleMode == "fixedHeight" ? divH / desH : divW / desW;
        s.scaleX = scale;
        s.scaleY = scale;
        //未设置自动转向，或不需要转向
        if (!s._autoSteering || isDesH == isDivH) {
            s.rotation = 0;
            //没设置过居中的位移置0
            if (!s._stageCenter) s.x = s.y = 0
        } else {
            //TODO,待考虑按什么决定角度正负
            if (desH > desW) {
                s.rotation = -90;
                //没设置过居中的
                if (!s._stageCenter) s.y = divW
            } else {
                s.rotation = 90;
                //没设置过居中的
                if (!s._stageCenter) s.x = divH
            }
        }
        //视窗计算
        s.viewRect = s.viewRect || new Rectangle();
        s.viewRect.x = (desW - divW / scale) >> 1;
        s.viewRect.y = (desH - divH / scale) >> 1;
        s.viewRect.width = desW - s.viewRect.x * 2;
        s.viewRect.height = desH - s.viewRect.y * 2;

        //额外逻辑,不设置舞台居中时,x,y为左上角,置0
        if (!s._stageCenter) s.viewRect.x = s.viewRect.y = 0;
    }
    /**
     * 移动端不常用
     * 微信浏览器使用情景:ios返回页面下面出现操作栏
     * 这里会按照原_dpi修改canvas的实际尺寸
     * 一般设备的dpi不会改变,
     * web全屏环境可不传参数,否则自行计算显示尺寸传入
     * @param divWidth 
     * @param divHeight 
     * @example
     *   window.addEventListener('resize', () => {stage.resize()});
     */
    public resize(divWidth?: number, divHeight?: number) {
        if (getEnv() == "web") {//web环境不传当作全屏处理
            divWidth = divWidth || document.body.clientWidth;
            divHeight = divHeight || document.body.clientHeight;
        }
        if (!divWidth || !divWidth) {
            console.error("parm divWidth or divHeight cannot be empty")
            return
        }
        let s = this, d = s._dpi, c = s.canvas;
        //如果一致不修改
        if (divWidth == s.divWidth && divHeight == s.divHeight) return;
        //这里修改canvas尺寸
        if (divWidth != s.divWidth) c.width = divWidth * d;
        if (divHeight != s.divHeight) c.height = divHeight * d;
        //赋值显示宽高
        s.divWidth = divWidth;
        s.divHeight = divHeight;
        //修改缩放显示及视窗
        s._setAlign();
        //渲染器修改尺寸
        s.renderObj.resize(c.width, c.height);
        //派发事件
        s.dispatchEvent(Event.RESIZE);
    }

    /**
     * 鼠标事件MouseEvent对象池
     * @property _ml
     * @type {Array}
     * @private
     */
    private _ml: MouseEvent[] = [];
    /**
     * 鼠标事件中用到的Point对象池
     * @property _mp
     * @type {Array}
     * @private
     */
    private _mp: Point[] = [];

    /**
     * 初始化鼠标事件
     * @method _initMouseEvent
     * @private
     */
    private _initMouseEvent(event: MouseEvent, cp: Point, sp: Point, identifier: number): void {
        event["_pd"] = false;
        event.clientX = cp.x;
        event.clientY = cp.y;
        event.stageX = sp.x;
        event.stageY = sp.y;
        event.identifier = identifier;
    }

    /**
     * 每一个手指事件的对象池，为了click事件，得保证down和up是同一个手指标识
     * @property _mouseDownPoint
     * @type {Object}
     * @private
     */
    private _mouseDownPoint: any = {};

    /**
     * 渲染器刷新stage
     * @method flush
     * @private
     * @return {void}
     */
    public flush(): void {
        this.renderObj.render(this);
    }

    /**
     * 原始（html或其他）的鼠标事件名和引擎鼠标事件名的映射，TODO，字符串统统用MouseEvent的静态属性代替，事件名只维护一个地方
     * touchcancel:"onMouseUp"不常用，先不加
     * @property _mouseEventTypes
     * @type {{mousedown: string, mouseup: string, mousemove: string, touchstart: string, touchmove: string, touchend: string}}
     * @private
     */
    private _mouseEventTypes: any = {
        //pc
        mousedown: "onMouseDown",
        mousemove: "onMouseMove",
        mouseup: "onMouseUp",
        //mobile
        touchstart: "onMouseDown",
        touchmove: "onMouseMove",
        touchend: "onMouseUp",
        //tbMini
        touchStart: "onMouseDown",//小程序返回的事件名是驼峰的
        touchMove: "onMouseMove",
        touchEnd: "onMouseUp"
    };
    /**
     * 舞台的鼠标事件是否要阻止默认事件，默认false
     * 对于同时写了监听pc和移动端的，
     */
    public webMouseEventPreventDefault: boolean;
    /**
     * 舞台的鼠标事件是否要阻止事件冒泡，默认false
     */
    public webMouseEventStopPropagation: boolean;
    /**
     * 鼠标事件方法
     * @method onMouseEvent
     * @param e
     * @private
     */
    onMouseEvent(e: any): void {
        //如果是web环境，且设置了阻止默认事件
        if (getEnv() == "web") {
            if (this.webMouseEventPreventDefault) e.preventDefault();
            if (this.webMouseEventStopPropagation) e.stopPropagation();
        }
        let s: Stage = this;
        //检查mouse或touch事件是否有，如果有的话，就触发事件函数
        if (EventDispatcher._totalMEC > 0) {
            let points: any;
            //事件类型

            let item = s._mouseEventTypes[e.type];
            let events: any;
            let event: any;
            //stageMousePoint
            let sp: Point;
            //localPoint;
            let lp: Point;
            //clientPoint
            let cp: Point;
            //事件个数
            let eLen: number;
            let identifier: any;
            if (getOsType() == "pc") {
                e.identifier = 0;
                points = [e];
            } else {
                // points = [e.changedTouches[0]];
                points = e.changedTouches;//都用上
            }
            // points = [e.changedTouches[0]];
            //  points = [e.touches[0]];//不能用这个
            let { x: offsetX, y: offsetY } = s.canvasOffsetTb;
            //
            if (getEnv() == "web") {
                let doc = document.documentElement;
                let box = s.canvas.getBoundingClientRect();
                offsetX = box.left + window.pageXOffset - doc.clientLeft;
                offsetY = box.top + window.pageYOffset - doc.clientTop;
            }


            for (let o = 0; o < points.length; o++) {
                if (!points[o]) continue;
                eLen = 0;
                events = [];
                identifier = "m" + points[o].identifier;
                if (s._mp.length > 0) {
                    cp = s._mp.shift();
                } else {
                    cp = new Point();
                }
                cp.x = ((points[o].pageX || points[o].x || points[o].b || 0) - offsetX) * s._dpi// devicePixelRatio;
                cp.y = ((points[o].pageY || points[o].y || points[o].c || 0) - offsetY) * s._dpi// devicePixelRatio;
                // my.alert({
                //   title: JSON.stringify(points[o])
                // });
                //计算舞台中的点
                sp = s.globalToLocal(cp, DisplayObject._bp);
                //检查是否有鼠标事件
                if (EventDispatcher.getMouseEventCount() > 0) {
                    if (!s._ml[eLen]) {
                        event = new MouseEvent(item);
                        s._ml[eLen] = event;
                    } else {
                        event = s._ml[eLen];
                        event.type = item;
                    }
                    events[events.length] = event;
                    s._initMouseEvent(event, cp, sp, identifier);
                    eLen++;
                }
                if (item == "onMouseDown") {
                    s._mouseDownPoint[identifier] = cp;
                    //清空上次存在的显示列表
                } else if (item == "onMouseUp") {
                    if (s._mouseDownPoint[identifier]) {
                        if (Point.distance(s._mouseDownPoint[identifier], cp) < 20) {
                            //检查是否有添加对应的click事件
                            if (EventDispatcher.getMouseEventCount("onMouseClick") > 0) {
                                if (!s._ml[eLen]) {
                                    event = new MouseEvent("onMouseClick");
                                    s._ml[eLen] = event;
                                } else {
                                    event = s._ml[eLen];
                                    event.type = "onMouseClick";
                                }
                                events[events.length] = event;
                                s._initMouseEvent(event, cp, sp, identifier);
                                eLen++;
                            }
                        }
                    }
                }
                if (eLen > 0) {
                    //有事件开始遍历显示列表
                    //找出最底层的显示对象
                    let d: DisplayObject = s.hitTestPoint(cp, true);
                    // console.log(d)
                    let displayList: Array<DisplayObject> = [];
                    // my.alert({
                    //   title: '55729:' + d.instanceId
                    // });
                    if (d) {
                        //证明有点击到事件,然后从最底层追上来,看看一路是否有人添加过mouse或touch事件,还要考虑mousechildren和阻止事件方法
                        //找出真正的target,因为有些父级可能会mouseChildren=false;
                        while (d) {
                            if (d["mouseChildren"] === false) {
                                //丢掉之前的层级,因为根本没用了
                                displayList.length = 0;
                            }
                            displayList[displayList.length] = d;
                            d = d.parent;
                        }
                    } else {
                        displayList[displayList.length] = s;
                    }
                    let len: number = displayList.length;
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[0];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j]);
                                }
                            }
                        }
                    }
                    //这里一定要反转一下，因为会影响mouseOut mouseOver
                    displayList.reverse();
                    for (let i = len - 1; i >= 0; i--) {
                        d = displayList[i];
                        for (let j = 0; j < eLen; j++) {
                            if (!events[j]["_pd"]) {
                                //有事件，且mouseEnable为true
                                if (d.hasEventListener(events[j].type, false) && d.mouseEnable) {
                                    events[j].target = d;
                                    events[j].currentTarget = displayList[eLen - 1];
                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                    events[j].localX = lp.x;
                                    events[j].localY = lp.y;
                                    d.dispatchEvent(events[j], null, false);
                                }
                            }
                        }
                    }
                    //最后要和上一次的遍历者对比下，如果不相同则要触发onMouseOver和onMouseOut
                    if (item != "onMouseDown") {
                        if (EventDispatcher.getMouseEventCount("onMouseOver") > 0 || EventDispatcher.getMouseEventCount("onMouseOut") > 0) {
                            if (s._lastDpList[identifier]) {
                                //从第二个开始，因为第一个对象始终是stage顶级对象
                                let len1 = s._lastDpList[identifier].length;
                                let len2 = displayList.length;
                                len = len1 > len2 ? len1 : len2;
                                let isDiff = false;
                                let overEvent: MouseEvent;
                                let outEvent: MouseEvent;
                                for (let i = 1; i < len; i++) {
                                    if (!isDiff) {
                                        if (s._lastDpList[identifier][i] != displayList[i]) {
                                            //确定哪些有onMouseOver,哪些有onMouseOut
                                            isDiff = true;
                                            if (!s._ml[eLen]) {
                                                overEvent = new MouseEvent("onMouseOver");
                                                s._ml[eLen] = overEvent;
                                            } else {
                                                overEvent = s._ml[eLen];
                                                overEvent.type = "onMouseOver";
                                            }
                                            s._initMouseEvent(overEvent, cp, sp, identifier);
                                            eLen++;
                                            if (!s._ml[eLen]) {
                                                outEvent = new MouseEvent("onMouseOut");
                                                s._ml[eLen] = outEvent;
                                            } else {
                                                outEvent = s._ml[eLen];
                                                outEvent.type = "onMouseOut";
                                            }
                                            s._initMouseEvent(outEvent, cp, sp, identifier);
                                        }
                                    }
                                    if (isDiff) {
                                        if (s._lastDpList[identifier][i]) {
                                            //触发onMouseOut事件
                                            if (!outEvent["_pd"]) {
                                                d = s._lastDpList[identifier][i];
                                                if (d.hasEventListener("onMouseOut")) {
                                                    outEvent.currentTarget = s._lastDpList[identifier][len1 - 1];
                                                    outEvent.target = d;
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    outEvent.localX = lp.x;
                                                    outEvent.localY = lp.y;
                                                    d.dispatchEvent(outEvent);
                                                }
                                            }
                                        }
                                        if (displayList[i]) {
                                            //触发onMouseOver事件
                                            if (!overEvent["_pd"]) {
                                                d = displayList[i];
                                                if (d.hasEventListener("onMouseOver")) {
                                                    overEvent.currentTarget = displayList[len2 - 1];
                                                    overEvent.target = d;
                                                    lp = d.globalToLocal(cp, DisplayObject._bp);
                                                    overEvent.localX = lp.x;
                                                    overEvent.localY = lp.y;
                                                    d.dispatchEvent(overEvent);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        s._mp[s._mp.length] = cp;
                    }
                    if (item == "onMouseUp") {
                        delete s._mouseDownPoint[identifier];
                        delete s._lastDpList[identifier];
                    } else {
                        s._lastDpList[identifier] = displayList;
                    }
                }
            }
        }
    };
    /**
     * 给舞台绑定canvas标签的鼠标代理事件（web环境下需要显示对象的鼠标事件时调用）
     * 自行调用，内部不主动调用，因为比如淘宝小程序环境方式不一致，需要用onMouseEvent
     */
    addWebMouseEvent() {
        var canvas = this.canvas;
        //有就可以执行
        if (!canvas || !canvas.addEventListener) return;
        var mouseEvent = this.onMouseEvent.bind(this);
        if (getOsType() == "pc") {
            canvas.addEventListener("mousedown", mouseEvent, false);
            canvas.addEventListener('mousemove', mouseEvent, false);
            canvas.addEventListener('mouseup', mouseEvent, false);
        } else {
            canvas.addEventListener("touchstart", mouseEvent, false);
            canvas.addEventListener('touchmove', mouseEvent, false);
            canvas.addEventListener('touchend', mouseEvent, false);
        }
    }
    // /**
    //  * 直接修改了，用视窗，
    //  * 一般用于滤镜等等，对于舞台，超出viewPort的肯定不显示，没必要测量（后续考虑也测量，万一测量出的更小呢）
    //  */
    // public getBounds(): Rectangle {20211202注释，因为滤镜上会有问题，stage是带矩阵的，viewRect没经过换算
    //     return this.viewRect;
    // }

    public destroy(): void {
        let s = this;
        // if (getEnv() == "tb") clearInterval(s.offsetTimeId);
        //web环境下，渲染方式webgl模式时，移除webgl上下文丢失事件
        if (getEnv() == "web" && s.renderObj.type == RENDERER_TYPE.WEBGL) {
            s.canvas.removeEventListener(
                'webglcontextlost',
                (s.renderObj as WebglRenderer).handleContextLost,
                false
            );
            s.canvas.removeEventListener(
                'webglcontextrestored',
                (s.renderObj as WebglRenderer).handleContextRestored,
                false
            );
        }
        //然后销毁渲染器
        s.renderObj.destroy();
        s.renderObj = null;
        s.viewRect = null;
        s._lastDpList = null;
        s._ml = null;
        super.destroy();
    }
}