import { Point, ObservablePoint, Rectangle, GroupD8, Matrix } from '../math';
import { sign, TextureCache, hex2rgb, getBackupCanvasCtx } from '../utils';
import { BLEND_MODES } from '../const';
import { Texture } from '../texture/Texture';
import Container from './Container';

import { DisplayObject } from "./DisplayObject";
import { CanvasRenderer } from '../renderers/CanvasRenderer';
import { WebglRenderer } from '../renderers/WebglRenderer';

const indices = new Uint16Array([0, 1, 2, 0, 2, 3]);
//临时矩阵
let tempMatrix: Matrix;
/**
 * Sprite类，显示图片和容器功能
 * @class
 * @extends Container
 */
export default class Sprite extends Container {
    /**
     * 混色模式
     * @default BLEND_MODES.NORMAL
     */
    _blendMode: BLEND_MODES = BLEND_MODES.NORMAL;
    //以后修改
    get blendMode(): BLEND_MODES {
        return this._blendMode
    }
    /**
     * 很多效果暂时无效，再查原因，先不能设置吧
     */
    set blendMode(value: BLEND_MODES) {
        // if (value != this._blendMode) this._blendMode = value
    }

    private _anchorTexture: ObservablePoint
    /**
     * 关于贴图的锚点，0到1，默认为texture自己的
     * 0，0标识左上角，0.5，0.5表示中间，1，1表示右下角
     * @member {ObservablePoint}
     */
    get anchorTexture(): ObservablePoint {
        return this._anchorTexture;
    }
    /**
     * 设置贴图锚点，copy方法，只要value存在x、y字段就行
     */
    set anchorTexture(value: { x: number, y: number }) {
        this._anchorTexture.copy(value);
    }
    /**
     * 色值调色
     */
    private _tint: number;
    /**
     * 获取调色
     */
    get tint(): number {
        return this._tint;
    }
    /**
     * 设置调色
     */
    set tint(value: number) {
        if (value === this._tint) return;
        this._tint = value;
        this._tintRGB = (value >> 16) + (value & 0xff00) + ((value & 0xff) << 16);
    }
    /**
     * RGB形式色值，webgl用
     */
    _tintRGB: number;
    /**
     * 和_tint比较用，用于canvas调色缓存
     */
    _cachedTint: number = 0xFFFFFF;
    /**
     * 使用的贴图
     * @member {Texture}
     */
    _texture: Texture;
    /**
     * 获取纹理
     * @member {Texture}
     */
    get texture(): Texture {
        return this._texture === Texture.EMPTY ? null : this._texture;//考虑是否返回null
    }
    /**
     * 设置纹理
     */
    set texture(value: Texture) {
        if (this._texture === value || this.destroyed) {//如果已经被销毁了,不用再执行下面,
            return;
        }
        //如果存在原贴图，且还没加载好，移除监听_onTextureUpdate；
        // if (this._texture != Texture.EMPTY &&
        //     this._texture != null &&
        //     !this._texture.baseTexture.hasLoaded) this._texture.removeEventListener('update', this._onTextureUpdate, this);

        //原来的纹理，其实直接全都要移除，
        if (this._texture) {
            this._texture.removeEventListener('update', this._onTextureUpdate, this);
        }

        //赋值
        this._texture = value || Texture.EMPTY;

        this._textureID = -1;
        this._textureTrimmedID = -1;
        this._cachedTint = 0xFFFFFF;

        if (value) {
            if (value.baseTexture.hasLoaded) {
                this._onTextureUpdate();
            }
            // else {
            //     value.once('update', this._onTextureUpdate, this);//只会监听一次
            // }
            //都监听下，其实没加载的纹理图片纹理确实只需要监听一次，这里先不管了，存在性能问题，当一个纹理被大量对象使用时，事件数组会很大，导致遍历性能，TODO
            value.addEventListener('update', this._onTextureUpdate, this);
        } else {
            //624修改。如果置空纹理，_localBoundsSelf置空
            // this.updateLocalBoundsSelf();
            //20211201修改，这样修改如果设置过width和height的，内部会通过设置scale保留，从而依旧能点击到
            this._onTextureUpdate()
        }
    }
    /**
     * 初始化为texture的宽度
     */
    _width: number;
    /**
     * 重写Container父类
     * 获取texture的宽度和缩放乘积
     * @member {number}
     */
    get width(): number {
        return Math.abs(this.scale.x) * this._texture.orig.width;
    }
    /**
     * 重写父级
     * 根据纹理宽度设置自身缩放x到设置的数值
     */
    set width(value: number) {
        const s = sign(this.scale.x) || 1;
        this.scale.x = s * value / this._texture.orig.width;
        this._width = value;
    }
    /**
     * 初始化为texture的宽度
     */
    _height: number;
    /**
     * 获取texture的高度和缩放乘积
     * @member {number}
     */
    get height(): number {
        return Math.abs(this.scale.y) * this._texture.orig.height;
    }
    /**
     * 根据纹理高度设置自身缩放y到设置的数值
     */
    set height(value: number) {
        const s = sign(this.scale.y) || 1;
        this.scale.y = s * value / this._texture.orig.height;
        this._height = value;
    }
    /**
     * 标记更新transform
     */
    _transformID: number = -1;
    /**
     * 标记更新过texture
     */
    _textureID: number = -1;
    _transformTrimmedID: number = -1;
    _textureTrimmedID: number = -1;

    /**
     * 顶点索引，下面几个均为webgl用到
     */
    _indices: Uint16Array = indices;
    // size: number;
    // start: number;
    _uvs: Float32Array = null;
    /**
     * 顶点数据，长度8
     */
    _vertexData: Float32Array = new Float32Array(8);
    /**
     * Trimmed的顶点数据，用于裁剪掉过透明像素的texture
     */
    _vertexTrimmedData: Float32Array = null;

    /**
     * @param {Texture} texture
     */
    constructor(texture?: Texture) {
        super();
        this._instanceType = "Sprite";
        this._anchorTexture = new ObservablePoint(
            this._onAnchorUpdate,
            this,
            (texture ? texture.defaultAnchor.x : 0),
            (texture ? texture.defaultAnchor.y : 0)
        );

        this._width = 0;
        this._height = 0;

        this._tint = null;
        this._tintRGB = null;
        this.tint = 0xFFFFFF;

        this._texture = null;
        this.texture = texture || Texture.EMPTY;
    }

    /**
     * texture更新时触发
     * @private
     */
    protected _onTextureUpdate() {
        //保证顶点要更新
        this._textureID = -1;
        this._textureTrimmedID = -1;
        this._cachedTint = 0xFFFFFF;
        //可用才赋值uv
        if (this._texture.valid) this._uvs = this._texture._uvs.uvsFloat32;
        //设置过宽高的话，就需要改变缩放值
        // so if _width is 0 then width was not set..
        if (this._width) {
            this.scale.x = sign(this.scale.x) * this._width / this._texture.orig.width;
        }

        if (this._height) {
            this.scale.y = sign(this.scale.y) * this._height / this._texture.orig.height;
        }

        //修改_localBoundsSelf
        this.updateLocalBoundsSelf();

        //包围盒id
        this._boundsID++;
    }

    /**
     * 当贴图锚点修改时
     * 
     * @private
     */
    private _onAnchorUpdate() {
        this._transformID = -1;
        this._transformTrimmedID = -1;
        //贴图锚点修改，也要修改自身盒子，因为鼠标点击有用到
        this.updateLocalBoundsSelf();
    }

    /**
     * 通过自身贴图修改自身盒子
     * 子类可修改，比如文本不需要padding的话，暂时还没做
     */
    protected updateLocalBoundsSelf() {
        //修改_localBoundsSelf
        const width = this._texture.orig.width;
        const height = this._texture.orig.height;
        this._localBoundsSelf.x = -width * this.anchorTexture.x;
        this._localBoundsSelf.y = -height * this.anchorTexture.y;
        this._localBoundsSelf.width = width;
        this._localBoundsSelf.height = height;
    }

    /**
     * 01——23
     * |   |
     * 67——45
     * 计算全局的顶点数据
     */
    calculateVertices() {
        if (this._transformID === this.transform._worldID && this._textureID === this._texture._updateID) {
            return;
        }

        this._transformID = this.transform._worldID;
        this._textureID = this._texture._updateID;

        const texture = this._texture;
        const wt = this.transform.worldMatrix;
        const a = wt.a;
        const b = wt.b;
        const c = wt.c;
        const d = wt.d;
        const tx = wt.tx;
        const ty = wt.ty;
        const vertexData = this._vertexData;
        const trim = texture.trim;
        const orig = texture.orig;
        const anchor = this._anchorTexture;

        let w0 = 0;
        let w1 = 0;
        let h0 = 0;
        let h1 = 0;

        if (trim) {
            // if the sprite is trimmed and is not a tilingsprite then we need to add the extra
            // space before transforming the sprite coords.
            //绘制永远根据frame，所以有trim过的，加个偏移
            w1 = trim.x - (anchor._x * orig.width);
            w0 = w1 + trim.width;

            h1 = trim.y - (anchor._y * orig.height);
            h0 = h1 + trim.height;
        }
        else {
            w1 = -anchor._x * orig.width;
            w0 = w1 + orig.width;

            h1 = -anchor._y * orig.height;
            h0 = h1 + orig.height;
        }

        // xy
        vertexData[0] = (a * w1) + (c * h1) + tx;
        vertexData[1] = (d * h1) + (b * w1) + ty;

        // xy
        vertexData[2] = (a * w0) + (c * h1) + tx;
        vertexData[3] = (d * h1) + (b * w0) + ty;

        // xy
        vertexData[4] = (a * w0) + (c * h0) + tx;
        vertexData[5] = (d * h0) + (b * w0) + ty;

        // xy
        vertexData[6] = (a * w1) + (c * h0) + tx;
        vertexData[7] = (d * h0) + (b * w1) + ty;
    }

    /**
     * 用于trim过的纹理，计算真实顶点位置，
     * 因为trim会裁切边缘透明像素，所以直接用orig的尺寸
     */
    calculateTrimmedVertices() {
        if (!this._vertexTrimmedData) {
            this._vertexTrimmedData = new Float32Array(8);
        }
        else if (this._transformTrimmedID === this.transform._worldID && this._textureTrimmedID === this._texture._updateID) {
            return;
        }

        this._transformTrimmedID = this.transform._worldID;
        this._textureTrimmedID = this._texture._updateID;

        // lets do some special trim code!
        const texture = this._texture;
        const vertexData = this._vertexTrimmedData;
        const orig = texture.orig;
        const anchor = this._anchorTexture;

        // lets calculate the new untrimmed bounds..
        const wt = this.transform.worldMatrix;
        const a = wt.a;
        const b = wt.b;
        const c = wt.c;
        const d = wt.d;
        const tx = wt.tx;
        const ty = wt.ty;

        const w1 = -anchor._x * orig.width;
        const w0 = w1 + orig.width;

        const h1 = -anchor._y * orig.height;
        const h0 = h1 + orig.height;

        // xy
        vertexData[0] = (a * w1) + (c * h1) + tx;
        vertexData[1] = (d * h1) + (b * w1) + ty;

        // xy
        vertexData[2] = (a * w0) + (c * h1) + tx;
        vertexData[3] = (d * h1) + (b * w0) + ty;

        // xy
        vertexData[4] = (a * w0) + (c * h0) + tx;
        vertexData[5] = (d * h0) + (b * w0) + ty;

        // xy
        vertexData[6] = (a * w1) + (c * h0) + tx;
        vertexData[7] = (d * h0) + (b * w1) + ty;
    }

    /**
    * 自身webgl绘制方法
    * @private
    * @param {WebglRenderer} renderer
    */
    _renderWebGL(renderer: WebglRenderer) {
        //先计算顶点
        this.calculateVertices();
        renderer.batchManager.setObjectRenderer(renderer.plugins["batch"]);
        renderer.plugins["batch"].render(this);
    }

    /**
    * 自身canvas绘制方法
    * @private
    * @param {CanvasRenderer} renderer
    */
    _renderCanvas(renderer: CanvasRenderer) {
        renderer.plugins["sprite"].render(this);
    }

    /**
     * 更新自己的bounds，计算全局
     * @private
     */
    _calculateBounds() {
        const trim = this._texture.trim;
        const orig = this._texture.orig;
        //无trim。或者trim的尺寸和orig相等
        if (!trim || (trim.width === orig.width && trim.height === orig.height)) {
            this.calculateVertices();
            Rectangle.createFromVertexData(this._bounds, this._vertexData);
        }
        else {
            //计算trimmed bounds...
            this.calculateTrimmedVertices();
            Rectangle.createFromVertexData(this._bounds, this._vertexTrimmedData);
        }
    }

    /**
     * 重写父类
     * @param {Rectangle} rect - The output rectangle.
     * @return {Rectangle} The bounds.
     */
    getLocalBounds(rect?: Rectangle): Rectangle {
        //如果没有child，直接
        if (this.children.length === 0) {
            if (!rect) {
                rect = DisplayObject.temBounds;
            }
            //直接用_localBoundsSelf
            rect.copy(this._localBoundsSelf)
            return rect;
        }

        return super.getLocalBounds.call(this, rect);
    }

    /**
     * 是否使用像素检测，对于png图片透明区域检测很有效， 默认关闭，Shape继承后默认开启
     * @property hitTestByPixel
     * @type {boolean}
     */
    public hitTestByPixel: boolean = false;
    /**
     * 重写碰撞检测方法
     * @param globalPoint 
     * @param isMouseEvent 
     */
    hitTestPoint(globalPoint: Point, isMouseEvent: boolean = false) {
        //不可见，直接返回
        if (!this.visible) return null
        //mouseChildren和mouseEnable在各自继承里判断，container和displayObject里都有
        //如果以后加缓存成位图，另写
        let hitDisplayObject;
        //先检查子级，因为子级层级更高
        hitDisplayObject = super.hitTestPoint(globalPoint, isMouseEvent);
        //子级已有，返回
        if (hitDisplayObject) return hitDisplayObject;
        //检查自己，包围盒检测。
        hitDisplayObject = this.displayObjectHitTestPoint(globalPoint, isMouseEvent);
        //自身包围盒检测不通过
        if (!hitDisplayObject) return null;
        //不是像素检测的，直接返回对象
        if (!this.hitTestByPixel) return hitDisplayObject;
        ///////////////下面开始像素检测//////////////////
        const { texture, _anchorTexture } = this;
        const { orig, frame, trim, rotate, baseTexture } = texture;
        //拷贝一个，以防globalPoint被修改
        let p: Point = DisplayObject._bp.copy(globalPoint);
        //鼠标事件转换下
        if (isMouseEvent) p = this.globalToLocal(p);
        //贴图锚点偏移
        p.x += _anchorTexture.x * orig.width;
        p.y += _anchorTexture.y * orig.height;
        //先判断是否在trim外面，orig已经在displayObjectHitTestPoint检测过了
        if (trim) {
            //不在里面直接返回false
            if (!trim.isPointIn(p)) return null;
            //还需要做点啥，转到trim中
            p.x -= trim.x;
            p.y -= trim.y;
        }
        //如果纹理带旋转，转换p，暂时就2和6有效，所以texture的rotate暂时只给图集用，自己千万别擅自修改
        if (rotate) {
            if (!tempMatrix) tempMatrix = new Matrix();
            // tempMatrix.set(1, 0, 0, 1, 0, 0)
            //其实只有2和6的话，直接能写出矩阵，但是先这样吧，先注释吧，后续都能算了再统一到GroupD8里计算
            // GroupD8.matrixAppendRotationInv(tempMatrix, GroupD8.inv(rotate));
            //还需要平移
            if (rotate == 2) {
                tempMatrix.set(0, 1, -1, 0, frame.width, 0);
            }
            else if (rotate == 6) {//待测试
                tempMatrix.set(0, -1, 1, 0, 0, frame.height);
            }
            tempMatrix.transformPoint(p.x, p.y, p);
        }
        //处理frame的偏移
        p.x += frame.x;
        p.y += frame.y;
        //数据形式，直接取
        if (baseTexture.source._data) {
            //对应点的像素
            if (baseTexture.source._data[((Math.round(p.y - 1)) * baseTexture.source.width + Math.round(p.x)) * 4 - 1] > 0) {
                return hitDisplayObject;
            }
            return null
        }
        //像素检测
        let ctx = getBackupCanvasCtx();
        ctx.setTransform(1, 0, 0, 1, 0, 0);//先移动位置，否则颜色清除有问题,原先修改尺寸就不用
        ctx.clearRect(0, 0, 1, 1);
        ctx.setTransform(1, 0, 0, 1, -p.x, -p.y);
        ctx.drawImage(baseTexture.source, 0, 0);
        //取imageData对象
        var imageData = ctx.getImageData(0, 0, 1, 1);
        //容错处理(暂时淘宝小程序bug，安卓有可能取到的imageData是undefined)，按照存在处理
        if (!imageData || !imageData.data) return hitDisplayObject;
        // console.log("alpha:", imageData.data[3])
        //像素透明度不为0
        if (imageData.data[3] > 0) return hitDisplayObject;
        return null
    }



    /**
     * 销毁
     */
    destroy() {
        super.destroy();

        //相应texture移除监听
        this._texture.removeEventListener('update', this._onTextureUpdate, this);

        this._anchorTexture = null;
        this._texture = null;
        this._indices = null;
        this._uvs = null;
        this._vertexData = null;
        this._vertexTrimmedData = null;

        //对于canvas可能有调色缓存
        if (this["tintedTexture"]) this["tintedTexture"] = null;
    }

    //一些静态类方法
    /**
     * 网络图片
     * @param url 链接
     * @return Sprite实例
     */
    static fromUrl(url: string): Sprite {
        return new Sprite(Texture.fromUrl(url));
    }

    /**
     * TextureCache缓存里取得frameId，通常图集里得名字
     * @static
     * @param {string} frameId - The frame Id of the texture in the cache
     * @return {Sprite} A new Sprite using a texture from the texture cache matching the frameId
     */
    static fromFrame(frameId: string): Sprite {
        const texture = TextureCache[frameId];
        if (!texture) {
            // throw new Error(`The frameId "${frameId}" does not exist in the texture cache`);
            console.warn(`TextureCache hasn‘t texture ${frameId}`)
        }
        return new Sprite(texture);
    }
}
