import { HashObject } from "../../2d/HashObject";
import { Texture } from "../../2d/texture";
import { hex2rgb } from "../../2d/utils";


/**
 * 环境贴图融合方式
 */
export enum EnvBlendType {
    ENVMAP_BLENDING_MULTIPLY = "ENVMAP_BLENDING_MULTIPLY",
    ENVMAP_BLENDING_MIX = "ENVMAP_BLENDING_MIX",
    ENVMAP_BLENDING_ADD = "ENVMAP_BLENDING_ADD"
}

export enum RenderSideType {
    /**
     * 正面才渲染，逆时针顶点
     */
    FrontSide = 0,
    /**
     * 反面才渲染，顺时针顶点顺序
     */
    BackSide,
    /**
     * 两面都渲染
     */
    DoubleSide,
}

/**
 * 材质基类，一些基本属性加上
 */
export class BaseMaterial extends HashObject {

    /**
     * 十六进制 hex2rgb ,转成0到1的数组
     */
    private _color: number = 0xffffff;
    private _colorArr: Float32Array = new Float32Array([1.0, 1.0, 1.0]);
    get color() {
        return this._color;
    }
    set color(value: number) {
        if (this._color === value) return;
        this._color = value;
        var arr = hex2rgb(value)
        this._colorArr[0] = arr[0];
        this._colorArr[1] = arr[1];
        this._colorArr[2] = arr[2];
    }
    //获取
    get colorArr() {
        return this._colorArr;
    }

    /**
     * 透明度0到1
     */
    alpha: number = 1
    /**
     * 漫反射贴图，一般贴图
     */
    public map: Texture;
    /**
     * 环境贴图
     */
    public envMap: Texture;
    public combine: EnvBlendType = EnvBlendType.ENVMAP_BLENDING_MULTIPLY;//环境贴图融合方式，
    public reflectivity: number = 1;
    // public refractionRatio = 0.98;//先只管反射
    // public envMapIntensity = 1.0

    morphTargets: boolean = false;
    morphNormals: boolean = false;
    skinning: boolean = false;
    /**
     * 是否使用顶点颜色
     */
    useVertexColor: boolean = false
    /**
     * 是否用线框形式绘制
     */
    wireframe: boolean = false;
    /**
     * 材质渲染面
     */
    side: RenderSideType = RenderSideType.FrontSide;
    /**
     * 场景的雾化是否对材质有效
     */
    useFog: boolean = true;
    /**
     * 是否光照影响
     */
    _lightAffect: boolean = false;
    constructor(parameters?: BaseMaterialParamsInt) {
        super();
        this._instanceType = "BaseMaterial";

        //所有属性
        if (parameters) {
            for (const key in parameters) {
                this[key] = parameters[key];
            }
        }
    }
    copy(material: BaseMaterial) {
        this.color = material.color;
        this.map = material.map;
        this.alpha = material.alpha;
        this.morphTargets = material.morphTargets;
        this.morphNormals = material.morphNormals;
        this.skinning = material.skinning;
        this.wireframe = material.wireframe;
        this.side = material.side;
        this._lightAffect = material._lightAffect;
        return this;
    }
    clone() {
        return new BaseMaterial().copy(this);
    }
    destroy() {
        this._color = null;
    }
}

export interface BaseMaterialParamsInt {
    color?: number,
    alpha?: number,
    map?: Texture,
    wireframe?: boolean,
    side?: RenderSideType,
    envMap?: Texture,
    reflectivity?: number,
    // refractionRatio?:number,
    useVertexColor?: boolean,
    morphTargets?: boolean,
    morphNormals?: boolean,
    skinning?: boolean,
}
