import { Mesh } from "../../mesh";
import { CanvasRenderer } from "../CanvasRenderer";
/**
 * canvas模式下网格渲染插件
 */
export class CanvasMeshRenderer {
    constructor(public renderer: CanvasRenderer) { }
    render(mesh: Mesh) {
        const renderer = this.renderer;
        const context = renderer.context;
        const transform = mesh.worldMatrix;
        context.setTransform(
            transform.a,
            transform.b,
            transform.c,
            transform.d,
            transform.tx,
            transform.ty
        );

        renderer.context.globalAlpha = mesh._worldAlpha;
        renderer.setBlendMode(mesh.blendMode);

        //当没有纹理,纯色的时候,可以用其他方式渲染,到时再说
        this._renderTriangles(mesh);
    }

    private _renderTriangles(mesh: Mesh) {
        const indices = mesh._indices;
        const length = indices.length;

        for (let i = 0; i < length; i += 3) {
            const index0 = indices[i] * 2;
            const index1 = indices[i + 1] * 2;
            const index2 = indices[i + 2] * 2;

            this._renderDrawTriangle(mesh, index0, index1, index2);
        }
    }


    private _renderDrawTriangle(mesh: Mesh, index0: number, index1: number, index2: number) {
        const context = this.renderer.context;
        const uvs = mesh._uvs;
        const vertices = mesh._vertices;
        const texture = mesh.texture;

        if (!texture.valid) return;

        const base = texture.baseTexture;
        const textureSource = base.source;
        const textureWidth = base.width;
        const textureHeight = base.height;


        let u0 = uvs[index0] * base.width;
        let u1 = uvs[index1] * base.width;
        let u2 = uvs[index2] * base.width;
        let v0 = uvs[index0 + 1] * base.height;
        let v1 = uvs[index1 + 1] * base.height;
        let v2 = uvs[index2 + 1] * base.height;

        let x0 = vertices[index0];
        let x1 = vertices[index1];
        let x2 = vertices[index2];
        let y0 = vertices[index0 + 1];
        let y1 = vertices[index1 + 1];
        let y2 = vertices[index2 + 1];

        const canvasPadding = mesh.canvasPadding

        if (canvasPadding > 0) {
            const paddingX = canvasPadding / Math.abs(mesh.worldMatrix.a);
            const paddingY = canvasPadding / Math.abs(mesh.worldMatrix.d);
            const centerX = (x0 + x1 + x2) / 3;
            const centerY = (y0 + y1 + y2) / 3;

            let normX = x0 - centerX;
            let normY = y0 - centerY;

            let dist = Math.sqrt((normX * normX) + (normY * normY));

            x0 = centerX + ((normX / dist) * (dist + paddingX));
            y0 = centerY + ((normY / dist) * (dist + paddingY));

            //

            normX = x1 - centerX;
            normY = y1 - centerY;

            dist = Math.sqrt((normX * normX) + (normY * normY));
            x1 = centerX + ((normX / dist) * (dist + paddingX));
            y1 = centerY + ((normY / dist) * (dist + paddingY));

            normX = x2 - centerX;
            normY = y2 - centerY;

            dist = Math.sqrt((normX * normX) + (normY * normY));
            x2 = centerX + ((normX / dist) * (dist + paddingX));
            y2 = centerY + ((normY / dist) * (dist + paddingY));
        }

        //设定截取三角面的路径
        context.save();
        context.beginPath();

        context.moveTo(x0, y0);
        context.lineTo(x1, y1);
        context.lineTo(x2, y2);

        context.closePath();

        context.clip();

        // Compute matrix transform
        const delta = (u0 * v1) + (v0 * u2) + (u1 * v2) - (v1 * u2) - (v0 * u1) - (u0 * v2);
        const deltaA = (x0 * v1) + (v0 * x2) + (x1 * v2) - (v1 * x2) - (v0 * x1) - (x0 * v2);
        const deltaB = (u0 * x1) + (x0 * u2) + (u1 * x2) - (x1 * u2) - (x0 * u1) - (u0 * x2);
        const deltaC = (u0 * v1 * x2) + (v0 * x1 * u2) + (x0 * u1 * v2) - (x0 * v1 * u2) - (v0 * u1 * x2) - (u0 * x1 * v2);
        const deltaD = (y0 * v1) + (v0 * y2) + (y1 * v2) - (v1 * y2) - (v0 * y1) - (y0 * v2);
        const deltaE = (u0 * y1) + (y0 * u2) + (u1 * y2) - (y1 * u2) - (y0 * u1) - (u0 * y2);
        const deltaF = (u0 * v1 * y2) + (v0 * y1 * u2) + (y0 * u1 * v2) - (y0 * v1 * u2) - (v0 * u1 * y2) - (u0 * y1 * v2);

        context.transform(
            deltaA / delta,
            deltaD / delta,
            deltaB / delta,
            deltaE / delta,
            deltaC / delta,
            deltaF / delta
        );

        context.drawImage(
            textureSource,
            0,
            0,
            textureWidth,
            textureHeight,
            0,
            0,
            textureWidth,
            textureHeight
        );

        context.restore();
    }

    // renderMeshFlat(mesh) {
    //     const context = this.renderer.context;
    //     const vertices = mesh.vertices;
    //     const length = vertices.length / 2;

    //     // this.count++;

    //     context.beginPath();

    //     for (let i = 1; i < length - 2; ++i) {
    //         // draw some triangles!
    //         const index = i * 2;

    //         const x0 = vertices[index];
    //         const y0 = vertices[index + 1];

    //         const x1 = vertices[index + 2];
    //         const y1 = vertices[index + 3];

    //         const x2 = vertices[index + 4];
    //         const y2 = vertices[index + 5];

    //         context.moveTo(x0, y0);
    //         context.lineTo(x1, y1);
    //         context.lineTo(x2, y2);
    //     }

    //     context.fillStyle = '#FF0000';
    //     context.fill();
    //     context.closePath();
    // }

    destroy() {
        this.renderer = null;
    }
}
