import { Mesh } from './Mesh';
import { Texture } from '../texture';

/**
 * @class
 * @extends Mesh
 *
 */
export class Plane extends Mesh {
    /**
     * @param {Texture} texture - 纹理
     * @param {number} [verticesX=10] - x轴顶点数
     * @param {number} [verticesY=10] - y轴顶点数
     */
    constructor(texture: Texture, public verticesX: number = 10, public verticesY: number = 10) {
        super(texture);
        this.verticesX = verticesX || 10;//先留着吧，万一有传null，或0的
        this.verticesY = verticesY || 10;
        this.refresh();
    }

    /**
     * Refreshes plane coordinates
     *
     */
    _refresh() {
        const texture = this.texture;
        const total = this.verticesX * this.verticesY;
        const verts = [];
        const uvs = [];
        const indices = [];

        const segmentsX = this.verticesX - 1;
        const segmentsY = this.verticesY - 1;

        const sizeX = texture.width / segmentsX;
        const sizeY = texture.height / segmentsY;

        for (let i = 0; i < total; i++) {
            const x = (i % this.verticesX);
            const y = ((i / this.verticesX) | 0);

            verts.push(x * sizeX, y * sizeY);

            uvs.push(x / segmentsX, y / segmentsY);
        }



        // console.log(indices)
        this._vertices = new Float32Array(verts);
        this._uvs = new Float32Array(uvs);
        this._indices = calculatePlaneIndices(this.verticesX, this.verticesY);

        this._vertexDirty++;

        this.multiplyUvs();
    }
}

/**
 * 计算平面的索引
 * @param verticesX x轴上的顶点数量，最小2
 * @param verticesY y轴上的顶点数量，最小2
 */
export function calculatePlaneIndices(verticesX: number, verticesY: number): Uint16Array {
    const segmentsX = verticesX - 1;
    const segmentsY = verticesY - 1;
    const totalSub = segmentsX * segmentsY;
    const indices = [];
    for (let i = 0; i < totalSub; i++) {
        const xpos = i % segmentsX;
        const ypos = (i / segmentsX) | 0;

        const value = (ypos * verticesX) + xpos;
        const value2 = (ypos * verticesX) + xpos + 1;
        const value3 = ((ypos + 1) * verticesX) + xpos;
        const value4 = ((ypos + 1) * verticesX) + xpos + 1;

        indices.push(value, value2, value3);
        indices.push(value2, value4, value3);
    }
    return new Uint16Array(indices);
}

// new Ripple(texutre, 50, 108, 3)

// class Ripple extends Plane {
//     /**
//      * 波源影响半径
//      */
//     private radius: number;
//     private oldNew: boolean;
//     private heightOld: number[][];
//     private heightNew: number[][];
//     private _verticesOri
//     /**
//      * 分段数
//      * @param segX 
//      * @param segY 
//      */
//     constructor(texture: Texture, segX: number = 10, segY: number = 10, radius: number = 5) {
//         super(texture, segX, segY)

//         this.radius = radius;
//         this.oldNew = false;
//         this.heightOld = [];
//         this.heightNew = [];
//         for (var i = 0; i < this.verticesX; i++) {
//             this.heightOld[i] = [];
//             this.heightNew[i] = [];
//             for (var j = 0; j < this.verticesY; j++) {
//                 this.heightOld[i][j] = 0;
//                 this.heightNew[i][j] = 0;
//             }
//         }
//         //拷贝一个源数据
//         this._verticesOri = this._vertices.slice();
//         let fun = (a, e) => {
//             var x1 = Math.round(e.localX / (texture.width / (this.verticesX - 1)));
//             var y1 = Math.round(e.localY / (texture.height / (this.verticesY - 1)));
//             // console.log(x1,y1)
//             this.disturb(x1, y1, a);
//         }
//         this.addEventListener(MouseEvent.MOUSE_MOVE, fun.bind(this, "m"), this)
//         this.addEventListener(MouseEvent.MOUSE_DOWN, fun.bind(this, "d"), this)
//     }

//     getEnergy(e, r, d) {
//         return e * (1 - d / r);
//     };
//     refractionOff(i, j, rect) {
//         var xl = i - 1;
//         var xr = i + 1;
//         if (i == 0) xl = 0;

//         if (i == this.verticesX - 1) xr = this.verticesX - 1;

//         var yu = j - 1;
//         var yd = j + 1;
//         if (j == 0) yu = 0

//         if (j == this.verticesY - 1) yd = this.verticesY - 1

//         var xoff = rect[i][yu] - rect[i][yd];
//         var yoff = rect[xr][j] - rect[xl][j];
//         return { xoff: xoff, yoff: yoff }
//     };
//     distance(a, b) {
//         var x = b.x - a.x, y = b.y - a.y;
//         return Math.sqrt(x * x + y * y);
//     };
//     getAmplitude(i, j, rect) {
//         var xl = i - 1;
//         var xr = i + 1;
//         if (i == 0) xl = 0;

//         if (i == this.verticesX - 1) xr = this.verticesX - 1;
//         var yu = j - 1;
//         var yd = j + 1;
//         if (j == 0) yu = 0;

//         if (j == this.verticesY - 1) yd = this.verticesY - 1
//         return (rect[i][yu] + rect[i][yd] + rect[xl][j] + rect[xr][j]) / 2
//     };
//     disturb(i, j, type = "d") {
//         var rectsv = this.heightOld;
//         var rectsNewv = this.heightNew;
//         var xmin = i - this.radius;
//         if (xmin < 0) xmin = 0;

//         var xmax = i + this.radius;
//         if (xmax > this.verticesX - 1) xmax = this.verticesX - 1;
//         var ymin = j - this.radius;
//         if (ymin < 0) ymin = 0;

//         var ymax = j + this.radius;
//         if (ymax > this.verticesY - 1) ymax = this.verticesY - 1;
//         for (var m = xmin; m <= xmax; m++) {
//             for (var n = ymin; n <= ymax; n++) {
//                 var dist = this.distance({ x: i, y: j }, { x: m, y: n });

//                 if (dist <= this.radius) {
//                     if (type == "d") {
//                         rectsv[m][n] += this.getEnergy(64, this.radius, dist);
//                         rectsNewv[m][n] += this.getEnergy(64, this.radius, dist);
//                     } else {
//                         rectsv[m][n] += this.getEnergy(16, this.radius, dist);
//                         rectsNewv[m][n] += this.getEnergy(16, this.radius, dist);
//                     }


//                 }
//             }
//         }
//     };
//     update() {
//         super.update();
//         for (var i = 0; i < this.verticesX; i++) {
//             for (var j = 0; j < this.verticesY; j++) {
//                 var rectsv = this.heightOld;
//                 var rectsNewv = this.heightNew;
//                 var d;
//                 var off;
//                 if (this.oldNew) {
//                     d = this.getAmplitude(i, j, rectsv);
//                     d -= rectsNewv[i][j];
//                     d -= d / 64;
//                     rectsNewv[i][j] = d;
//                     off = this.refractionOff(i, j, this.heightNew);
//                 } else {
//                     d = this.getAmplitude(i, j, rectsNewv);
//                     d -= rectsv[i][j];
//                     d -= d / 64;
//                     rectsv[i][j] = d;
//                     off = this.refractionOff(i, j, this.heightOld);
//                 }
//                 var index = j * this.verticesX + i;
//                 this._vertices[index * 2] = this._verticesOri[index * 2] - off.xoff;
//                 this._vertices[index * 2 + 1] = this._verticesOri[index * 2 + 1] - off.yoff;
//                 // this.vets[index].z = d;
//             }
//         }
//         this.oldNew = !this.oldNew;
//         //确保会更新
//         this._vertexDirty++
//     };

// }