import { AnimationTrack3D } from "./AnimationTrack3D";
import { HashObject } from "../../2d/HashObject";
import { EventDispatcher, Event } from "../../2d/events";

/**
 * 不需要挂到节点上
 * 暂时没有帧，只有时间，以后再说
 * 用update更新
 */
export class AnimationClip3D extends EventDispatcher {
    /**
     * 所有的动画数据
     */
    private tracks: AnimationTrack3D[];
    private _totalTime: number;
    get totalTime() {
        return this._totalTime;
    }
    constructor(tracks: AnimationTrack3D[]) {
        super()
        this._instanceType = "AnimationClip"
        this.tracks = tracks;
        this.calculateTotalTime();
    }
    private calculateTotalTime() {
        var tracks = this.tracks, duration = 0;
        for (var i = 0, n = tracks.length; i !== n; ++i) {
            var track = this.tracks[i];
            duration = Math.max(duration, track.times[track.times.length - 1]);
        }
        this._totalTime = duration;
        return this;
    }

    private _isPlaying: boolean = true;
    public get isPlaying(): boolean {
        return this._isPlaying;
    }

    private _isFront: boolean = true;
    get isFront(): boolean {
        return this._isFront;
    }

    /**
     * 上个时间，用来确定是否更新
     */
    private lastTime: number = null;
    /**
     * 记录时间
     */
    private curTime: number = 0;

    /**
     * 需要挂在循环里的方法，传时间间隔
     * @param time 
     */
    update(time: number) {
        if (!this.tracks || !this.tracks.length) return;
        //时间不等，直接播放
        if (this.curTime !== this.lastTime) {
            this.rectify()
            return;
        }
        //时间没有，或没在播放
        if (time <= 0 || !this._isPlaying) return;
        if (this._isFront) {
            this.curTime += time;
            if (this.curTime > this._totalTime) this.curTime = 0;
        } else {
            this.curTime -= time;
            if (this.curTime < 0) this.curTime = this._totalTime;
        }
        if (this.curTime !== this.lastTime) {
            //矫正
            this.rectify();
            //派发事件
            this.dispatchEvent(Event.ENTER_FRAME);
        }
    }
    /**
     * 从当前时间点播放
     * @param isFront 默认true正向
     */
    play(isFront: boolean = true) {
        this._isFront = isFront;
        this._isPlaying = true;
    }
    /**
     * 停在当前时间
     */
    stop() {
        this._isPlaying = false;
    }
    /**
     * 
     * @param time 
     * @param isFront 默认true，正向播放
     */
    public gotoAndPlay(time: number, isFront: boolean = true): void {
        let s = this;
        s._isFront = isFront;
        s._isPlaying = true;
        if (time > s._totalTime) time = s._totalTime;
        if (time < 1) time = 0;
        s.curTime = time;
    }
    /**
     * 停在指定时间
     * @param time 
     */
    public gotoAndStop(time: number): void {
        this._isPlaying = false;
        if (time > this.totalTime) time = this.totalTime;
        if (time < 0) time = 0;
        this.curTime = time;;
    }

    private startAniRangeFun;

    public startAniRange(
        beginTime: number = 0,
        endTime: number = this._totalTime,
        loops: number = 1,
        callback?: Function
    ) {
        if (beginTime < 0) beginTime = 0;
        if (beginTime > this._totalTime) beginTime = this._totalTime;
        if (endTime < 0) endTime = 0;
        if (endTime > this._totalTime) endTime = this._totalTime;
        if (beginTime === endTime) {
            this.gotoAndStop(beginTime)
            //如果相等
            return
        } else if (beginTime < endTime) {
            this._isFront = true;
        } else {
            this._isFront = false;
            var temp = beginTime;
            beginTime = endTime;
            endTime = temp;
        }
        //移除原先的绑定吧
        if (this.startAniRangeFun) this.removeEventListener(Event.ENTER_FRAME, this.startAniRangeFun, this)
        this.curTime = beginTime;
        this._isPlaying = true;
        let loopCount = loops ? (loops + 0.5 >> 0) : Infinity;
        this.addEventListener(Event.ENTER_FRAME, this.startAniRangeFun = (e: Event) => {
            let s: AnimationClip3D = e.target;
            if (s._isFront) {
                if (s.curTime >= endTime) {
                    loopCount--;
                    if (loopCount <= 0) {
                        s.gotoAndStop(endTime);
                        s.removeEventListener(Event.ENTER_FRAME, this.startAniRangeFun, this);
                        this.startAniRangeFun = null;
                        callback && callback();
                    } else {
                        s.gotoAndPlay(beginTime);
                    }
                }
            } else {
                if (s.curTime <= beginTime) {
                    loopCount--
                    if (loopCount <= 0) {
                        s.gotoAndStop(beginTime);
                        s.removeEventListener(Event.ENTER_FRAME, this.startAniRangeFun, this);
                        this.startAniRangeFun = null;
                        callback && callback();
                    } else {
                        s.gotoAndPlay(endTime, false);
                    }
                }
            }
        }, this)
    }
    /**
     * 矫正
     */
    private rectify() {
        if (!this.tracks || !this.tracks.length) return;
        for (var i = 0; i < this.tracks.length; i++) {
            this.tracks[i].setValue(this.curTime)
        }
        //设置相等
        this.lastTime = this.curTime;
    }
}