import { Slot } from "../Slot";
import { Bone } from "../Bone";


//attachment用于记录数据用
export abstract class Attachment {
    constructor(public name: string) {
    }
}

export abstract class VertexAttachment extends Attachment {
    private static nextID = 0;

    public id = (VertexAttachment.nextID++ & 65535) << 11;
    //用到的骨骼
    bones: Array<number>;
    vertices: ArrayLike<number>;
    worldVerticesLength = 0;
    deformAttachment: VertexAttachment = this;

    constructor(name: string) {
        super(name);

    }
    applyDeform (sourceAttachment) {
        return this == sourceAttachment;
    };
    computeWorldVerticesOld(slot: Slot, worldVertices: ArrayLike<number>, skeletonBones: Bone[]) {
        this.computeWorldVertices(slot, 0, this.worldVerticesLength, worldVertices, 0, 2, skeletonBones);
    }

    /** Transforms local vertices to world coordinates.
     * @param start The index of the first local vertex value to transform. Each vertex has 2 values, x and y.
     * @param count The number of world vertex values to output. Must be <= {@link #getWorldVerticesLength()} - start.
     * @param worldVertices The output world vertices. Must have a length >= offset + count.
     * @param offset The worldVertices index to begin writing values. */
    computeWorldVertices(
        slot: Slot,
        start: number,
        count: number,
        worldVertices: ArrayLike<number>,
        offset: number,
        stride: number,
        skeletonBones: Bone[]//全部的骨骼数组
    ) {
        count = offset + (count >> 1) * stride;
        let deformArray = slot.deform;
        let vertices = this.vertices;
        let bones = this.bones;
        if (bones == null) {
            if (deformArray.length > 0) vertices = deformArray;
            let mat = slot.bone.matrix//.worldMatrix;
            let x = mat.tx;
            let y = mat.ty;
            let a = mat.a, b = mat.c, c = mat.b, d = mat.d;
            for (let v = start, w = offset; w < count; v += 2, w += stride) {
                let vx = vertices[v], vy = vertices[v + 1];
                //@ts-ignore
                worldVertices[w] = vx * a + vy * b + x;
                //@ts-ignore
                worldVertices[w + 1] = vx * c + vy * d + y;
            }
            return;
        }
        let v = 0, skip = 0;
        for (let i = 0; i < start; i += 2) {
            let n = bones[v];
            v += n + 1;
            skip += n;
        }
        if (deformArray.length == 0) {
            for (let w = offset, b = skip * 3; w < count; w += stride) {
                let wx = 0, wy = 0;
                let n = bones[v++];
                n += v;
                for (; v < n; v++, b += 3) {
                    let mat = skeletonBones[bones[v]].matrix//.worldMatrix;
                    let vx = vertices[b], vy = vertices[b + 1], weight = vertices[b + 2];
                    wx += (vx * mat.a + vy * mat.c + mat.tx) * weight;
                    wy += (vx * mat.b + vy * mat.d + mat.ty) * weight;
                }
                //@ts-ignore
                worldVertices[w] = wx;
                //@ts-ignore
                worldVertices[w + 1] = wy;
            }
        } else {
            let deform = deformArray;
            for (let w = offset, b = skip * 3, f = skip << 1; w < count; w += stride) {
                let wx = 0, wy = 0;
                let n = bones[v++];
                n += v;
                for (; v < n; v++, b += 3, f += 2) {
                    let mat = skeletonBones[bones[v]].matrix//.worldMatrix;
                    let vx = vertices[b] + deform[f], vy = vertices[b + 1] + deform[f + 1], weight = vertices[b + 2];
                    wx += (vx * mat.a + vy * mat.c + mat.tx) * weight;
                    wy += (vx * mat.b + vy * mat.d + mat.ty) * weight;
                }
                //@ts-ignore
                worldVertices[w] = wx;
                //@ts-ignore
                worldVertices[w + 1] = wy;
            }
        }
    }
}
