import BaseRenderTexture from './BaseRenderTexture';
import Texture from './Texture';
import { Rectangle } from '../math';
import { SCALE_MODES } from '../const';

/**
 * A RenderTexture is a special texture that allows any display object to be rendered to it.
 *
 * __Hint__: All DisplayObjects (i.e. Sprites) that render to a RenderTexture should be preloaded
 * otherwise black rectangles will be drawn instead.
 *
 * A RenderTexture takes a snapshot of any Display Object given to its render method. For example:
 *
 * ```js
 * let renderer = autoDetectRenderer(1024, 1024, { view: canvas, ratio: 1 });
 * let renderTexture = RenderTexture.create(800, 600);
 * let sprite = Sprite.fromImage("spinObj_01.png");
 *
 * sprite.position.x = 800/2;
 * sprite.position.y = 600/2;
 * sprite.anchorTexture.x = 0.5;
 * sprite.anchorTexture.y = 0.5;
 *
 * renderer.render(sprite, renderTexture);
 * ```
 *
 * The Sprite in this case will be rendered using its local transform. To render this sprite at 0,0
 * you can clear the transform
 *
 * ```js
 *
 * sprite.setTransform()
 *
 * let renderTexture = new RenderTexture.create(100, 100);
 *
 * renderer.render(sprite, renderTexture);  // Renders to center of RenderTexture
 * ```
 *
 * @class
 * @extends Texture
 */
export default class RenderTexture extends Texture {
    baseTexture: BaseRenderTexture
    /**
     * @param {BaseRenderTexture} baseRenderTexture - The renderer used for this RenderTexture
     * @param {Rectangle} [frame] - The rectangle frame of the texture to show
     */
    constructor(baseRenderTexture: BaseRenderTexture, frame?: Rectangle) {

        super(
            baseRenderTexture,
            frame
        );

        this.valid = true;

        this.updateUvs();
    }

    /**
     * Resizes the RenderTexture.
     *
     * @param {number} width - The width to resize to.
     * @param {number} height - The height to resize to.
     * @param {boolean} doNotResizeBaseTexture - Should the baseTexture.width and height values be resized as well?
     */
    resize(width: number, height: number, doNotResizeBaseTexture: boolean = false) {
        width = Math.ceil(width);
        height = Math.ceil(height);

        // TODO - could be not required..
        this.valid = (width > 0 && height > 0);

        this._frame.width = this.orig.width = width;
        this._frame.height = this.orig.height = height;

        if (!doNotResizeBaseTexture) {
            this.baseTexture["resize"](width, height);
        }

        this.updateUvs();
    }

    /**
     * A short hand way of creating a render texture.
     *
     * @param {number} [width=100] - The width of the render texture
     * @param {number} [height=100] - The height of the render texture
     * @param {number} [scaleMode=settings.SCALE_MODE] - See {@link SCALE_MODES} for possible values
     * @return {RenderTexture} The new render texture
     */
    static create(width: number, height: number, scaleMode: number = SCALE_MODES.LINEAR): RenderTexture {
        return new RenderTexture(new BaseRenderTexture(width, height, scaleMode));
    }
}
