import React, { Component } from 'react'
import { findDOMNode } from 'react-dom'
import { DragSource, DropTarget } from 'react-dnd'

const type = 'SliderCard'

const dragSource = {
  beginDrag(props) {
    return {
      id: props.id,
      index: props.index,
      src: props.src,
      link: props.link
    }
  }
}

const dropTarget = {
  // component是dropTarget的实例
  hover(props, monitor, component) {
    // 如果当前drop实例为空则不做处理
    if (!component) return

    // 获取drag元素以及hover元素索引
    const dragIndex = monitor.getItem().index
    const hoverIndex = props.index

    // 如果是自己则不做任何处理
    if (dragIndex === hoverIndex) return

    // 获取被hover dom 的rect
    const hoverBoundingRect = findDOMNode(component).getBoundingClientRect()

    // 获取垂直和水平坐标
    const hoverMiddleY = (hoverBoundingRect.bottom - hoverBoundingRect.top) / 2
    const hoverCenterX = (hoverBoundingRect.right - hoverBoundingRect.left) / 2

    // 获取鼠标当前位置
    const clientOffset = monitor.getClientOffset()

    // 鼠标相对被hover元素偏移量
    const hoverClientY = clientOffset.y - hoverBoundingRect.top
    const hoverClientX = clientOffset.x - hoverBoundingRect.left

    // 拖拽方向判断
    // 向后拖拽排序
    if (dragIndex < hoverIndex && hoverClientY < hoverMiddleY && hoverClientX < hoverCenterX) return
    // 向前拖拽排序
    if (dragIndex > hoverIndex && hoverClientY > hoverMiddleY && hoverClientX > hoverCenterX) return

    // 重新排序钩子
    props.onReSort(dragIndex, hoverIndex)

    // 关键性能优化
    monitor.getItem().index = hoverIndex
  }
}

@DropTarget(type, dropTarget, connect => ({
  connectDropTarget: connect.dropTarget()
}))
@DragSource(type, dragSource, (connect, monitor) => ({
  connectDragSource: connect.dragSource(),
  isDragging: monitor.isDragging()
}))
export default class SliderCard extends Component {
  constructor(props) {
    super(props)
    this.state = {
      hovering: false
    }
  }
  render() {
    const {
      index,
      src,
      isDragging,
      connectDragSource,
      connectDropTarget,
      onClick,
      DeleteCom,
      style
    } = this.props
    const { hovering } = this.state
    const opacity = isDragging ? 1 : 1

    return (
      connectDragSource &&
      connectDropTarget &&
      connectDragSource(
        connectDropTarget(
          <div
            key={index}
            style={Object.assign(
              {
                position: 'relative',
                backgroundImage: `url("${src}")`,
                opacity,
                cursor: 'move'
              },
              style
            )}
            onClick={() => onClick(index)}
            onMouseEnter={() => {
              this.setState({
                hovering: true
              })
            }}
            onMouseLeave={() => {
              this.setState({
                hovering: false
              })
            }}
          >
            {!isDragging && hovering && DeleteCom}
          </div>
        )
      )
    )
  }
}
