import { observable, action, computed } from 'mobx'
import {
  mutateValueByPath,
  getValueByPath,
  deepClone,
  getNumber
} from 'wp/utils/helper'
import snapshot from './snapshot'
import uuidv4 from 'uuid/v4'
import { sceneGen } from 'wp/constructors/other-default'
import HookStore from './hook'
import * as types from 'wp/utils/itemType'

class SceneStore {
  pageId = null
  @observable
  editingTarget = {
    id: ''
  }
  @observable
  itemList = []
  @observable
  scene = sceneGen()

  @observable
  version = null
  // 标签
  @observable
  tags = []
  @observable
  promoteTag = []

  // 重置数据
  @action
  reset = () => {
    this.pageId = null
    this.editingTarget = { id: '' }
    this.itemList = []
    this.scene = sceneGen()
    this.promoteTag = []
    snapshot.addDep({
      sceneSnapshot: {
        target: this,
        keys: ['scene', 'itemList', 'editingTarget']
      }
    })
  }

  // 初始化数据
  @action
  init = (page = null, elements = []) => {
    this.itemList = elements
    if (page) {
      this.scene = page
      HookStore.walkHooks('onSceneInnerHeightChangeEnd', {
        height: getNumber(page.height)
      })
      HookStore.walkHooks('onSceneInnerWidthChangeEnd', {
        width: getNumber(page.width)
      })
    }
  }

  // 改变场景属性
  @action
  changeSceneProps = obj => {
    // 支持.操作符修改属性
    Object.keys(obj).forEach(key => {
      const path = key
      const value = obj[key]
      mutateValueByPath(this.scene, path, value)
    })
    snapshot.addSnapshot()
  }

  // 向场景中增加部件
  @action
  addItem = item => {
    if (types.PopupType !== item.widgetType) {
      // 处理zIndex
      const maxIndex = this.itemList
        .filter(item => {
          return item.widgetType !== types.PopupType
        })
        .reduceRight((acc, item) => {
          const zIndex = item[item.widgetType].style.zIndex
          if (zIndex > acc) {
            acc = zIndex
          }
          return acc
        }, 0)
      if (maxIndex > 0) {
        item[item.widgetType].style.zIndex = maxIndex
      }
    }
    this.itemList.push(item)
    this.chooseTarget({ targetId: item.targetId })
  }

  // 复制场景部件
  @action
  copyAndPasteItem = targetId => {
    let item = this.itemList.find(item => item.targetId === targetId)
    if (item) {
      let newItem = deepClone(item)
      // 重新更改id
      newItem.targetId = `${newItem.widgetType}_${uuidv4()}`
      newItem[newItem.widgetType].id = newItem.targetId
      // 重命名
      newItem[newItem.widgetType].name += '-copy'
      // 重新设置位置
      newItem[newItem.widgetType].style.top =
        getNumber(newItem[newItem.widgetType].style.top) + 20 + 'px'
      newItem[newItem.widgetType].style.left =
        getNumber(newItem[newItem.widgetType].style.left) + 20 + 'px'
      this.addItem(newItem)
    }
  }

  // 删除当前选中item
  @action
  removeCurrItem = () => {
    const index = this.itemList.findIndex(
      item => item.targetId === this.editingTarget.id
    )
    if (index !== -1) {
      this.editingTarget.id = ''
      this.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }

  // 选中编辑部件
  @action
  chooseTarget = ({ targetId }) => {
    this.editingTarget.id = targetId
    snapshot.addSnapshot()
  }

  // !请勿直接使用此方法，使用changeItemProps代替
  @action
  changeItemProp = (path, value) => {
    if (!this.editingTarget.id) return false
    const target = this.itemList.find(
      item => item.targetId === this.editingTarget.id
    )
    if (!target) return false
    mutateValueByPath(target[target.widgetType], path, value)
  }

  // 改变当前正在编辑的部件多个属性
  @action
  changeItemProps = obj => {
    Object.keys(obj).forEach(key => {
      this.changeItemProp(key, obj[key])
    })
    snapshot.addSnapshot()
  }

  // 根据targetId操作数组项
  @action
  mutateArrayProp = (targetId, path, arrayFuncName, ...params) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return
    let arr = getValueByPath(target[target.widgetType], path)
    if (!(arr instanceof Array)) return
    Array.prototype[arrayFuncName].call(arr, ...params)
  }

  // 根据targetId改变属性
  @action
  changeItemPropsById = (targetId, obj) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return
    Object.keys(obj).forEach(key => {
      mutateValueByPath(target[target.widgetType], key, obj[key])
    })
  }

  // 根据targetId删除元素
  @action
  removeItemById = targetId => {
    const index = this.itemList.findIndex(item => item.targetId === targetId)
    if (index !== -1) {
      this.editingTarget.id = ''
      this.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }

  // 根据targetId查询元素属性（非响应式）
  @action
  getItemPropById = (targetId, path) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return '404'
    return getValueByPath(target[target.widgetType], path)
  }

  // 获取层级
  @action
  getZIndex = () => {
    // 处理zIndex
    const items = this.itemList.filter(item => {
      return item.widgetType !== types.PopupType
    })
    const maxZIndex = items.reduceRight((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex > acc) {
        acc = zIndex
      }
      return acc
    }, 0)
    const minZIndex = items.reduce((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex < acc) {
        acc = zIndex
      }
      return acc
    }, 100000)
    return {
      maxZIndex,
      minZIndex
    }
  }
  @action
  changeTags = tags => {
    this.tags = tags
  }
  @action
  changePromoteTag = tags => {
    console.log(tags)
    this.promoteTag = tags
  }
  /**
   * 当前编辑的Item
   * @return {Item | undefined}
   */
  @computed
  get currentItem() {
    return (this.scene.itemList || []).find(
      item => item.targetId === this.editingTarget.id
    )
  }
}

const sceneStore = new SceneStore()

sceneStore.reset()

export default sceneStore
