import React from 'react'
import 'whatwg-fetch'

import utils from 'pro/lib/utils'
import Cookie from './cookie'

import { message, Modal } from 'antd'

import Loading from 'pro/components/loading'

// message提示框延迟时间
const MESSAGE_DURATION = 3
// 关闭加载提示框
const close = res => {
  setTimeout(() => {
    Loading.close()
  }, 300)
  return res
}
// 拼接请求参数
const concatUrl = (params = {}) => {
  const query = Object.keys(params)
    .map(k => encodeURIComponent(k) + '=' + encodeURIComponent(params[k] || ''))
    .join('&')
  return query
}

const common = {
  // 错误处理框，不阻断用户操作
  handleError(msg) {
    message.error(msg, MESSAGE_DURATION)
  },
  // 成功处理框，不阻断用户操作
  handleSuccess(msg) {
    message.success(msg, MESSAGE_DURATION)
  },
  // 错误提示框，阻断用户操作，需要用户自己手动点击关闭
  handleErrorNeedClick(msg) {
    Modal.error({
      title: '系统提示',
      content: (
        <div>
          <p>{msg}</p>
        </div>
      )
    })
  },
  // 成功提示框，阻断用户操作，需要用户自己手动点击关闭
  handleSuccessNeedClick(msg) {
    Modal.success({
      title: '系统提示',
      content: (
        <div>
          <p>{msg}</p>
        </div>
      )
    })
  },
  handleWarn(msg) {
    message.warn(msg, MESSAGE_DURATION)
  },
  /**
   * fetch请求封装
   * @param {*} action 请求线上地址
   * @param {*} data 请求入参
   * @param {*} method 请求方式，默认get
   * @param {*} options 其他参数
   * options为对象格式,值：
   * isLoading(是否激活请求加载动画)
   * isJson(是否设置post请求头contentType为application/json)
   * content 自定义请求参数
   */
  fetch(action, params = {}, method = 'get', options) {
    let url = action
    let option = {
      method,
      credentials: 'same-origin',
      headers: {
        Accept: 'application/json',
        'Content-Type': 'application/json',
        'X-Csrf-Token': Cookie.getItem('csrf_token'),
        'token-site': Cookie.getItem('token-site')
      }
    }
    if (options && options.content) {
      option = Object.assign({}, option, options.content)
    }
    const methods = ['post', 'put', 'patch', 'delete']
    if (methods.indexOf(method) !== -1) {
      if (options && options.isJson) {
        option.body = JSON.stringify(params)
      } else {
        option = Object.assign({}, option, {
          headers: {
            Accept: 'application/json,text/plain,*/*',
            'Content-Type': 'application/x-www-form-urlencoded;charset=utf-8'
          }
        })
        option.body = utils.serialize(params)
      }
    }
    if (method === 'get') {
      url = Object.keys(params).length ? url + '?' + concatUrl(params) : url
    }
    if (options && options.isLoading) {
      Loading.open()
    }
    return fetch(url, option) // eslint-disable-line no-undef
      .then(response => {
        if (response.status >= 200 && response.status < 300) {
          return response
        } else {
          let error = new Error(response.statusText)
          error.response = response
          throw error
        }
      })
      .then(close, close)
      .then(
        response => {
          return response.json().then(
            res => {
              // SSO登录验证
              if (res.success !== undefined) {
                if (!res.success && !!res.notLogin) {
                  utils.logout()
                }
              } else {
                // 业务接口成功
                if ([0, '0'].indexOf(res.code) === -1) {
                  this.handleErrorNeedClick(res.desc)
                  throw new Error(res.desc)
                }
              }
              return res
            },
            ex => {
              throw new Error(`解析JSON字符串出错:${url} ${ex.message}`)
            }
          )
        },
        ex => {
          throw new Error(`请求服务器出错:${url} ${ex.message}`)
        }
      )
  },
  newFetch(option) {
    const defaultOption = {
      method: 'get',
      credentials: 'same-origin',
      headers: {
        Accept: 'application/json',
        'Content-Type': 'application/json',
        ...(option.headers || {})
      },
      isLoading: false,
      postWithQuery: false
    }
    const targetOpts = utils.deepClone(Object.assign({}, defaultOption, option))
    let { url } = targetOpts
    const {
      method = 'get',
      params,
      data,
      isLoading,
      postWithQuery
    } = targetOpts
    if (method.toLowerCase() === 'post' || method.toLowerCase() === 'put') {
      // 默认POST参数为JSON格式
      targetOpts.body = JSON.stringify(data)
    }
    if (method === 'get' || postWithQuery) {
      url =
        params && Object.keys(params).length
          ? url + '?' + concatUrl(params)
          : url
    }
    if (isLoading) {
      Loading.open()
    }
    return fetch(url, targetOpts) // eslint-disable-line no-undef
      .then(response => {
        if (response.status >= 200 && response.status < 300) {
          return response
        } else {
          const error = new Error(response.statusText)
          error.message = response.toString()
          throw error
        }
      })
      .then(close(targetOpts), close(targetOpts))
      .then(
        response => {
          return response.json().then(
            (res: any) => {
              // SSO登录验证
              if (res.success !== undefined) {
                if (!res.success && !!res.notLogin) {
                  utils.logout()
                }
              } else {
                // 业务接口成功
                if (res.code !== '0') {
                  this.handleError(res.desc)
                  throw new Error(res.desc)
                }
              }
              return res
            },
            (ex: any) => {
              throw new Error(`解析JSON字符串出错:${url} ${ex.message}`)
            }
          )
        },
        ex => {
          this.handleError(ex.message)
          throw new Error(`请求服务器出错:${url} ${ex.message}`)
        }
      )
  }
}

export default common
