const KEYFRAMES_NAME_PREFIX = 'spirit_keyframes_name_'
const ANIMATION_CLASS_NAME_PREFIX = 'spirit_class_name_'
const STYLE_ELEMENT_ID_PREFIX = 'spirit_style_element_'

/**
 * 从html中插入含动画CSS的style标签
 * @param {string} id
 * @param {Record<string,any>} spiritAnimationOpts 精灵动画配置项
 */
export function addAnimationStyleElement(id, spiritAnimationOpts, cb) {
  const keyframesName = `${KEYFRAMES_NAME_PREFIX}${id}`
  const animationClassName = `${ANIMATION_CLASS_NAME_PREFIX}${id}`
  const styleElementId = `${STYLE_ELEMENT_ID_PREFIX}${id}`

  if (
    id === null ||
    id === undefined ||
    spiritAnimationOpts === null ||
    spiritAnimationOpts === undefined
  ) {
    return cb(null)
  }

  var styleElement = document.createElement('style')
  styleElement.setAttribute('id', styleElementId)
  styleElement.innerHTML = getCSSText(
    keyframesName,
    animationClassName,
    spiritAnimationOpts
  )
  document.body.appendChild(styleElement)

  styleElement.addEventListener('load', () => {
    // 这是个魔法setTimeout
    // 没了它，非循环播放的动画不生效
    setTimeout(() => {
      cb(animationClassName)
    })
  })
}

/**
 * 从html中移除style标签
 * @param {*} id
 * @return {string | null} style标签删除成功，返回动画的class名称，否则返回null
 */
export function removeAnimationStyleElement(id) {
  const styleElementId = `${STYLE_ELEMENT_ID_PREFIX}${id}`
  const animationClassName = `${ANIMATION_CLASS_NAME_PREFIX}${id}`

  const styleElement = document.getElementById(styleElementId)
  if (styleElement) {
    const styleElementParent = styleElement.parentNode
    styleElementParent.removeChild(styleElement)
    return animationClassName
  }

  return null
}

/**
 * 获得动画的css
 * @param {string} keyframesName
 * @param {string} animationClassName
 * @param {Record<string,any>} spiritAnimationOpts 精灵动画配置项
 */
function getCSSText(keyframesName, animationClassName, spiritAnimationOpts) {
  const { isLoop, frameNumber, durationPerFrame } = spiritAnimationOpts
  let result = ''

  // 帧数小于等于1，没有动画
  if (frameNumber <= 1) {
    return result
  }

  result += `
    .${animationClassName} {
      animation-name: ${keyframesName};
      animation-duration: ${frameNumber * durationPerFrame}s;
      animation-timing-function: steps(1);
      animation-delay: 0s;
      animation-iteration-count: ${isLoop ? 'infinite' : 1};
      animation-direction: normal;
      animation-fill-mode: forwards;
    }\n
    `

  let stepPercent = 100 / (frameNumber - 1)

  if (isLoop) {
    // 循环播放时，最后一帧得和第一帧衔接起来
    stepPercent = 100 / frameNumber
  }

  const steps = new Array(frameNumber).fill(1).map((_, ind) => {
    return formatNumber(stepPercent * ind, 4)
  })

  result += `@keyframes ${keyframesName} {\n`

  steps.forEach((percent, ind) => {
    result += `${percent}% {background-position: ${-ind * 100}% 0%;}\n`
  })

  result += '}'
  return result
}

/**
 * @param {number} num 数
 * @param {number} precise 保留精度
 */
function formatNumber(num, precise) {
  const t = Math.pow(10, precise)
  return Math.round(num * t) / t
}
