import { observable, action, autorun } from 'mobx'
import sceneStore from './scene'
import snapshot from './snapshot'
import * as types from 'wp/utils/itemType'
import {
  mutateValueByPath,
  getValueByPath,
  deepClone,
  getNumber
} from 'wp/utils/helper'

import uuidv4 from 'uuid/v4'

import switchStore from './switch'
import { STAGE } from 'wp/utils/sceneType'

class PopupStore {
  @observable
  showingPopupId = ''
  @observable
  showOverlay = false
  @observable
  itemList = []
  @observable
  editingTarget = {
    id: ''
  }

  // 重置store
  @action
  reset = () => {
    this.showingPopupId = ''
    this.showOverlay = false
    snapshot.addDep({
      popupStore: {
        target: this,
        keys: ['showingPopupId', 'showOverlay']
      }
    })
  }

  @action
  changeId = targetId => {
    // 展示右侧编辑器以及添加快照
    action(sceneStore.chooseTarget)({ targetId })
    // 从sceneStore回流数据
    let popupItem = sceneStore.itemList.find(item => item.targetId === targetId)
    popupItem &&
      this.refluxItems(popupItem[popupItem.widgetType].elements || [])
    this.showingPopupId = targetId
  }

  @action
  changeOverlay = state => {
    this.showOverlay = state
    if (!state) {
      this.showingPopupId = ''
      // 还原右侧舞台编辑器以及添加快照
      action(sceneStore.chooseTarget)({ targetId: '' })
      switchStore.switchScene(STAGE)
    }
  }

  // 向场景中增加部件
  @action
  addItem = item => {
    if (types.PopupType !== item.widgetType) {
      // 处理zIndex
      const maxIndex = this.itemList
        .filter(item => {
          return item.widgetType !== types.PopupType
        })
        .reduceRight((acc, item) => {
          const zIndex = item[item.widgetType].style.zIndex
          if (zIndex > acc) {
            acc = zIndex
          }
          return acc
        }, 0)
      if (maxIndex > 0) {
        item[item.widgetType].style.zIndex = maxIndex
      }
    }
    this.itemList.push(item)
    this.chooseTarget({ targetId: item.targetId })
  }
  // 选中编辑部件
  @action
  chooseTarget = ({ targetId }) => {
    this.editingTarget.id = targetId
    // snapshot.addSnapshot()
  }
  // 复制场景部件
  @action
  copyAndPasteItem = targetId => {
    let item = this.itemList.find(item => item.targetId === targetId)
    if (item) {
      let newItem = deepClone(item)
      // 重新更改id
      newItem.targetId = `${newItem.widgetType}$${uuidv4()}`
      newItem[newItem.widgetType].id = newItem.targetId
      // 重命名
      newItem[newItem.widgetType].name += '-copy'
      // 重新设置位置
      newItem[newItem.widgetType].style.top =
        getNumber(newItem[newItem.widgetType].style.top) + 20 + 'px'
      newItem[newItem.widgetType].style.left =
        getNumber(newItem[newItem.widgetType].style.left) + 20 + 'px'
      this.addItem(newItem)
    }
  }

  // 删除当前选中item
  @action
  removeCurrItem = () => {
    const index = this.itemList.findIndex(
      item => item.targetId === this.editingTarget.id
    )
    if (index !== -1) {
      this.editingTarget.id = ''
      this.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }

  // 选中编辑部件
  @action
  chooseTarget = ({ targetId }) => {
    this.editingTarget.id = targetId
    snapshot.addSnapshot()
  }

  // !请勿直接使用此方法，使用changeItemProps代替
  @action
  changeItemProp = (path, value) => {
    if (!this.editingTarget.id) return false
    const target = this.itemList.find(
      item => item.targetId === this.editingTarget.id
    )
    if (!target) return false
    mutateValueByPath(target[target.widgetType], path, value)
  }

  // 改变当前正在编辑的部件多个属性
  @action
  changeItemProps = obj => {
    Object.keys(obj).forEach(key => {
      this.changeItemProp(key, obj[key])
    })
    snapshot.addSnapshot()
  }

  // 根据targetId操作数组项
  @action
  mutateArrayProp = (targetId, path, arrayFuncName, ...params) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return
    let arr = getValueByPath(target[target.widgetType], path)
    if (!(arr instanceof Array)) return
    Array.prototype[arrayFuncName].call(arr, ...params)
  }

  // 根据targetId改变属性
  @action
  changeItemPropsById = (targetId, obj) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return
    Object.keys(obj).forEach(key => {
      mutateValueByPath(target[target.widgetType], key, obj[key])
    })
  }

  // 根据targetId删除元素
  @action
  removeItemById = targetId => {
    const index = this.itemList.findIndex(item => item.targetId === targetId)
    if (index !== -1) {
      this.editingTarget.id = ''
      this.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }

  // 根据targetId查询元素属性（非响应式）
  @action
  getItemPropById = (targetId, path) => {
    const items = [...this.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return '404'
    return getValueByPath(target[target.widgetType], path)
  }

  // 获取层级
  @action
  getZIndex = () => {
    // 处理zIndex
    const items = this.itemList.filter(item => {
      return item.widgetType !== types.PopupType
    })
    const maxZIndex = items.reduceRight((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex > acc) {
        acc = zIndex
      }
      return acc
    }, 0)
    const minZIndex = items.reduce((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex < acc) {
        acc = zIndex
      }
      return acc
    }, 100000)
    return {
      maxZIndex,
      minZIndex
    }
  }

  @action
  // 回流当前弹层数据
  refluxItems = list => {
    this.itemList = list
  }
}

const popupStore = new PopupStore()

popupStore.reset()

// 更新到sceneStore
autorun(
  () => {
    console.log('autorun')
    sceneStore.changeItemPropsById(popupStore.showingPopupId, {
      elements: [...popupStore.itemList],
      // * elementsLength字段是无用的，为了让autorun把itemlist的length收集为依赖
      // * 这里只在数组长度改变时执行，后续item改变不用同步，以为这里只保存了引用
      // ! 依赖mutable data特性，后续重构考虑更好的方案
      elementsLength: popupStore.itemList.length
    })
  },
  // ! 因为依赖showingPopupId和itemList，所以切换弹层时，refluxItems和changeId会触发两次autorun，造成第一次触发时showingPopupId和itemList不是正确对应的。所以必须加上防抖的delay
  {
    delay: 300
  }
)

export default popupStore
