import React, { Component } from 'react'
import { Table, message, Tooltip } from 'antd'
import CopyToClipboard from 'react-copy-to-clipboard'
import nanoid from 'nanoid'
import { getFormData } from '../../api'
import PubSub from 'pubsub-js'
import { FormDataConditionEvent } from '../../config/pbTopics'
import StatisticStore from '../../store'
import update from 'immutability-helper'
import { FormDataTableOptions } from '../../config'
import moment from 'moment'

export default class FormData extends Component {
  state = {
    data: [],
    columns: [],
    loading: false,
    conditions: {
      pagination: {
        current: 1,
        pageSize: 20,
        total: 20
      },
      startDate: FormDataTableOptions.defaultRangeValue[0].format('YYYY-MM-DD'),
      endDate: moment(FormDataTableOptions.defaultRangeValue[1])
        .add(1, 'days')
        .format('YYYY-MM-DD'),
      form: {}
    }
  }
  fetch = () => {
    this.setState({
      loading: true,
      data: []
    })
    const { pagination, startDate, endDate, form } = this.state.conditions
    getFormData({
      currentPage: pagination.current,
      pageSize: pagination.pageSize,
      startDate,
      endDate,
      formName: form ? form.name : '',
      formId: form ? form.id : '',
      landId: StatisticStore.landId
    }).then(res => {
      this.setState({ loading: false })
      if (res.code === '0') {
        if (!res.data.dataList.list) {
          message.error('接口数据有误')
          return
        }
        if (res.data.dataList.list.length === 0) {
          message.info('无数据')
          return
        }
        this.setState({
          data: res.data.dataList.list,
          columns: genColumns(res.data.headers),
          conditions: update(this.state.conditions, {
            pagination: {
              $set: Object.assign({}, this.state.conditions.pagination, {
                total: res.data.dataList.totalCount
              })
            }
          })
        })
      }
    })
  }
  handleTableChange = (pagination, filters, sorter) => {
    const pager = { ...this.state.pagination }
    // 如果切换页大小则重置current
    if (pagination.pageSize !== pager.pageSize) {
      pager.current = 1
    } else {
      pager.current = pagination.current
    }
    // 更新条件后触发fetch
    this.setState(
      {
        conditions: update(this.state.conditions, {
          pagination: {
            $set: Object.assign({}, this.state.conditions.pagination, {
              current: pagination.current,
              pageSize: pagination.pageSize
            })
          }
        })
      },
      this.fetch
    )
  }
  componentWillMount() {
    PubSub.subscribe(FormDataConditionEvent, (msg, data) => {
      this.setState(
        {
          conditions: Object.assign({}, this.state.conditions, data)
        },
        this.fetch
      )
    })
  }
  componentWillUnmount() {
    PubSub.unsubscribe(FormDataConditionEvent)
  }
  render() {
    // 动态计算滚动区域高度
    let scrollHeight = window.innerHeight - 118 - 8 - 45 - 10
    let { pagination } = this.state.conditions
    let { data, columns } = this.state
    return (
      <div
        style={{
          width: window.innerWidth - 360,
          padding: '0 24px',
          marginTop: 8
        }}
        className="statistic-table"
      >
        <Table
          bordered={false}
          loading={this.state.loading}
          rowKey={() => nanoid()}
          pagination={Object.assign(
            {
              size: 'normal',
              showQuickJumper: true,
              pageSizeOptions: ['10', '20', '30'],
              showSizeChanger: true
            },
            pagination
          )}
          columns={columns}
          dataSource={data}
          onChange={this.handleTableChange}
          scroll={{ x: 180 * columns.length, y: scrollHeight }}
        />
      </div>
    )
  }
}

function genColumns(headers) {
  const defaultWidth = 180
  const genColumn = item => ({
    title: item.desc,
    dataIndex: item.header,
    key: item.header,
    width: defaultWidth,
    render(value) {
      if (item.header === 'infoBusiness') {
        return (
          <Tooltip title={value}>
            <CopyToClipboard
              text={value}
              onCopy={() => message.info('复制成功')}
            >
              <div
                style={{
                  lineHeight: '21px',
                  height: 21,
                  overflow: 'hidden',
                  whiteSpace: 'nowrap',
                  textOverflow: 'ellipsis',
                  width: defaultWidth - 16 * 2
                }}
              >
                {value}
              </div>
            </CopyToClipboard>
          </Tooltip>
        )
      } else {
        return (
          <div style={{ lineHeight: '21px', height: 21, overflowY: 'auto' }}>
            {value}
          </div>
        )
      }
    },
    onHeaderCell() {
      return {
        style: {
          backgroundColor: '#F5F5F5'
        }
      }
    }
  })

  // 固定表头
  let columns = []
  // 创建时间固定表头 --start--
  let fixedLeftItem = headers.find(item => item.header === 'gmtCreate')
  if (fixedLeftItem) {
    columns.push(
      Object.assign(genColumn(fixedLeftItem), {
        fixed: 'left'
      })
    )
  }
  headers = headers.filter(item => item.header !== 'gmtCreate')
  // 创建时间固定表头 --end--

  // 动态表头，表头由后端决定
  columns = [...columns, ...headers.map(item => genColumn(item))]

  return columns
}
