import React, { Component } from 'react'
import { message } from 'antd'
import PVChart from './pv-chart'
import PVTable from './pv-table'
import StatisticStore from '../../store'
import { BIDataConditionsEvent, BIDataExportEvent } from '../../config/pbTopics'
import PubSub from 'pubsub-js'
import update from 'immutability-helper'
import { getBIData, exportBIData, getBIChartData } from '../../api'
import { BIDataTableOptions } from '../../config'
import moment from 'moment'

// 动态计算滚动区域高度
let tableScrollHeight = window.innerHeight - 118 - 8 - 45 - 10
let tableScrollWidth = window.innerWidth - 360 - 24 * 2

const genColumn = item => ({
  title: item.title,
  dataIndex: item.key,
  key: item.key,
  width: item.width || BIDataTableOptions.defaultWidth,
  render(item) {
    return (
      <div style={{ lineHeight: '21px', height: 21, overflowY: 'hidden' }}>
        {item}
      </div>
    )
  },
  onHeaderCell() {
    return {
      style: {
        backgroundColor: '#F5F5F5',
        fontFamily: 'PingFangSC-Semibold'
      }
    }
  },
  className: item.isCoversion ? 'ConversionColumn' : '',
  fixed: item.fixed
})

//! 此处逻辑复杂，请谨慎修改
const washTableData = originData => {
  const relations = StatisticStore.componentRelations
  //! 无对应关系 === 组件被删除。然后作过滤逻辑
  let shouldDeleteIndexes = []
  originData.headers.forEach((item, index) => {
    let relation = relations.find(relation => relation.id === item.header)
    if (!relation) {
      shouldDeleteIndexes.push(index)
    }
  })
  //! （由于数据结构比较复杂，修改者请自行注意）过滤掉dataList.list中的动态数据
  // 注意👇修改了原数据结构，因为这里常用场景是直接拿ajax的数据，理论上不存在引用问题。后续开发者自己使用时注意
  originData.dataList.list &&
    originData.dataList.list.forEach(item => {
      for (let i = shouldDeleteIndexes.length - 1; i >= 0; i--) {
        item.buttonStaticDataDtos.splice(shouldDeleteIndexes[i], 1)
      }
    })
  // 过滤掉headers中的废弃表头
  for (let i = shouldDeleteIndexes.length - 1; i >= 0; i--) {
    originData.headers.splice(shouldDeleteIndexes[i], 1)
  }
  //! 过滤完成
  // 组装data
  let data =
    originData.dataList.list &&
    originData.dataList.list.map(item => {
      // 固定列数据
      let base = {
        [BIDataTableOptions.defineColumns['PVUV'].header]: `${item.visitPv ||
          '-'}/${item.visitUv || '-'}`,
        [BIDataTableOptions.defineColumns['curDate'].header]: item.curDate,
        [BIDataTableOptions.defineColumns['EPVUV'].header]: `${item.effectPv ||
          '-'}/${item.effectUv || '-'}`
      }
      // 动态数据
      item.buttonStaticDataDtos.forEach(widgetClickData => {
        base[widgetClickData.buttonId] = `${widgetClickData.buttonClickPv}`
      })
      return base
    })
  // 动态组装headers
  // 固定列配置
  // 动态表头列数+固定列数
  let columnLength = originData.headers.length + 3
  // 表格容器宽度是否小于总列数宽度，如果是则不用fixed布局
  let containerLessThanTotalColumn =
    tableScrollWidth < columnLength * BIDataTableOptions.defaultWidth
  let baseColumns = [
    genColumn({
      key: BIDataTableOptions.defineColumns['curDate'].header,
      title: BIDataTableOptions.defineColumns['curDate'].desc,
      fixed: containerLessThanTotalColumn ? 'left' : ''
    }),
    genColumn({
      key: BIDataTableOptions.defineColumns['PVUV'].header,
      title: BIDataTableOptions.defineColumns['PVUV'].desc,
      fixed: containerLessThanTotalColumn ? 'left' : ''
    }),
    genColumn({
      key: BIDataTableOptions.defineColumns['EPVUV'].header,
      title: BIDataTableOptions.defineColumns['EPVUV'].desc,
      fixed: containerLessThanTotalColumn ? 'left' : ''
    })
  ]
  // 动态列配置
  let columns = baseColumns.concat(
    originData.headers.map(item => {
      let relation =
        relations.find(relation => relation.id === item.header) || {}
      return genColumn({
        // 此处key必须与上面widgetClickData.buttonId一一对应
        key: item.header,
        title: relation ? relation.name : '无对应关系',
        isCoversion: relation ? relation.isCoversion : 0
      })
    })
  )
  // 总条数
  let totalCount = originData.dataList.totalCount
  return {
    data,
    columns,
    totalCount
  }
}

//! 自动补齐数据
const washChartData = (originData, startDate, endDate) => {
  let originList = originData || []
  let xAxisData = []
  let visitPVData = []
  let visitUVData = []
  // 用于匹配数据中curDate字段
  let curDateHeader = BIDataTableOptions.defineColumns['curDate'].header
  for (let i = 0; i < moment(endDate).diff(moment(startDate), 'days'); i++) {
    let dateString = moment(startDate).add(i, 'day').format('YYYY-MM-DD')
    // 按开始与结束日期生成x轴数据
    xAxisData.push(dateString)
    // 生成y轴数据，优先取服务端数据，如果没有则以0补齐
    let item = originList.find(item => item[curDateHeader] === dateString)
    // visitPV
    visitPVData.push(item ? item.visitPv : 0)
    // visitUV
    visitUVData.push(item ? item.visitUv : 0)
  }

  let seriesData = [visitPVData, visitUVData]
  return {
    xAxisData,
    seriesData
  }
}
export default class BI extends Component {
  state = {
    // 表格数据
    data: [],
    // 自定义表格头
    columns: [],
    loading: false,
    // 查询条件，通过pubsub更新
    conditions: {
      pagination: {
        current: 1,
        pageSize: 20,
        total: 20
      },
      startDate: moment(BIDataTableOptions.defaultRangeValue[0]).format(
        'YYYY-MM-DD'
      ),
      // 因为后端需要而加的逻辑
      endDate: moment(BIDataTableOptions.defaultRangeValue[1])
        .add(1, 'days')
        .format('YYYY-MM-DD')
    },
    // 表格数据
    chartData: {
      xAxisData: [],
      seriesData: []
    }
  }
  // 获取表格数据
  tableFetch() {
    // reset data
    this.setState({
      loading: true
    })
    const { startDate, endDate, pagination } = this.state.conditions
    getBIData({
      currentPage: pagination.current,
      startDate,
      endDate,
      landId: StatisticStore.landId,
      pageSize: pagination.pageSize
    }).then(res => {
      this.setState({ loading: false })
      if (res.code === '0') {
        let { data, columns, totalCount } = washTableData(res.data)
        if (res.data.dataList.list.length === 0) {
          message.info('无数据')
          return
        }
        this.setState({
          data,
          columns,
          conditions: update(this.state.conditions, {
            pagination: {
              $set: update(this.state.conditions.pagination, {
                total: {
                  $set: totalCount
                }
              })
            }
          })
        })
      }
    })
  }
  // 获取图表数据
  chartFetch() {
    // reset data
    this.setState({
      chartLoading: true
    })
    const { startDate, endDate } = this.state.conditions
    let limitStartDate = startDate
    if (Math.abs(moment(startDate).diff(moment(new Date(endDate)), 'year', true)) > BIDataTableOptions.maxYear) {
      limitStartDate = moment(new Date(endDate)).subtract(BIDataTableOptions.maxYear, 'years').format('YYYY-MM-DD')
      message.warn('趋势图最大时间范围为半年，已自动为您选择近半年的数据！')
    }
    getBIChartData({
      startDate: limitStartDate,
      endDate,
      landId: StatisticStore.landId
    }).then(res => {
      this.setState({ chartLoading: false })
      if (res.code === '0') {
        let { xAxisData, seriesData } = washChartData(res.data, limitStartDate, endDate)
        this.setState({
          chartData: {
            xAxisData,
            seriesData
          }
        })
      }
    })
  }
  componentWillMount() {
    PubSub.subscribe(BIDataConditionsEvent, (msg, conditions) => {
      // conditions中自定义的负载字段
      let _dataToFetch = []
      if (conditions && conditions._dataToFetch) {
        _dataToFetch = conditions._dataToFetch
        delete conditions._dataToFetch
      }
      let params = Object.assign({}, this.state.conditions, conditions)
      this.setState(
        {
          conditions: params
        },
        () => _dataToFetch.forEach(item => this[`${item}Fetch`]())
      )
    })
    PubSub.subscribe(BIDataExportEvent, () => {
      const { landId, landName, componentRelations } = StatisticStore
      const { startDate, endDate } = this.state.conditions
      // 前端定义后端导出表头，谨慎改动
      exportBIData({
        landId,
        landName,
        landHeaders: [
          ...componentRelations.map(item => ({
            desc: item.name,
            header: item.id
          })),
          ...[
            {
              header: 'curDate',
              desc: '日期'
            },
            {
              header: 'visitPv',
              desc: '访问PV'
            },
            {
              header: 'visitUv',
              desc: '访问UV'
            },
            {
              header: 'effectPv',
              desc: '转化PV'
            },
            {
              header: 'effectUv',
              desc: '转化UV'
            }
          ]
        ],
        startDate,
        endDate
      })
    })
  }
  componentWillUnmount() {
    PubSub.unsubscribe(BIDataConditionsEvent)
    PubSub.unsubscribe(BIDataExportEvent)
  }
  render() {
    return (
      <div>
        <PVChart chartData={this.state.chartData} />
        <PVTable
          tableScrollHeight={tableScrollHeight}
          tableScrollWidth={tableScrollWidth}
          data={this.state.data}
          columns={this.state.columns}
          loading={this.state.loading}
          pagination={this.state.conditions.pagination}
        />
      </div>
    )
  }
}
