import React, { Component } from 'react'
import { isEqual, get, cloneDeep, debounce } from 'lodash'
import { observer, inject } from 'mobx-react'
import Widget from 'layersWp/constructors/widget/index'
import {
  addAnimationStyleElement,
  removeAnimationStyleElement
} from './animationStyleHelper'

@Widget()
@inject('WPLayersStore')
@observer
class SpiritAnimationWidgets extends Component {
  state = {
    preAttrsSnapshot: null
  }

  setSnapshot = () => {
    const { widgetAttrs } = this.props
    this.setState({
      preAttrsSnapshot: cloneDeep(widgetAttrs)
    })
  }

  hasSpiritAnimationChanged = () => {
    const { widgetAttrs } = this.props

    if (!this.state.preAttrsSnapshot) return true

    return !isEqual(
      get(widgetAttrs, 'spiritAnimation'),
      get(this.state.preAttrsSnapshot, 'spiritAnimation')
    )
  }

  getSpiritDom = () => {
    const { targetId } = this.props
    const targetElement = document.querySelector(`div[wid="${targetId}"]`)
    return targetElement && targetElement.querySelector('.ant-dropdown-trigger')
  }

  updateAnimation = debounce(() => {
    const { widgetAttrs, targetId } = this.props
    this.unloadAnimation()
    addAnimationStyleElement(
      targetId,
      get(widgetAttrs, 'spiritAnimation'),
      animationClassName => {
        const spiritDom = this.getSpiritDom()
        if (spiritDom) {
          spiritDom.classList.add(animationClassName)
        }
      }
    )
  }, 1000)

  unloadAnimation = () => {
    const { targetId } = this.props
    const animationClassName = removeAnimationStyleElement(targetId)
    const spiritDom = this.getSpiritDom()
    if (spiritDom) {
      spiritDom.classList.remove(animationClassName)
    }
  }

  componentDidUpdate() {
    const hasSpiritAnimationChanged = this.hasSpiritAnimationChanged()

    if (hasSpiritAnimationChanged) {
      this.updateAnimation()
      this.setSnapshot()
    }
  }

  componentDidMount() {
    this.updateAnimation()
    this.setSnapshot()
  }

  componentWillUnmount() {
    this.unloadAnimation()
  }

  render() {
    return <React.Fragment />
  }
}

export default SpiritAnimationWidgets
