import React, { Component } from 'react'
import {
  Form,
  Tabs,
  Row,
  Col,
  Upload,
  message,
  Switch,
  InputNumber
} from 'antd'
import styles from '../edits.less'
import ButtonEditStyles from './index.less'
import { formItemProps } from '../helper'
import { SpiritAnimation } from 'layersWp/utils/itemType'
import {
  uploadParamsGen,
  beforeUploadGen,
  uploadStatusChangeGen,
  staticResourcePrefix
} from 'layersWp/config'
import { getNetUrl, readImageProps } from 'layersWp/utils/helper'
import { unset } from 'lodash-es'
import Editor from 'layersWp/constructors/editor/index'
import { observer, inject } from 'mobx-react'
import defaultAttrs from './attrs'

const FormItem = Form.Item
const TabPane = Tabs.TabPane

function getNumber(val, isInt = true, minNumber) {
  if (isInt) {
    val = Number.parseInt(val)
  } else {
    val = Number.parseFloat(val)
  }

  if (Number.isNaN(val) || (minNumber !== undefined && val < minNumber)) {
    val = minNumber
  }
  return val
}

const MIN_FRAME_NUMBER = 1
const MIN_DURATION = 0.01

@Editor({
  type: SpiritAnimation,
  basic: {
    jump: {}
  },
  attrs: defaultAttrs
})
@inject('WPLayersStore')
@observer
export default class ImageEditor extends Component {
  handleFrameNumberChange = value => {
    const { target } = this.props
    const { imageOriginSize, imageLoaded } = target.spiritAnimation
    let currFrameNum = getNumber(value, true, MIN_FRAME_NUMBER)

    if (imageLoaded) {
      const { dpr } = this.props.WPLayersStore.scene

      const divHeight = Number.parseFloat(target.style.height)
      const zoomRatio = divHeight / (imageOriginSize.height / dpr)
      const currentImageWidth = zoomRatio * (imageOriginSize.width / dpr)
      const currentDivWidth = currentImageWidth / currFrameNum

      this.props.handleChange({
        'style.width': currentDivWidth + 'px'
      })
    }

    this.props.handleChange({
      'spiritAnimation.frameNumber': currFrameNum
    })
  }

  handleDurationPerFrameChange = value => {
    let val = getNumber(value, false, MIN_DURATION)
    this.props.handleChange({
      'spiritAnimation.durationPerFrame': val
    })
  }

  render() {
    const {
      target,
      BasicAttrsComponent,
      WidgetNameCom,
      FixedAttrCom,
      JumpCom,
      EventCom,
      ActionCom
    } = this.props
    const ImageStyle = Object.assign({}, target.style, {
      maxWidth: '100%',
      maxHeight: '100%'
    })
    unset(ImageStyle, 'width')
    unset(ImageStyle, 'height')
    unset(ImageStyle, 'top')
    unset(ImageStyle, 'left')
    unset(ImageStyle, 'position')
    unset(ImageStyle, 'cursor')
    return (
      <div className={styles.edit}>
        <Tabs defaultActiveKey="1" style={{ height: '100%' }}>
          <TabPane tab="图片" key="1">
            <Form layout="inline">
              <Row style={{ marginBottom: '8px' }}>
                <Col offset={1} span={22}>
                  {WidgetNameCom}
                </Col>
              </Row>
              <Row>
                <Col offset={1} span={22}>
                  <div
                    className={ButtonEditStyles['uploader']}
                    style={{
                      backgroundImage: `url("${staticResourcePrefix}/masker@2x.png")`
                    }}
                  >
                    <Upload
                      {...{
                        ...uploadParamsGen(),
                        beforeUpload: beforeUploadGen(),
                        showUploadList: false,
                        onChange: uploadStatusChangeGen(info => {
                          if (info.file.status === 'done') {
                            const netUrl = getNetUrl(info.file.name)
                            const { dpr } = this.props.WPLayersStore.scene
                            this.props.handleChange({
                              'style.backgroundImage': `url("${netUrl}")`
                            })
                            // 帧数
                            const { frameNumber } = target.spiritAnimation
                            // resize图片尺寸
                            readImageProps(netUrl).then(res => {
                              this.props.handleChange({
                                'style.width':
                                  res.width / (dpr * frameNumber) + 'px',
                                'style.height': res.height / dpr + 'px',
                                'style.backgroundSize': 'auto 100% ',
                                'style.backgroundPosition': '0% 0%',
                                'spiritAnimation.imageOriginSize.width':
                                  res.width,
                                'spiritAnimation.imageOriginSize.height':
                                  res.height,
                                'spiritAnimation.imageLoaded': true
                              })
                            })
                          } else if (info.file.status === 'error') {
                            message.error(`${info.file.name}。文件上传失败`)
                          }
                        })
                      }}
                    >
                      <img
                        src={ImageStyle.backgroundImage.replace(
                          /url\("([\s\S]*)"\)/,
                          (_, matches) => matches
                        )}
                        style={ImageStyle}
                      />
                    </Upload>
                  </div>
                </Col>
              </Row>
              <div>
                <FormItem
                  label="动画帧数"
                  {...formItemProps}
                  {...{ labelCol: { span: 8 }, wrapperCol: { span: 15 } }}
                >
                  <InputNumber
                    value={target.spiritAnimation.frameNumber}
                    onChange={this.handleFrameNumberChange}
                  />
                </FormItem>
                <FormItem
                  label="单帧时长(s)"
                  {...formItemProps}
                  {...{ labelCol: { span: 8 }, wrapperCol: { span: 15 } }}
                >
                  <InputNumber
                    value={target.spiritAnimation.durationPerFrame}
                    step={0.01}
                    onChange={this.handleDurationPerFrameChange}
                  />
                </FormItem>
                <FormItem
                  label="是否循环"
                  {...formItemProps}
                  {...{ labelCol: { span: 8 }, wrapperCol: { span: 15 } }}
                >
                  <Switch
                    checked={target.spiritAnimation.isLoop}
                    onChange={value => {
                      this.props.handleChange({
                        'spiritAnimation.isLoop': value
                      })
                    }}
                  />
                </FormItem>
              </div>

              <BasicAttrsComponent
                basic={{ border: 'hidden', borderRadius: 'hidden' }}
              />
              {FixedAttrCom}
            </Form>
          </TabPane>
          <TabPane tab="点击" key="2">
            {JumpCom}
          </TabPane>
          <TabPane tab="事件" key="3">
            {EventCom}
          </TabPane>
          <TabPane tab="行为" key="4">
            {ActionCom}
          </TabPane>
        </Tabs>
      </div>
    )
  }
}
