import React, { Component } from 'react'
import { DragSource, DropTarget } from 'react-dnd'
import { Table, Icon, Tooltip, Popconfirm, message } from 'antd'
import CommonEditor from 'layers/components/common-editor'
import { observer } from 'mobx-react'
import { deepClone, dispatchEvent } from 'layersWp/utils/helper'
import CommonStyle from 'layersWp/config/commonStyle'
import FormTypeEditor from './form-type-editor'
import { verifyOption, typeOptions, groupSelectOptions } from './common'
import store from 'layersWp/store'
import styles from './index.less'
import generateDefaultAttrs from './default'

function TextHeader({ value }) {
  return (
    <div
      style={{
        color: CommonStyle['@secondary-font-color'],
        fontWeight: 'normal'
      }}
    >
      {value}
    </div>
  )
}

function dragDirection(
  dragIndex,
  hoverIndex,
  initialClientOffset,
  clientOffset,
  sourceClientOffset
) {
  const hoverMiddleY = (initialClientOffset.y - sourceClientOffset.y) / 2
  const hoverClientY = clientOffset.y - sourceClientOffset.y
  if (dragIndex < hoverIndex && hoverClientY > hoverMiddleY) {
    return 'downward'
  }
  if (dragIndex > hoverIndex && hoverClientY < hoverMiddleY) {
    return 'upward'
  }
}

class BodyRow extends React.Component {
  render() {
    const {
      isOver,
      connectDragSource,
      connectDropTarget,
      moveRow,
      dragRow,
      clientOffset,
      sourceClientOffset,
      initialClientOffset,
      ...restProps
    } = this.props
    const style = { ...restProps.style, cursor: 'move' }

    let className = restProps.className
    if (isOver && initialClientOffset) {
      const direction = dragDirection(
        dragRow.index,
        restProps.index,
        initialClientOffset,
        clientOffset,
        sourceClientOffset
      )
      if (direction === 'downward') {
        className += ' drop-over-downward'
      }
      if (direction === 'upward') {
        className += ' drop-over-upward'
      }
    }

    return connectDragSource(
      connectDropTarget(
        <tr {...restProps} className={className} style={style} />
      )
    )
  }
}

const rowSource = {
  beginDrag(props) {
    return {
      index: props.index
    }
  }
}

const rowTarget = {
  drop(props, monitor) {
    const dragIndex = monitor.getItem().index
    const hoverIndex = props.index

    if (dragIndex === hoverIndex) {
      return
    }

    props.moveRow(dragIndex, hoverIndex)

    monitor.getItem().index = hoverIndex
  }
}

const DragableBodyRow = DropTarget('row', rowTarget, (connect, monitor) => ({
  connectDropTarget: connect.dropTarget(),
  isOver: monitor.isOver(),
  sourceClientOffset: monitor.getSourceClientOffset()
}))(
  DragSource('row', rowSource, (connect, monitor) => ({
    connectDragSource: connect.dragSource(),
    dragRow: monitor.getItem(),
    clientOffset: monitor.getClientOffset(),
    initialClientOffset: monitor.getInitialClientOffset()
  }))(BodyRow)
)

@observer
export default class ControlSetting extends Component {
  constructor(props) {
    super(props)
    this.state = {
      selected: {
        data: null,
        index: null
      }
    }
  }
  components = {
    body: {
      row: DragableBodyRow
    }
  }
  moveRow = (dragIndex, hoverIndex) => {
    const { controlSetting, targetId } = this.props
    const dragger = deepClone(controlSetting[dragIndex])
    this.props.WPLayersStore.mutateArrayProp(
      targetId,
      'controlSetting',
      'splice',
      dragIndex,
      1
    )
    this.props.WPLayersStore.mutateArrayProp(
      targetId,
      'controlSetting',
      'splice',
      hoverIndex,
      0,
      dragger
    )
  }
  render() {
    const { targetId, controlSetting, handleChange, submitVerify } = this.props
    // hack read
    deepClone(controlSetting)

    const columns = [
      {
        key: 'index',
        className: 'form-table-column',
        render(_, __, index) {
          return <div style={{ width: 22 }}>{index + 1}</div>
        }
      },
      {
        title: <TextHeader value="表单类型" />,
        dataIndex: 'type',
        key: 'type',
        className: 'form-table-column',
        width: 120,
        render: (text, record, index) => {
          let value = typeOptions.find(
            item => `${text}$$${record.childType}` === item.value
          ).value
          return (
            <CommonEditor
              style={{
                color: CommonStyle['@primary-font-color'],
                fontWeight: 'bold'
              }}
              width={136 - 16}
              type="select-group"
              value={value}
              className="commoneditor"
              options={groupSelectOptions}
              disabledTooltip
              onChange={value => {
                handleChange({
                  [`controlSetting.${index}`]: generateDefaultAttrs(value)
                })
              }}
            />
          )
        }
      },
      {
        title: <TextHeader value="表单名称" />,
        dataIndex: 'title',
        key: 'title',
        className: 'form-table-column',
        width: 84,
        render: (text, _, __) => {
          if (text) {
            return (
              <div className="ellipsis" style={{ width: 84 }}>
                {text}
              </div>
            )
          } else {
            return '-'
          }
        }
      },
      {
        title: (
          <TextHeader
            value={
              <div>
                提示文字
                <Tooltip
                  placement="top"
                  title="用户输入之前显示在输入框中的提示文字。注意不同的输入框类型，需要选择与之兼容的提示文字类型。例如，日期类型的提示文字为YYYY-MM-DD的时间格式。"
                >
                  <Icon
                    style={{ marginLeft: 5, cursor: 'pointer' }}
                    type="question-circle-o"
                  />
                </Tooltip>
              </div>
            }
          />
        ),
        dataIndex: 'placeholder',
        key: 'placeholder',
        align: 'left',
        width: 220,
        className: 'form-table-column',
        render: (text, _, __) => {
          if (text) {
            return (
              <div className="ellipsis" style={{ width: 220 }}>
                {text}
              </div>
            )
          } else {
            return '-'
          }
        }
      },
      {
        title: <TextHeader value="类型校验" />,
        dataIndex: 'verify',
        key: 'verify',
        align: 'left',
        width: 80,
        className: 'form-table-column',
        render: (text, _, __) => {
          if (text !== 'none') {
            let value = verifyOption.find(item => text === item.value).text
            return value
          } else {
            return '-'
          }
        }
      },
      {
        title: <TextHeader value="是否必填" />,
        dataIndex: 'required',
        key: 'required',
        align: 'center',
        width: 60,
        className: 'form-table-column',
        render: (text, _, index) => {
          return <div style={{ width: 60 }}>{text ? '是' : '否'}</div>
        }
      },
      {
        title: <TextHeader value="更多操作" />,
        dataIndex: 'operations',
        key: 'operations',
        align: 'center',
        className: 'form-table-column',
        render: (_, __, index) => {
          return (
            <Popconfirm
              title="确认删除?"
              onConfirm={e => {
                e.stopPropagation()
                if (controlSetting.length === 1) {
                  message.warning('最少存在一条')
                } else {
                  this.props.WPLayersStore.mutateArrayProp(
                    targetId,
                    'controlSetting',
                    'splice',
                    index,
                    1
                  )
                }
              }}
              onCancel={e => e.stopPropagation()}
              okText="确认"
              cancelText="取消"
            >
              <a href="javascript:;" onClick={e => e.stopPropagation()}>
                删除
              </a>
            </Popconfirm>
          )
        }
      }
    ]
    const { data, index } = this.state.selected
    return (
      <div>
        <Table
          size="small"
          align="center"
          className="form-table sortable-table"
          pagination={false}
          dataSource={controlSetting}
          columns={columns}
          rowClassName={(_, index) => {
            let rowClassName = 'form-table-row'
            if (index === this.state.selected.index) {
              rowClassName += ' selected'
            }
            return rowClassName
          }}
          components={this.components}
          onRow={(record, index) => ({
            onClick: () => {
              const { selected } = this.state
              // 正反选
              this.setState({
                selected: {
                  data: selected.index === index ? null : record,
                  index: selected.index === index ? null : index
                }
              })
            },
            index,
            moveRow: this.moveRow
          })}
        />
        {data && (
          <FormTypeEditor
            index={index}
            data={data}
            handleChange={handleChange}
            changeState={selected => {
              this.setState({
                selected
              })
            }}
          />
        )}
        <p className={styles['addControl']}>
          <Icon type="plus" />
          <span
            className={styles['plus-btn']}
            onClick={() => {
              // TODO mobx和react table的问题
              let award = generateDefaultAttrs('input$$normal')
              this.props.WPLayersStore.mutateArrayProp(
                targetId,
                'controlSetting',
                'push',
                award
              )
              let timer = setTimeout(() => {
                document.querySelector(
                  '.form-table .ant-table-body'
                ).scrollTop = 9999
                const item = document.querySelector(
                  `.form-table .ant-table-row:last-child .commoneditor`
                )
                dispatchEvent(item, 'click')
                // 计算内部元素高度
                let a = document.querySelectorAll('#___form_wrapper___ > *')
                let height = 0
                a.forEach(node => {
                  height += node.offsetHeight
                })
                handleChange({
                  'style.height': height + submitVerify.dis + 'px'
                })
                clearTimeout(timer)
              }, 1)
            }}
          >
            添加表单项
          </span>
          <span className={styles['help-text']}>（拖动可调整表单顺序）</span>
        </p>
      </div>
    )
  }
}
